%IPM2MEX  Interior point method for AX>B and no equality constraints
%
% This is the MATLAB equivalent implementation of libmeq/ipm2.c.
% A more detailed help is available in IPM2MEX.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [x,s,z,kt,stat] = ipm2mexm(uH,c,A,b,x0,s0,z0,tol,nit,verb)
 if nargin<10, verb=0; end
 ni = size(A ,1);
 x = x0;
 s = s0;
 z = z0;
 n = size(c,1);
 % Construct H from upper triangular form
 H = zeros(n);
 ku = triu(true(n,n));
 H(ku) = uH;
 H = H + triu(H,1)';
 % Iteration loop
 for kt = 1:nit
  % Sanity check
  if any(s == 0), stat = false; return, end
  % Predictor step
  d = z./s;
  if any(~isfinite(d)), stat = false; return, end
  [U,p] = chol(H + A'*diag(d)*A);
  if p>0, stat = false; return, end
  rd = H*x + c - A'*z;
  rp = b - A*x;
  xp = d.*rp;
  dx = A'*xp - rd; % xd = A'*xp - rd;
  dx = U\(U'\dx);  % dx = U\(U'\xd);
  dz = A*dx - rp;  % dy = rp - A*dx;
  ds = s - dz;     % ds = -s - dy;
  dz = d.*dz;      % dz = d.*dy;
  % Largest step ensuring s>0 and z>0 (Note that we computed -ds and -dz)
  k = (ds > 0); alphas = min(s(k)./ds(k));
  k = (dz > 0); alphaz = min(z(k)./dz(k));
  alpha = min([1,alphas,alphaz]);
  % Update barrier parameter
  tau = (s - alpha*ds)'*(z - alpha*dz); % Since r0 and r1 can be both very small,
  tau = (tau/(s'*z)).^2*(tau/ni);       % this is more precise and less prone to NaNs
  % Centering-corrector step
  ec = (tau - ds.*dz)./z;
  res = max(abs([rd;s+rp;(ec-s).*z]));  % evaluate residual
  xp = xp + d.*ec;
  dx = A'*xp - rd;     % xd = A'*xp - rd;
  dx = U\(U'\dx);      % dx = U\(U'\xd);
  dz = A*dx - rp - ec; % dy = rp + ec - A*dx;
  ds = s - dz - ec;    % ds = -s + ec - dy;
  dz = d.*dz;          % dz = d.*dy;
  % Largest step ensuring s>0 and z>0 with some margin
  k = (ds > 0); alphas = min(s(k)./ds(k));
  k = (dz > 0); alphaz = min(z(k)./dz(k));
  alpha = min([1,0.99*alphas,0.99*alphaz]);
  % Update
  x = x + alpha*dx;
  s = s - alpha*ds;
  z = z - alpha*dz;
  % Exit conditions
  step = max(abs([dx;ds;dz]));
  if verb
   fprintf('ipm-2 it=%3d res=%5.2e step=%5.2e tol=%5.2e\n',kt,res,step,tol);
  end
  stat = res  < tol;
  if stat, break, end
  stat = step < tol;
  if stat, break, end
 end
 
end
