%MEQXK  extracts samples from LX/LY structure
% LXK = MEQXK(LX,K,fields) extracts the K (logical or index array) samples from
% LX. K='last' selects last index.
% Optional parameter 'fields': cell array of fields to extract
% (default: all).
% 
% See also MEQSK, MEQIK
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function LXk = meqxk(LX,k,varargin)

if islogical(k), k=find(k); end

% recursive call for cells
if iscell(LX)
  LXk = cell(size(LX));
  for ii=1:numel(LX)
    LXk{ii} = meqxk(LX{ii},k,varargin{:});
  end
  return
end

if ischar(k) && strcmp(k,'last'), k = numel(LX.t);   end
if ischar(k) && strcmp(k,'all' ), LXk=LX; return; end
if islogical(k), k = find(k); end
if any(k > numel(LX.t))
  error('requested index k=%d exceeds number of elements in LX.t (%d)',max(k),numel(LX.t))
end
if any(k<1)
  error('can not request k<1');
end

if nargin==2
  fields = fieldnames(LX)';
else
  fields = varargin{1};
  if ~iscell(fields), error('fields must be a cell'); end
  if ~isrow(fields), fields=fields'; end
end


nt = numel(LX.t);
% Shortcut when LX has a single time slice
if nt==1
  LXk = meqlpack(repmat(LX,1,numel(k)));
  return
end

for n = fields
  myfld = n{1};
  switch myfld
    case 'shot'
      LXk.shot = LX.shot(min(k,numel(LX.shot))); % shot may be scalar or vector
    case 'tokamak'
      LXk.tokamak = LX.tokamak;
    otherwise
      datn = LX.(myfld);
      
      switch myfld
        case {'Fx','Iy','Opy'}
          % handle special case when nry or nrx = nt
          hastime = (ndims(datn)==3);
        otherwise
          hastime = (size(datn,ndims(datn)) == nt);
      end
      
      if hastime
        % Works for any rank
        siz = size(datn);
        datn = reshape(datn,[],siz(end));
        LXk.(myfld) = reshape(datn(:,k),[siz(1:end-1),numel(k)]);
      else
        % This field does not depend on time
        LXk.(myfld) = datn;
      end
  end
end

