function LX = meqxdoublet(L,LX)
% MEQXDOUBLET - Adds missing fields per domain in case of doublet
% LX = meqxdoublet(L,LX)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

nt = numel(LX.t);
for ifield = {'Ip','bp','bt','Wk','li'}
  fld1 = ifield{:};
  fldD = [fld1,'D'];

  fld1_exists = isfield(LX,fld1);
  fldD_exists = isfield(LX,fldD);

  fld1_only =  fld1_exists && ~fldD_exists;
  fldD_only = ~fld1_exists &&  fldD_exists;

  if ~fld1_exists && ~fldD_exists
    % neither field is provided, error will be thrown later if it is needed
    continue
  end
  
  idoublet = L.P.idoublet;
  
  if fld1_only && idoublet
    % assign default doublet values from singlet - this is needed for
    % e.g. TCV legacy MGAMS shots like 55331
    fld1val = LX.(fld1);
    fldDval = singlet2doublet(fld1val,fldD,L.nD);
    LX.(fldD)    = fldDval;
    LX.(fld1)(:) = NaN; % set singlet value to nan
  elseif fldD_only && ~idoublet
    % copy to singlet field and set doublet value to NaN
    LX.(fld1)    = LX.(fldD)(1,:);
    LX.(fldD)(:) = NaN;
    warning('LX has %s for nD=1, it will be copied to %s and %s will be set to NaN',fldD,fld1,fldD)
  elseif fld1_exists && fldD_exists && ~isequal(fld1,'Ip')
    if       idoublet && ~all(isnan(LX.(fld1)))
      LX.(fld1)(:) = NaN; % set the other to nan
      warning('LX has both %s and %s for nD>1, %s will be set to NaN and ignored',fld1,fldD,fld1);
    elseif  ~idoublet && ~all(isnan(LX.(fldD)))
      LX.(fldD)(:) = NaN; % set the other to nan
      warning('LX has both %s and %s for nD=1, %s will be set to NaN and ignored',fldD,fld1,fldD);
    end
  end
end

if isfield(LX,'qA')
  % qA special, this is always 'per A(xis)' so no D counterpart
  if size(LX.qA,1)==1 && any(L.P.idoublet)
    qA = zeros(2,nt); % init
    qA(1:2,:) = repmat(LX.qA,2,1); % assume same qA for all lobes
    LX.qA = qA;
  end
end

% Ip is special because we need it in various places, so we define it here
% in case only IpD was passed for multidomain cases.
if ~isfield(LX,'Ip') || any(isnan(LX.Ip))
  if isfield(LX,'IpD')
    LX.Ip = sum(LX.IpD,1); % we always need Ip
  else
    error('must provide at least Ip or IpD');
  end
elseif isfield(LX,'IpD') && ~all(isnan(LX.IpD(:)))
  % check the values are compatible
  if any(sum(LX.IpD,1)-LX.Ip)
    error('LX.IpD and LX.Ip provided but sum(IpD) ~= Ip. max(diff)=%e',...
      max(sum(LX.IpD,1)-LX.Ip))
  end
end
end

function doubletval = singlet2doublet(singletval,fldD,nD)

doubletval = zeros(nD,size(singletval,2)); % placeholder

% scaling to translate desired global parameters into scalar ones for
% doublets/droplets
switch fldD
  case 'IpD', s = 1/2;   % Half the current in each axial domain
  case 'bpD', s = 2;     % Half the energy and current in each axial domain, bpD ~ Wk/Ip^2
  case 'liD', s = 2;     % Half the magnetic energy and current in each axial domain, liD ~ Ip/Ip^2
  case 'btD', s = 1;     % Half the energy in each axial domain, but toroidal flux is also halved
  case 'WkD', s = 1/2;   % Half the energy in each axial domain
end
doubletval(1:2,:) = s*repmat(singletval,2,1);
end
