function [LX2] = meqxconvert(L,LX,L2,minimal)
% [LX2] = meqxconvert(L,LX,L2[,minimal])
%
% convert LX data structure based on L
% into LX2 structure based on L2
% L,L2 may have different dima, dimu, bfct, x grid
% If minimal=true, also remove all fields not essential for running L2.code (default=false)
%
% Only the fields Ia, Iu, ag, Fx, Iy, PpQ, TTpQ, PQ, TQ, iTQ, PpQg, TTpQg, are updated.
% Some specific fields used only in some codes are added if they are absent.
%
% NB: This does *not* yield a consistent equilibruim structure, and should
% only be used to convert LX inputs to be ingested by the various codes.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if nargin<4
  minimal=false; % default
end
LX2 = LX; % copy

%% Sanity checks
assert(L2.nD>=L.nD || strcmp(L2.code,'liu'), 'output nD < input nD not valid request')

%% Number of time slices
nt = numel(LX2.t);

%% Convert dima
if ~isequal(L.G.dima,L2.G.dima)
  [lia,ia2] = ismember(L.G.dima,L2.G.dima); % check all coil currents have been picked
  LX2.Ia = zeros(L2.G.na,nt);
  LX2.Ia(ia2(lia),:) = LX.Ia(lia,:);
  coilok = (lia | ~any(LX.Ia,2)); % coil selected or has no current
  assert(all(coilok),'Current in %s was not selected but is nonzero',strjoin(L.G.dima(~coilok),','))

  if isfield(LX,'Va')
    % also convert Va
    LX2.Va = zeros(L2.G.na,nt);
    LX2.Va(ia2(lia),:) = LX.Va(lia,:);
  end
end

%% Convert dimu
if ~isequal(L.G.dimu,L2.G.dimu)
  LX2.Iu = L2.G.Tvu \ LX.Iv;
end

%% Number of axes
if ~isfield(LX2,'nA')
  LX2.nA = zeros(1,nt);
  for kt = 1:nt
    nA = find(~isnan(LX2.FA(:,kt)),1,'last');
    if isempty(nA), continue; end % nA=0
    LX2.nA(kt) = nA;
  end
end

%% Number of domains
if ~isfield(LX2,'nB')
  LX2.nB = zeros(1,nt);
  for kt = 1:nt
    nB = find(~isnan(LX2.FB(:,kt)),1,'last');
    if isempty(nB), continue; end % nB=0
    LX2.nB(kt) = nB;
  end
end

%% Domain limits
if ~all(isfield(LX2,{'F0','F1'})) % For stored runs without F0/F1
  LX2.F0 = zeros(L2.nD,nt);
  LX2.F1 = LX2.F0;
  for kt = 1:nt
    % This reimplements F0,F1 calculation from meqpdom, valid for up to 3 domains
    assert(L.nD<=3,'Recomputing F0/F1 requires 3 domains or less');
    nA = min(L2.nD,LX2.nA(kt));
    nB = min(L2.nD,LX2.nB(kt));

    LX2.F0(   1:nA,kt) = LX2.FA(:      ,kt);
    LX2.F0(nA+1:nB,kt) = LX2.FB(1:nB-nA,kt);
    LX2.F1(   1:nB,kt) = LX2.FB(:      ,kt);
  end
end

%% Convert bfct parameters
% fit profiles from source to new basis for FGS

if strcmp(L2.code, 'liu')
  LX2.ag = zeros(L2.ng, nt); % LIUQE will find them with LSQ fit. No ag constraint considered yet.
else
  refitag = ~isfield(L,'bfct') || ~isequal(L.bfct,L2.bfct) || ~isequal(L.bfp,L2.bfp) || ~isfield(LX,'ag');
  if refitag
    LX2.ag = meqfitprof(L2,L.pQ.^2,LX2.F0,LX2.F1,LX2.PpQ,LX2.TTpQ);
  else
    if L2.ng > L.ng % ag=0 for additional functions
      LX2.ag = zeros(L2.ng, nt);
      LX2.ag(1:L.ng,:) =  LX.ag;
    end
  end
  
  if refitag || ~isequal(L.pQ,L2.pQ)
    % Re-compute basis functions on new pQ grid or following new BFs
    FN = L2.pQ.^2;
    [PpQ,TTpQ,PQ,TQ,iTQ] = deal(zeros(L2.npq+1,L2.nD,nt)); % init
    [PpQg,TTpQg] = deal(zeros(L2.npq+1,L2.nD,L2.ng,nt));
    
    for kt = 1:nt
      [PpQ(:,:,kt),TTpQ(:,:,kt),PQ(:,:,kt),TQ(:,:,kt),iTQ(:,:,kt),PpQg(:,:,:,kt),TTpQg(:,:,:,kt)] = ...
        meqprof(L2.fPg,L2.fTg,LX2.ag(:,kt),FN,LX2.F0(:,kt),LX2.F1(:,kt),LX2.rBt(:,kt),L2.bfct,L2.bfp,L2.idsx,L2.smalldia);
    end
    if L2.nD == 1, PpQ  = reshape( PpQ,[L2.npq+1,nt]);
      TTpQ = reshape(TTpQ,[L2.npq+1,nt]);
      PQ   = reshape(  PQ,[L2.npq+1,nt]);
      TQ   = reshape(  TQ,[L2.npq+1,nt]);
      iTQ  = reshape( iTQ,[L2.npq+1,nt]);
    end
    PpQg  = reshape( PpQg,[L2.npq+1,L.nD,L2.ng,nt]);
    TTpQg = reshape(TTpQg,[L2.npq+1,L.nD,L2.ng,nt]);
    LX2 = meqlarg(LX2,PpQ,TTpQ,PQ,TQ,iTQ,PpQg,TTpQg);
  end
end

%% Reinterpolate Fx, Iy, Opy if necessary
if ~isequal([L2.zx;L2.rx],[L.zx;L.rx]) && ...
  ~(minimal && strcmp(L2.code,'liu') && isfield(LX2,'Xt')) % Fx/Iy not needed for minimal LIU input
                                                           % except if Xt needs to be recomputed
  % Fx
  FFx = griddedInterpolant({L.zx,L.rx},LX.Fx (:,:,1));
  ntFx = size(LX.Fx,3);
  Fx = zeros(L2.nzx,L2.nrx,ntFx);
  for it=1:ntFx
    if it>1, FFx.Values = LX.Fx(:,:,it); end
    Fx(:,:,it) = FFx({L2.zx,L2.rx});
  end
  % grid may be too wide
  Fx(isnan(Fx))=0;
  LX2.Fx = Fx;

  % Iy
  FIy  = griddedInterpolant({L.zy,L.ry},LX.Iy (:,:,1));
  ntIy = size(LX.Iy,3);
  Iy  = zeros(L2.nzy,L2.nry,ntIy);
  for it=1:ntIy
    if it>1, FIy.Values = LX.Iy(:,:,it); end
    Iy(:,:,it) = FIy({L2.zy,L2.ry});
  end
  % grid may be too wide
  Iy (isnan(Iy ))=0;
  Iy = Iy.*reshape(LX.Ip./sum(reshape(Iy,L2.ny,[])),1,1,[]); % ensure same total Ip
  LX2.Iy = Iy;

  % Opy
  if isfield(LX,'Opy')
    FOpy = griddedInterpolant({L.zy,L.ry},double(LX.Opy(:,:,1)));
    ntOpy = size(LX.Opy,3);
    Opy = zeros(L2.nzy,L2.nry,ntOpy,'int8');
    for it=1:ntOpy
      if it>1, FOpy.Values = double(LX.Opy(:,:,it)); end
      Opy(:,:,it) = int8(FOpy({L2.zy,L2.ry}));
    end
    % grid may be too wide
    Opy(isnan(Opy))=0;
    LX2.Opy = Opy;
  end
end

%% contour post-processing initialization
initW = L2.P.iterq && ...
  ( L.P.iterq==0 || ... % no previous postprocessing
  (L2.G.nW~=L.G.nW) || (L2.P.nFW ~= L.P.nFW) || ... % sizes compatibility
  (L.G.nW==0 || ~(isequal(L2.G.rW,L.G.rW) && isequal(L2.G.zW,L.G.zW) && isequal(L2.G.oW,L.G.oW)))... % gaps gap settings incompatibility
  );

initq =  L2.P.iterq && ...
  ((L.P.iterq==0) || ... % no previous postprocessing
  L2.noq ~= L.noq || L2.npq ~= L.npq); % sizes mismatch

if initW, LX2.aW = []; end
if initq, LX2.aq = []; end

%% Non-defined quantities for specific code targets
if any(strcmp(L2.code,{'fge','rzp'}))
  if ~isfield(LX2,'Ini'   ), LX2.Ini    = zeros(1,nt); end % Ini total
  if ~isfield(LX2,'IniD'  ), LX2.IniD   = zeros(L2.nD,nt); end % Ini total
  if ~isfield(LX2,'signeo') 
    LX2.signeo = repmat((1-0.9*L2.pQ')*L2.P.signeodflt,1,L2.nD,nt); % Default for all domains
    if L2.nD == 3 % Doublet special
      LX2.signeo(:, 3, :) = repmat((1-0.9*L2.pQ'.^2)*L2.P.signeodflt,1,1,nt)*0.1; % scale down mantle
    end
  end
  if ~isfield(LX2,'Lp'    ), LX2.Lp     = evalLp(L2,LX2); end % Plasma self inductance
  if ~isfield(LX2,'SQ'    ), LX2.SQ     = evalSQ(L2,LX2); end
  if ~isfield(LX2,'Rp'    ), LX2.Rp     = reshape(1./LX2.SQ(end, :), L2.nD, nt); end % Resistance of the plasma.
elseif strcmp(L2.code,'liu')
  if ~isfield(LX2,'Xt'    ), LX2.Xt     = LX2.Ft + L2.Mty*reshape(LX2.Iy,L2.ny,nt); end % DML measurement
end

%% Keep only relevant fields - toss the rest
if minimal
  switch L2.code
    case 'liu' % Implemented only for liu at this time
      required_fields = {'t','Bm','Ff','Ia','Xt','rBt','Iu','Ip'};
      optional_fields = {'Ft','Uf','Is','shot'};
    otherwise
      warning('''minimal=true'' option not implemented yet for code=%s',L2.code)
      required_fields = {};
      optional_fields = {};
  end

  if ~isempty([required_fields,optional_fields])
    fields = fieldnames(LX2);

    % check that all required fields are there
    ireqok = contains(required_fields,fields);
    assert(all(ireqok),'missing required field(s): %s',strjoin(required_fields(~ireqok)));

    % remove unneeded fields
    unneeded = ~ismember(fields,[required_fields,optional_fields]);
    LX2 = rmfield(LX2,fields(unneeded));
  end
end

%% Remove doublet/singlet fields if they are redundant
for fields = {'bp','bt','Wk','li'} % keep Ip
  fld1 =  fields{:};
  fldD = [fields{:},'D'];
  if isfield(LX2,fldD) && isfield(LX2,fld1) % both fields present
    if L.nD>1
      LX2 = rmfield(LX2,fld1);
    else
      LX2 = rmfield(LX2,fldD);
    end
  end
end

end

function Lp = evalLp(L, LX)
% Compute LpD = IyD.' Myy IyD / IpD^2
% This is a bit cumbersome but not all loaded equilibria have Opy, so need
% to recompute

Lp = zeros(L.nD, numel(LX.t));
for kt=1:numel(LX.t)
  LXt = meqxk(LX,kt);
  
  % take from LXt
  Fx = LXt.Fx; sIp = sign(LXt.Ip); Iy = LXt.Iy;
  
  % meqpdom
  [~,~,~,~,~,~,~,~,~,~,~,~,...
    ~,~,~,~,~,Opy,~,~,stat,msg,id] = meqpdom(Fx,sIp,L.P.isaddl,L);
  if ~stat
    meqmsge('e',mfilename,L.P.tok,LXt.t,NaN,LXt.shot,msg,id);
  end

  for iD=1:L.nD
    IyD = Iy .* (Opy == iD);
    IpD = sum(IyD(:));
    if IpD ~= 0
      IyN = IyD / IpD;
    else
      IyN = (Opy == iD) / sum((Opy(:) == iD));
    end
    
    FxN = meqFx(L, IyN); FyN = FxN(2:end-1, 2:end-1);
    Lp(iD, kt) = IyN(:).' * FyN(:);
  end
end
end

function SQ = evalSQ(L,LX)
% Compute SQ = B0/2/pi int_0^Phi(pQ) (sigma/T^2) dPhi
% This is a bit cumbersome but not all loaded equilibria have Opy and all
% the FA, FB etc, so need to recompute everything to finally get SQ

SQ = zeros(L.nQ,L.nD,numel(LX.t));
if ~any(L.icde) || ~any(LX.Ip),return; end

for kt=1:numel(LX.t)
  LXt = meqxk(LX,kt);
  
  % take from LXt
  Fx = LXt.Fx; sIp = sign(LXt.Ip); ag = LXt.ag; rBt = LXt.rBt;
  
  % meqpdom
  [~,~,~,~,~,~,~,~,~,~,~,~,...
    ~,~,FB,~,~,Opy,F0,F1,stat,msg,id] = meqpdom(Fx,sIp,L.P.isaddl,L);
  if ~stat
    meqmsge('e',mfilename,L.P.tok,LXt.t,NaN,LXt.shot,msg,id);
  end
  
  [~,~,~,~,FtQ,Ft0Q] = meqintQ(L,F0,F1,rBt,ag,Fx,Opy);
  FtPQ = FtQ+Ft0Q; % total Plasma toroidal flux on pQ grid
  [~,~,~,TQ] = meqprof(L.fPg,L.fTg,ag,L.pQ(:).^2,F0,F1,rBt,L.bfct,L.bfp,L.idsx);
  
  iTsQ = LX.signeo./TQ.^2; % sigma/F^2 profile appearing in various places
  for iD=1:numel(FB)
    SQ(:,iD,kt) = cumtrapz(FtPQ(:,iD), iTsQ(:,iD));
  end
  SQ(:,:,kt) = SQ(:,:,kt) * rBt/(2*pi);
end
SQ = squeeze(SQ);
end
