% MEQVER Returns current version number for the MEQ toolbox.
%
% Now also returns information about commit, tag and url of meq and
% dependencies (genlib, psitbx).
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [v,code] = meqver()

v=0; % Init meq version number

% Get meq, genlib and psitbx
codepaths = {fileparts(mfilename('fullpath')),fileparts(which('greenem')),fileparts(which('psitbxtcv'))};
if nargout < 2, codepaths = codepaths(1);end % Only meq is needed for its version number
ncodes = numel(codepaths);
[code.commit,code.tag,code.url] = deal(cell(ncodes,1));
for ii = 1:ncodes
 codepath = codepaths{ii};
 [sha1,tag,url] = deal('');
 % Parse version info from SPC .this-deployment.info file
 deploy_info = fullfile(codepath,'.this-deployment.info');
 if exist(deploy_info,'file')
  fid = fopen(deploy_info);
  while(true)
   tline = fgetl(fid);
   if ~ischar(tline), break;end
   name = regexp(tline,'^([^:]*): (.*)','tokens');
   if isempty(name), continue;end
   switch name{1}{1}
     case 'GIT_COMMIT',         sha1 = name{1}{2};
     case 'GIT_TAG',            tag  = name{1}{2};
     case 'GITLAB_PROJECT_URL', url  = name{1}{2};
   end
  end
 % else from existing git dir
 elseif exist(fullfile(codepath,'.git'),'dir')
  gitcmd = ['git -C ',codepath];
  [~,sha1] = system([gitcmd,' rev-parse --short HEAD']);
  [~,tag ] = system([gitcmd,' describe --tag']);
  [~,url ] = system([gitcmd,' remote -v | grep fetch | cut -f 2 | cut -d " " -f 1']);
 end
 
 code.commit{ii}  = strtrim(sha1);
 code.tag{ii}     = strtrim(tag);
 code.url{ii}     = strtrim(url);
 
 tokens = regexp(strtrim(tag),'^crpptbx-release-v(\d+)\.(\d+)\.(\d+)$','tokens');
 
 if ~isempty(tokens)
  % recognised tag
  if ii == 1, v=str2double(tokens{1}{1})+str2double(tokens{1}{2})*0.01; end % Get meq version number (assumes minor version < 100)
  assert(str2double(tokens{1}{2})<100,'minor version number must be <100')
 end
end