function [qpsolve,qpopts] = meqqpsetup(P)
% MEQQPSETUP Setup MEQ QP solver
% 
% [qpsolve,qpopts] = meqqpsetup(P)
%
% Inputs:
%  P: MEQ parameter object
%
% Outputs:
%  qpsolve: function handle with QP solver to use
%  qpopts: structure or object with QP solver options
%
% SEE ALSO: MEQLSQLIN
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

qpmaxiter = 200; % default for QP tolerance / iteration limit
switch P.qpsolver
  case 'quadprog'
    if ~isempty(which('optimoptions')) % optimoptions not available in Octave
      % Debug output
      switch P.debug
        case {0,1},   display = 'off'  ;
        case 2,       display = 'final';
        otherwise,    display = 'iter' ;
      end
      qpopts = optimoptions('quadprog',...
                            'Display',display,...
                            'OptimalityTolerance',1e-12,...
                            'Algorithm','interior-point-convex',...
                            'MaxIterations',qpmaxiter);
    else
      qpopts = optimset('quadprog');
      qpopts.MaxIter = qpmaxiter;
    end
    qpsolve = @quadprog;
  case {'ipm-optimized','ipm'}
    useopt = strcmp(P.qpsolver,'ipm-optimized');
    qpopts = ipmopts('useoptimized',useopt,...
                     'niter',qpmaxiter,...
                     'debug',P.debug>2,...  % QP verbosity (for debugging)
                     'presolve',true); % use presolver
    qpsolve = @ipmwrapper; % local wrapper for ipm
  otherwise
    error('unrecognized optimizer %s',P.optimizer)
end
  end
