function P = meqptcv(shot)
% MEQ-TCV:specific parameters:
% .sela        Active coil selection, cell array, e.g. {'E','F','G','OH'}
% .selc        If true, return Tca matrix in G structure
% .selx        Computational grid selection, string, e.g. 'X', 'XF'
% .static      Selects the TCV static tree version
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if nargin < 1 || isempty(shot), shot = -1; end
% defaults for TCV
P.tokamak = 'TCV';
P.tok     = 'tcv';
P.r0 = 0.88; % default radius used for outputs/normalizations
P.b0 = 1.43; % default toroidal field used for outputs/normalizations
P.izgrid = false;
P.selc = true; % if true, return Tca matrix connecting coils and circuits
P.mdsserver = meqmdsserver();
P.selx = 'X'; % default grid
P.ilackner = 2; % Matrix-form of Lackner's trick, numerically equivalent to 1 but faster for default grid

% parameters related to coil combination chosen to inductively drive Ip
P.ohcoils   = 'OH'; % coil labels used for 'ohmic' inductive drive
P.compcoils = {'E','F'}; % used to compensate field by ioha coilss
P.ohequal = true; % Force currents of coils for OH induction to be equal

% TCV static tree version
if shot>1e6, tcvshot = floor(shot/1e3); % Handle personal trees
else,        tcvshot = shot;
end

% When writing to TCV trees with iterq=0, the Psi-toolbox will be used
%   This sets up the standard Psi-toolbox grid size for TCV trees
%   as this will later be a parameter in MDS
P.pq  = (0:40)/40;
P.noq = 128;

meqmdsopen(-1,'static',[],P.mdsserver)
[P.static,st] = mdsvalue('\version[$1]',int32(tcvshot));
assert(isodd(st),'Unable to retrieve static tree version for TCV shot %d',tcvshot);
% Values corresponding to the default grid (stored as singles in MDSplus)
meqmdsopen(P.static,'static',[],P.mdsserver);
[rx,str] = mdsvalue('static("r_x:axis")');
[zx,stz] = mdsvalue('static("z_x:axis")');
assert(isodd(str)&&isodd(stz),'Unable to retrieve default grid from the TCV static tree');
P.ri = rx(1);
P.ro = rx(end);
P.zl = zx(1);
P.zu = zx(end);
P.nr =  numel(rx)-1;
P.nz = (numel(zx)-1)/2;
% Default coils
if shot > -1 && shot < 9e5, shotm = shot;
else,                       shotm = -1;   end
meqmdsopen(shotm,'magnetics',[],P.mdsserver);
[dima,sta] = mdsvalue('\IPOL:DIM');
assert(isodd(sta),'Unable to retrieve active coil names from TCV magnetics tree for shot %d',shotm);
if ismember('G_001',deblank(dima)), P.sela = {'E','F','G','OH'};
else,                               P.sela = {'E','F',    'OH'}; end
