function [LY] = meqpostq(L,LY)
%MEQPOSTQ Updates LY with quantities related to contour integrals and wall gaps
% function [LY] = meqpostq(L,LY)
% 
% For details of gap/contour finding algorithm, see find_contours.m, [MEQ-redbook]
% 
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if isempty(LY), return; end

% plasma flux surfaces, rho-ray origin points and directions are returned by rtci
[rq, zq, aq, rO, zO, crq, czq, nDeff] = rtciplasma(L, LY);
% get flux surface quantities and shape parameters
[Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ,...
 VQ,AQ,IpVQ,FtPVQ,...
 rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
 rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = fsgi(L, LY, rq, zq, aq, rO, zO);

% wall gaps are computed if needed
[aW, FW] = rtciwall(L, LY);

% q95
q95 = 1./(L.c95*iqQ(L.i95,:))';
% jtorQ = R0*<jphi/R>
jtorQ = 2*pi*L.P.r0 * (LY.PpQ + (LY.TTpQ.*Q2Q)/mu0);
% domain outer surface length and r-barycenter
lp = LpQ(end, :)';
rbary = rbQ(end, :)';
% rhotornorm
rhotornorm = sqrt((FtPVQ-FtPVQ(1, :))./(FtPVQ(end, :)-FtPVQ(1, :)));

% outboard radius and q surface location
sq = sign(LY.Ip) * sign(LY.rBt);
raQ = zeros(L.nQ, L.nD);
raqmin = zeros(L.nD, 1);
qmin = zeros(L.nD, 1);
% iterate over domains
for iD=1:nDeff
  if iD>LY.nA
    % for mantle domain, the ray to the right of the first axis is chosen
    [~, icr] = min(abs(zq(:, 1, iD) - LY.zA(1)) - rq(:, 1, iD));
  else
    % for geometric radial coordinate in axis domain, most "outboard" (+R) constant-theta ray is used
    [~, icr] = max(crq(:, iD));
  end
  raQ(:, iD) = [0; aq(icr, :, iD)']; raQ(:, iD) = raQ(:, iD) / raQ(end, iD);
  [raqmin(iD), iqmin] = minQmex(raQ(:, iD), iqQ(:, iD), -sq, 1);
  qmin(iD) = 1 ./ iqmin;
end

% collect new additions in LY
LY = meqlarg(LY,iqQ,jtorQ,q95,lp,rbary,IpVQ,FtPVQ,rhotornorm,...
  rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
  rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin,...
  Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,ItQ,LpQ,rbQ,Q5Q,VQ,AQ,SlQ,...
  aq,rq,zq,aW,FW,...
  raqmin,qmin,raQ);

% raR computation if needed
if L.nR
  raR = zeros(L.nR, L.P.naR, L.nD);
  for iD=1:nDeff
    raR(:, :, iD) = locRmex(LY.raQ(:, iD), LY.iqQ(:, iD), sq * L.P.iqR, L.P.naR, L.raN);
  end
  LY = meqlarg(LY, raR);
end

% rS zS computation if needed
if L.nS
  rS = zeros(L.noq, L.nS, L.nD);
  zS = zeros(L.noq, L.nS, L.nD);
  for iD=1:nDeff
    aS = locSmex([zeros(L.noq, 1), LY.aq(:, :, iD)], LY.raQ(:, iD), L.P.raS, L.raN);
    rS(:, :, iD) = crq(:, iD) .* aS + rO(:, iD);
    zS(:, :, iD) = czq(:, iD) .* aS + zO(:, iD);
  end
  LY = meqlarg(LY, rS, zS);
end

% Additional output arguments
for k = 1:numel(L.argoutc)
  if strcmp(L.argoutc{k}.file,mfilename)
    LY.(L.argoutc{k}.fld) = eval(L.argoutc{k}.exp);
  end
end
end
