function [LY] = meqpost(L,LY,ag,...
  Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lB,lX, ...
  rX,zX,FX,dr2FX,dz2FX,drzFX, ...
  rBt,Ia,Iu,Iy,Opy,F0,F1)
%MEQPOST Post-processing for various meq functions
% LY = meqpost(L,LY,ag,...
%  Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lX,...
%  rBt,Ia,Iu,Iy,Opy)
% Computes integral quantities, estimated magnetic measurements,
% profiles, vessel currents. Given:
% MEQ structure L, 
% LY with shot, t, and containing optional initial aq,aW guesses.
% flux map Fx(Z,R), axis, boundary flux FA,FB, axis position and
% flux hessians rA,zA,dr2FA,dz2FA,drzFA.
% Boundary flux position rB,zB. Locicals lB, lX indicating whether boundary
% was found and whether the plasma is diverted.
% rX,zX,FX,dr2FX,dz2FX,drzFX positions,fluxes and flux hessians of all X points.
% rBt the vacuum r*Bt.
% Ia,Iu,Iy the circuit, vessel and plasma currents.
% Opy the plasma current mask.
% F0,F1 the limiting fluxes for each domain.
%
% if L.P.iterq>0, then initial guesses for aq and aW can be supplied in LY
%
%   See MEQINT, MEQPOSTQ, MEQPROF, BFHELP
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% bfct
fPg = L.fPg; fTg = L.fTg;

nA = int32(numel(FA)); % number of axes
nB = int32(numel(FB)); % number of bounded domains
nX = int32(numel(FX)); % number of X points

% When L.P.LYall is true, some slices can have more domains than is allowed
if nA>L.nD
  nA    = L.nD;
  FA    =    FA(1:L.nD);
  rA    =    rA(1:L.nD);
  zA    =    zA(1:L.nD);
  dr2FA = dr2FA(1:L.nD);
  dz2FA = dz2FA(1:L.nD);
  drzFA = drzFA(1:L.nD);
end
if nB>L.nD
  nB = L.nD;
  FB = FB(1:L.nD);
  rB = rB(1:L.nD);
  zB = zB(1:L.nD);
  lX = lX(1:L.nD);
end

FR = calcFR(F0,F1,FX,lX); % X-point/boundary normalized flux ratio

% cumulative integrals per domain
[IpQ,WkQ,WpQ,VpQ,FtQ,Ft0Q] = meqintQ(L,F0,F1,rBt,ag,Fx,Opy);
FtPQ = FtQ+Ft0Q;

% Integrals per domain
IpD  = IpQ (end,:).';
WkD  = WkQ (end,:).';
WpD  = WpQ (end,:).';
VpD  = VpQ (end,:).';
FtD  = FtQ (end,:).';
Ft0D = Ft0Q(end,:).';
WtD  = 5.0e6*rBt*FtD;
Wt0D = 2.5e6*rBt*Ft0D;
WND  = 1e-7*pi*L.P.r0*IpD.^2;
iWND = 1./WND;
iWND(IpD==0) = 0; % bp, mu, li = 0
bpD  = 2/3*WkD.*iWND;
iWt0D = 1./Wt0D;
iWt0D(Ft0D==0) = 0; % bt = 0;
btD  = 2/3*WkD.*iWt0D;
liD  = WpD.*iWND;

% Centroids per domain
rIpD = zeros(L.nD,1);
zIpD = rIpD;
for iD=1:nB
  rIpD(iD) = sum(L.rry(Opy(:)==iD).*Iy(Opy(:)==iD));
  zIpD(iD) = sum(L.zzy(Opy(:)==iD).*Iy(Opy(:)==iD));
end
rYD = rIpD./IpD;
zYD = zIpD./IpD;


% Total integrals
Ip  = sum(IpD ,1);
Wk  = sum(WkD ,1);
Wp  = sum(WpD ,1);
Vp  = sum(VpD ,1);
Ft  = sum(FtD ,1);
Ft0 = sum(Ft0D,1);
Wt  = sum(WtD ,1);
Wt0 = sum(Wt0D,1);
WN  = 1e-7*pi*L.P.r0*Ip.^2;
if Ip== 0, iWN = 0; % bp, mu, li = 0
else,      iWN = 1./WN;
end
bp  = 2/3*Wk.*iWN;
bt  = 2/3*Wk./Wt0;
li  = Wp.*iWN;
mu  = Wt.*iWN;
bpli2 = bp + li/2;

% for the Ipmin case, recompute Ip from Iy
if Ip==0 || all(ag==0)
  Ip = sum(Iy(:));
end

% Centroids
rIp = sum(L.rry(:).*Iy(:));
zIp = sum(L.zzy(:).*Iy(:));
rY  = rIp/Ip;
zY  = zIp/Ip;

% q axis using small diamagnetic approximation (LIUQE 2015 paper eq. 88)
qA = zeros(nA,1);
[gAg,IgAg] = L.bfct(5,L.bfp,[],F0,F1); IgAg = 2e-7*L.idsx*IgAg;
for iA=1:nA
  TyAg = gAg(iA,:).*(rA(iA)*fPg+fTg/rA(iA)).'; % TyAg: Axis current (jphi.dR.dZ) contribution from each basis function
  cqA = 4e-7*pi*rA(iA)^2*rBt*sqrt(dr2FA(iA)*dz2FA(iA)-drzFA(iA)^2)/(abs(dr2FA(iA)+dz2FA(iA))*L.dsx);
  qA(iA) = (rBt.^2+(IgAg(iA,:).*fTg.')*ag)/(cqA*TyAg*ag);
end

% Profiles
[PpQ,TTpQ,PQ,TQ,iTQ,PpQg,TTpQg] = ...
  meqprof(fPg,fTg,ag,L.pQ(:).^2,F0,F1,rBt,L.bfct,L.bfp,L.idsx,L.smalldia);

% Fictional measurements
Bm  = L.G.Bma*Ia + L.G.Bmx(:,L.lxy)*Iy(:) + L.G.Bmu*Iu;
Ff  = L.G.Mfa*Ia + L.G.Mfx(:,L.lxy)*Iy(:) + L.G.Mfu*Iu;

% Vessel current from generic representation
Iv = L.G.Tvu*Iu;
Is = L.G.Tius*Iu;

% Store in LY
LY = meqlarg(LY,...
  Fx,rB,zB,FB,rA,zA,FA,lB,lX,dr2FA,dz2FA,drzFA,...                        % Flux map
  rX,zX,FX,nX,Opy,nA,nB,FR,dr2FX,dz2FX,drzFX,F0,F1,...                    % ...
  Iy,Ia,Iu,Iv,Is,Bm,Ff,ag,...                                             % Currents, measurements
  IpQ,VpQ,PpQg,TTpQg,PpQ,TTpQ,PQ,TQ,iTQ,FtPQ,...                          % integrals per Q
  IpD,zYD,rYD,zIpD,rIpD,WkD,WpD,WtD,Wt0D,WND,VpD,FtD,Ft0D,bpD,btD,liD,... % integrals per domain
  Ip,zY,rY,zIp,rIp,rBt,Wk,Wp,Wt,Wt0,WN,Vp,Ft,Ft0,bp,bt,mu,li,bpli2,qA...  % integrals
  );

if L.P.iterq
  LY = meqpostq(L,LY);
end

% optional interpolation of flux, fields in desired points
if L.nn
  [Fn,Brn,Bzn,Brrn,Brzn,Bzrn,Bzzn] = L.P.infct(L.rx,L.zx,Fx,L.P.rn,L.P.zn,L.inM);
else
  zn = zeros(0,1); Fn = zn; Brn=zn; Bzn=zn; Brrn=zn; Brzn=zn; Bzrn=zn; Bzzn=zn;
end
LY = meqlarg(LY,Fn,Brn,Bzn,Brrn,Brzn,Bzrn,Bzzn);

% optional computation of Br,Bz fields on x grid
if L.P.ifield
  [Brx,Bzx] = meqBrBz(Fx,L.i4pirxdzx,L.i4pirxdrx,L.nzx,L.nrx);
  Btx = meqBt(L,Fx,Opy,ag,rBt,F0,F1,TQ);
  LY = meqlarg(LY,Brx,Bzx,Btx);
end

% vacuum flux
if L.P.ivacuum
  if ~any(Iy(:)) % no plasma, Fx is already vacuum case
    F0x = Fx;
  else
    F0x = meqFx(L,zeros(L.nzy,L.nry),[Ia;Iu]); % recompute flux with Iy=0
  end
end

% optional computation of fields/fluxes on z grid
if L.P.izgrid

  lzx = L.G.lzx(:);
  % Compute whole grid with Green's functions 
  % This is faster than calculating a sub-part using logical indexing
  F0z = L.G.Mza*LY.Ia + L.G.Mzu*LY.Iu;
  Fz = F0z + L.G.Mzy*Iy(:);

  % overwrite part overlapping with x grid
  if L.P.izxoverlap 
    Fz(lzx) = Fx(:);
    if L.P.ivacuum,  F0z(lzx) = F0x(:); end
  end

  Fz  = reshape(Fz,L.nzz,L.nrz);
  F0z = reshape(F0z,L.nzz,L.nrz);

  LY.Fz = Fz;

  % fields 
  if L.P.ifield
    [Brz,Bzz] = meqBrBz(Fz,L.i4pirzdzz,L.i4pirzdrz,L.nzz,L.nrz);
    LY = meqlarg(LY,Brz,Bzz);
  end
end

% optional vacuum field/flux computation
if L.P.ivacuum
  LY.F0x = F0x;
  if ~any(Iy(:))
    % no plasma, vacuum values same as Br,Bz,Fx
    if L.P.izgrid,                LY.F0z  = Fz;                 end
    if L.P.ifield,                LY.Br0x = Brx; LY.Bz0x = Bzx; end
    if L.P.ifield && L.P.izgrid,  LY.Br0z = Brz; LY.Bz0z = Bzz; end
  else
    if L.P.izgrid,                LY.F0z = F0z;                 end
    if L.P.ifield,               [LY.Br0x,LY.Bz0x] = meqBrBz(LY.F0x,L.i4pirxdzx,L.i4pirxdrx,L.nzx,L.nrx); end
    if L.P.ifield && L.P.izgrid, [LY.Br0z,LY.Bz0z] = meqBrBz(LY.F0z,L.i4pirzdzz,L.i4pirzdrz,L.nzz,L.nrz); end
  end
end
    
% Additional output arguments
for k = 1:numel(L.argoutc)
  if strcmp(L.argoutc{k}.file,mfilename)
    LY.(L.argoutc{k}.fld) = eval(L.argoutc{k}.exp);
  end
end

end

function FR = calcFR(F0,F1,FX,lX)
%% Calculate ratio of normalized flux at x-point vs normalized flux at boudnary.
% For limited cases, FR>1 indicating the distance from the diverted case
% FR=1;
% For doublet cases, FR<1 for mantle domains, indicating the ratio of flux
% contained within the mantle w.r.t. maximum flux contained in the
% separatrix-enclosed domains.

sFBA = sign(F1(1)-F0(1)); % +1 is flux is increasing
nD = size(F1,1);
FR = zeros(nD,1);
for iD = 1:nD
  if F0(iD)==F1(iD),  FR(iD)=nan; % empty domain -> nan
  elseif lX(iD),      FR(iD)=1;   % diverted -> 1
  elseif isempty(FX), FR(iD)=inf; % no X points-> inf
  else
    FB = F1(iD);
    if any(F0(iD)==FX)
      % mantle domain - choose minimum axis flux and inner limiting X point.
      FA = sFBA*min(F0*sFBA);
      FXi = F0(iD);
      FR(iD) = (FXi-FA)/(FB-FA);
    else % non-diverted axis domain
      FA = F0(iD); % axis flux for this domain
      [~,iX] = min(abs(FX-FB)); % closest X point in terms of flux
      FXi = FX(iX);
    end
    FR(iD) = (FXi-FA)/(FB-FA);
  end
end
end

function [Br,Bz] = meqBrBz(Fx,i4pirdz,i4pirdr,nz,nr)
% [Br,Bz] = meqBrBz(Fx,i4pirdz,i4pirdr,nz,nr)
% Compute Br,Bz fields
% General version that also accepts time-varying Fx

[Br,Bz] = deal(zeros(nz,nr,size(Fx,3))); % init

% Br = -1/(2*pi*R)* dF/dz
% Central differences dF/dz[i] =  F[i-1] - F[i+1]/(2*dz)
Br(2:end-1,:,:) = -i4pirdz.* (Fx(3:end,:,:) - Fx(1:end-2,:,:));
% At grid boundary i, use: dF/dz[i] = (-F(i+2) + 4*F(i+1) - 3*F(i))/(2*dz)
Br(end,:  ,:) = -i4pirdz          .* (+Fx(end-2,:,:) - 4*Fx(end-1,:,:) + 3*Fx(end,:,:));
Br(1  ,:  ,:) = -i4pirdz          .* (-Fx(    3,:,:) + 4*Fx(    2,:,:) - 3*Fx(  1,:,:));

% Bz = 1/(2*pi*R)* dF/dr
Bz(:,2:end-1,:) =  i4pirdr(2:end-1) .* (+Fx(:,  3:end,:) - Fx(:,1:end-2,:));
% Same as for Br
Bz(:,end    ,:) =  i4pirdr(end)     .* (+Fx(:,end-2,:) - 4*Fx(:,end-1,:) + 3*Fx(:,end,:));
Bz(:,1      ,:) =  i4pirdr(1)       .* (-Fx(:,    3,:) + 4*Fx(:,    2,:) - 3*Fx(:,  1,:));
end

function Btx = meqBt(L,Fx,Opy,ag,rBt,F0,F1,TQ)
% Btx = meqBt(L,Fx,Opy,ag,rBt,F0,F1,TQ)
% Computes toroidal field on x grid

Btx  = rBt*repmat(1./L.rx',L.nzx,1);
Bty = Btx(L.lxy);

nB = numel(F1);
for iB = 1:nB
  Opyi = (Opy==iB); % mask for this domain
  if isequal(func2str(L.bfct),'bf3pmex')
    Btyi = L.bfct(8,L.bfp, Fx,F0(iB),F1(iB),int8(Opyi),ag(:,iB),rBt,L.idsx,L.iry);
    Bty(Opyi) = Btyi(Opyi);
  else
    % Mode 8 not yet available for other bfs, to be added later
    FyN = (Fx(L.lxy)-F0(iB))/(F1(iB)-F0(iB)); % rhopol.^2
    Bty(Opyi) = interp1(L.pQ.^2,TQ(:,iB)',FyN(Opyi))./L.rry(Opyi);
  end
end
Btx(L.lxy) = Bty;
end
