classdef meqplotslider < handle
  % MEQPLOTSLIDER plot equilibrium with slider for inspecting multiple slices
  %
  % meqplotslider(L,LY)        % basic call
  % meqplotslider(L,LX,LY)     % plotfun accepts 3 L,LX,LY
  % meqplotslider(L,{LY1,LY2}) % compare two sequences
  % NB: then plotting multiple LX or LY structures, no attempt is made
  % to match the time indices.
  %
  % meqplotslider(.... 'parameter',value) % pass more options
  %
  % optional parameter-value pair arguments
  %   parent: parent figure handle
  %   plotfun: function handle with signatures
  %            hax = plotfun(L,LY,'parent',hax)
  %            or hax = plotfun(L,LX,LY,'parent',hax)
  %            which plots a single slice on axis handle hax
  %   titlefun: title function with signature str = titlefun(L,LY);
  %
  % Use with GUIs:
  %  obj = meqplotslider(...)   % returns object for use in GUIs
  %  useful methods:
  %     obj.set_time_slice(it); % sets eq. plot to given time slice
  %  events:
  %     obj.updated_time % notification is sent when user changes eq. slice
  %
  %
  % SEE ALSO: MEQGUI, MEQPLOTTRACES
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  events
    updated_time
  end

  properties % public properties, e.g. their positions can later be moved around when setting up a GUI
    hf % figure handle
    slider % slider object handle
    ax % axis object handle
    text % time indicator text
    btn % play button
  end

  properties(Hidden)
    eq_update_fun % function handle for equilibrium update
  end

  methods
    function obj = meqplotslider(L,varargin)
      % constructor

      % parse inputs
      structargs = {}; cellargs = {};
      nt = 0; % init

      for ii=1:numel(varargin)
        myarg = varargin{ii};
        if isstruct(myarg)
          % structure inputs like LX or LY
          structargs = [structargs,myarg]; %#ok<AGROW>
          assert(isfield(myarg,'t'),'expected t field. Is this an LX or LY structure?')
          nt = max(nt,numel(myarg.t));
        elseif iscell(myarg)
          % cell inputs like {LY1,LY2}
          for jj = 1:numel(myarg)
            assert(isfield(myarg{jj},'t'),'expected t field. Is this an LX or LY structure?')
            nt = max(nt,numel(myarg{jj}.t));
          end
          cellargs = myarg;
        elseif ischar(myarg)
          % assume remaining parameters are 'parameter',value pairs
          varargin = varargin(ii:end);
          break
        end
        if ii==numel(varargin)
          % no parameter,value pairs passed
          varargin={};
        end
      end

      % parse parameters
      p=inputParser;
      p.addParameter('parent',[],@(x) isa(x,'matlab.ui.Figure'))
      p.addParameter('plotfun',@meqplott,@(x) isa(x,'function_handle'))
      p.addParameter('titlefun',[],@(x) isa(x,'function_handle'))
      p.parse(varargin{:});

      plotfun = p.Results.plotfun;
      titlefun = p.Results.titlefun;
      
      % setup figure
      if isempty(p.Results.parent)
        obj.hf = figure();
      else
        obj.hf = p.Results.parent; clf(obj.hf);
      end
      obj.hf.MenuBar = 'none';

      % create equilibrium axis
      obj.ax = axes(obj.hf);
      obj.ax.Box = 'on';
      obj.ax.Position = [0.05 0.2 0.9 0.7];
      if ~verLessThan('matlab','9.5')
        disableDefaultInteractivity(obj.ax); % turn off the little axis menu
      end

      % eq indicator
      obj.text = uicontrol(obj.hf,'style','text');
      obj.text.Units = 'normalized';
      obj.text.Position = [0.75 0.03 0.15 0.04];

      % slider bar
      obj.slider = uicontrol(obj.hf, 'style', 'slider','Tag','Slider');
      obj.slider.Units = 'normalized';
      obj.slider.Position = [0.05 0.02 0.7 0.05];
      obj.slider.Min = 1;
      obj.slider.Max = nt;
      obj.slider.Value = 1;
      obj.slider.SliderStep = [1 1] / (obj.slider.Max - obj.slider.Min);

      % play button
      obj.btn = uicontrol(obj.hf,'style','pushbutton','Tag','Play');
      obj.btn.Units = 'normalized';
      obj.btn.String = '>';
      obj.btn.Position = [0.90 0.03 0.05 0.04];
      obj.btn.Callback = @(src,event) obj.scroll_slider();

      % Function handle for updating equilibrium plot
      obj.eq_update_fun = @(it) obj.update_equilbrium_plot(it,...
        plotfun,titlefun,...
        L,structargs{:},cellargs... % L,LY or  L,LX,LY or L,{LY1,LY2} etc
        );

      %%% Set callbacks
      % slider value change
      obj.slider.Callback = @(src,event) set_time_slice(obj,round(src.Value));
      
      % key pressing
      set(obj.hf,'KeyPressFcn',...
        @(src,event) key_press_callback(obj,event))

      % draw first eq
      set_time_slice(obj,1)
    end

    function set_time_slice(obj,it)
      % update all graphics elements to time slice index it

      % limit to max/min
      it = apply_time_slice_limits(obj,it);

      % update slider value
      obj.slider.Value = it;

      % update equilibrium plot calling function handle
      time = feval(obj.eq_update_fun,it);

      % update text string
      obj.text.String = sprintf('%d/%d,t=%2.3f',it,obj.slider.Max,time);

      notify(obj,'updated_time'); % notify any listeners
    end

    function it = get_current_time_slice(obj)
      it = obj.slider.Value;
    end
  end


  methods(Hidden)

    function [it] = apply_time_slice_limits(obj,it)
      itmin = obj.slider.Min;
      itmax = obj.slider.Max;
      it = min(max(it,itmin),itmax); % clip
    end

    % Callback function for key press
    function key_press_callback(obj,event)

      current_value = obj.get_current_time_slice;

      switch event.Key
        case {'l', 'rightarrow'}
          new_value = current_value + 1;
        case {'h', 'leftarrow'}
          new_value = current_value - 1;
        case {'j', 'downarrow'}
          new_value = Inf;
        case {'k', 'uparrow'}
          new_value = -Inf;
        otherwise
          return; % Ignore other keys
      end
      % clip to min/max value
      new_value = apply_time_slice_limits(obj,new_value);
      obj.set_time_slice(new_value);
    end

  end

  methods(Hidden)

    function scroll_slider(obj)
      % scroll forward through slider values until max is reached
      % called when clicking the play button
      i1 = obj.slider.Value;
      imax = obj.slider.Max;
      for ii = i1:imax
        set(obj.slider,'Value',ii);
        obj.slider.Callback(obj.slider,[])
      end
    end

    function time = update_equilbrium_plot(obj,it,...
        plotfun,titlefun,...
        L,varargin)

      % update equilibrium indicator
      assert(it==round(it) && it>0,'must pass integer it>0')

      % update main equilibrium figure

      % parse inputs
      if numel(varargin) == 1 && iscell(varargin{1})
        % {LY1,LY2}, do plotfun looping over inputs
        obj.ax.ColorOrderIndex = 1; % reset color order index
        LYt = cell(1,numel(varargin{1}));
        cla(obj.ax);
        for ii=1:numel(varargin{1})
          myLY = varargin{1}{ii};
          LYt{ii} = meqxk(myLY,it);
          obj.ax = plotfun(L,LYt{ii},'parent',obj.ax);
          time = LYt{ii}.t; % time for display
        end
        if ~isempty(titlefun)
          % call title fun
          title(obj.ax,titlefun(L,LYt{:}))
        end
      elseif all(cellfun(@isstruct,varargin(1:end-1)))
        % LY or LX,LY. Pass all to plottfun
        % slice each input sepately
        LXYt = cell(1,numel(varargin)-1); % holds LX or LY
        for jj=1:numel(varargin)-1
          LXY = varargin{jj};
          LXYt{jj} = meqxk(LXY,it);
        end
        % pass all inputs
        obj.ax = plotfun(L,LXYt{:},'parent',obj.ax);
        time = LXYt{end}.t; % time for display

        if ~isempty(titlefun)
          % call title fun
          title(obj.ax,titlefun(L,LXYt{:}))
        end
      else
        error('unsupported call, use cell or struct inputs')
      end
    end
  end
end
