function hax = meqplotseq(L,LY,varargin)
%MEQPLOTSEQ Plots a time-sequence of MEQ equilibria
%
% h = meqplotseq(L,LY,'parameter',value,...)
%
% Inputs:
%   L : MEQ L structures from any code.
%   LY: MEQ LY structure. Equilibrium time slices to plot.
% 
%   If LY is empty but 'nt' is passed (see below), then just generate graphics
%   handles
%
%
%   Optional parameter-value pairs:
%     parent: figure handle on which to plot, OR... 
%             array of handles on which to plot (must match number of time slices)
%     nc: number of columns (default: auto-determined based on number of slices and figure aspect ratio)
%     nr: number of rows (default: auto-determined based on number of slices and figure aspect ratio)
%     plotfun: function handle for plotting function, signature for single time slice:  
%              plotfun(ax,L,LY)    by default, where ax is the target axis handle
%              plotfun(ax,L,LX,LY) if non-empty LX is passed
%     LX: Optional MEQ LX struture if required by plotting function.
%     top_spacing:   Offset between figure top    and first row    (figure normalized units)
%     bot_spacing:   Offset between figure bottom and last row     (figure normalized units)
%     left_spacing:  Offset between figure left   and first column (figure normalized units)
%     right_spacing: Offset between figure right  and last column  (figure normalized units)
%     row_spacing: Spacing between rows    (figure normalized units)
%     col_spacing: Spacing between columns (figure normalized units)
%     force: force plotting even for large amounts of slices (otherwise prompt user)
%     nt: If passing empty LY, an integer indicating how many axes to prepare and return hax
%
% Output:
%   h: array of axis handles, one per slice
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Defaults
default_plotfun = @(ax,L,LY) meqplott(L,LY,'parent',ax);
%% Prameter parsing
p=inputParser;
p.addParameter('parent',gcf,@(x) all(ishandle(x))); % parent figure handle
p.addParameter('plotfun',default_plotfun,@(x) isa(x,'function_handle')); % MEQ plotting function
p.addParameter('top_spacing',0.05); % figure edge offset
p.addParameter('bot_spacing',0.05); % figure edge offset
p.addParameter('left_spacing',0.05); % figure edge offset
p.addParameter('right_spacing',0.05); % figure edge offset
p.addParameter('row_spacing',0.01); % spacing between tiles
p.addParameter('col_spacing',0.02); % spacing between tiles
p.addParameter('LX',struct.empty,@(x) isstruct(x)); % optional LX structure
p.addParameter('nt',[],@(x) isscalar(x)); % number of slices if not passing LY
p.addParameter('force',false,@(x) isscalar(x)&&islogical(x))

% parse first set of parameters allowing unmatched ones
p.KeepUnmatched = true;
parse(p,varargin{:}); P = p.Results; LX=P.LX;

% checks
if isempty(LY)
  assert(~isempty(P.nt),'must pass nt if LY is empty');
else
  assert(isempty(P.nt),'nt must be empty if LY is not empty')
end

% number of time slices
if ~isempty(LY)
  if iscell(LY); nt = max(cellfun(@(Y) numel(Y.t),LY));
  else, nt = numel(LY.t); end
  noplot=false; 
else
  nt = P.nt; % custom passed
  noplot=true; 
end

if ~P.force && (nt>30)
  s=input(sprintf('attempting to plot more than 30 slices (%d) are you sure? Y/[N]',nt),'s');
  if ~strcmp(lower(s),'y'), hax=[]; return;  end
end

%% Set up axes
if isa(P.parent,'matlab.ui.Figure')
  % Set up axes from scratch
  
  % figure aspect ratio
  hf = P.parent; % figure handle
  pos = get(hf,'position'); Ar = pos(4)/pos(3); % height/width
  % optimal axes aspect ratio based on plotting grid
  if contains(func2str(P.plotfun),'fancy') % fancyplot uses z grid
    At = (max(L.G.zz)-min(L.G.zz))/(max(L.G.rz)-min(L.G.rz)); % z grid
  else
    At = (max(L.G.zx)-min(L.G.zx))/(max(L.G.rx)-min(L.G.rx)); % x grid
  end
  [nr,nc] = find_optimal_rows_columns(At,Ar,nt,P);
  
  % now we have default nc, nr and we can parse the user input
  p.addParameter('nc',nc); % number of columns
  p.addParameter('nr',nr); % number of rows
  
  % parse everything again
  parse(p,varargin{:}); P = p.Results;
  
  %% setup axes based on nr,nc and spacing setup
  width  = (1-P.left_spacing-P.right_spacing-(P.nc-1)*P.col_spacing)/P.nc;
  height = (1-P.top_spacing -P.bot_spacing  -(P.nr-1)*P.row_spacing)/P.nr;
  hax = gobjects(nt,1); % init
  for kt=1:nt
    irow = ceil(kt/P.nc);
    icol = rem(kt-1,P.nc)+1;
    pos = [P.left_spacing + (icol-1)*(width+P.col_spacing),  P.bot_spacing + (P.nr-irow)*(height+P.row_spacing), width,height]; % axis position
    hax(kt) = axes('position',pos);
    set(hax(kt),'xticklabel','','yticklabel','','box','on');
  end
  if noplot, return; end % return directly, no plotting
elseif isa(P.parent,'matlab.graphics.axis.Axes')
  % axes passed by user
  hax = P.parent;
  assert(~isempty(LY) && numel(LY.t) == numel(hax),'number of axes must match number of time slices in LY')
else
  error('unrecognized class of ''parent'', %s',class(P.parent));
end

%% Loop to plot all slices
for kt = 1:nt
  LYt = meqxk(LY,kt);
  ax = hax(kt);
  if isempty(LX) % case without LX
    P.plotfun(ax,L,LYt);
  else % case with LX
    LXt = meqxk(LX,kt);
    P.plotfun(ax,L,LXt,LYt);
  end
end

end

function [nr,nc] = find_optimal_rows_columns(At,Ar,nt,P)
% Find optimal number of rows/columns based on Area covered
area = zeros(1,nt);
% try all possible number of rows
for nr = 1:nt
  nc = ceil(nt/nr); % columns

  colsp = (nc-1) * P.col_spacing + P.left_spacing + P.right_spacing; % col white spacing
  rowsp = (nr-1) * P.row_spacing + P.bot_spacing  + P.top_spacing; % row white spacing
  
  % figure has width 1 and height Ar*1, axis has width w and height At*w
  wmaxw = ( 1-colsp)/nc;    % maximum axis width to have total width 1
  wmaxh = (Ar-rowsp)/nr/At; % maximum axis width to have total height Ar
  width = min(wmaxw,wmaxh); % minimum one that fits the figure
  height = At*width;

  % total area covered with axes
  area(nr) = width * height * nt;
end

% choose the one with largest covered area
[~,nr] = max(area);
nc = ceil(nt/nr); % columns
end
