function hax = meqplotcontours(L,LY,varargin)
% MEQPLOTCONTOURS basic plotting of flux and LCFS contours
%
% ax = meqplotcontours(L,LY)
% ax = meqplotcontours(L,LY,'parent',ax); % pass parent axis handle
%      meqplotcontours(....,'color','r'); % pass LCFS color
%      meqplotcontours(....,'parameter',value); % other options passed to contour() calls
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if numel(LY.t) > 1
  error('supports only one LY slice, use meqxk() to slice')
end

p=inputParser();
p.KeepUnmatched = true; % additional parameters are passed to contour() and will throw an error if invalid
p.addParameter('Parent',[])
p.addParameter('Color',[])
p.parse(varargin{:});

hax = p.Results.Parent;
color = p.Results.Color;
plotopts = p.Unmatched; % other options passed
plotopts = [fieldnames(plotopts),struct2cell(plotopts)]';

if isempty(hax) || ~isgraphics(hax,'axes')
  hax = gca;
end

if isempty(color)
  color = hax.ColorOrder(hax.ColorOrderIndex,:);
end

% flux color similar to LCFS color but shifted towards white
lam = 0.2; % interpolation parameter
fluxcolor = (1-lam)*color + lam*[1 1 1];

switch LY.nA
  case 0
    LCFS_levels = [];
    flux_levels = linspace(min(LY.Fx(:)),max(LY.Fx(:)),41);
  case 1
    LCFS_levels = LY.F1*[1 1];
    in_plasma   = (LY.F1-LY.F0) * linspace(0,1,11) + LY.F0;
    out_plasma  = (LY.F1-LY.F0) * linspace(0,3,31) + LY.F1;
    flux_levels = sort([in_plasma,out_plasma]);
  otherwise
    LCFS_levels = LY.F1;
    in_plasma   = (LY.F1(end)-LY.F0(1)) * linspace(0,1,11) + LY.F0(1);
    out_plasma  = (LY.F1(end)-LY.F0(1)) * linspace(0,3,31) + LY.F1(end);
    flux_levels = sort([in_plasma,out_plasma]);
end

hold(hax,'on')
contour(hax,L.rx,L.zx,LY.Fx,LCFS_levels,...
  'LineColor',color,'LineWidth',2,...
  'tag','LCFS',plotopts{:});
contour(hax,L.rx,L.zx,LY.Fx,flux_levels,...
  'LineColor',fluxcolor,'tag','flux',plotopts{:});

plot(hax,LY.rA,LY.zA,'o','Color',color)
plot(hax,LY.rX,LY.zX,'x','Color',color)
plot(hax,LY.rB,LY.zB,'s','Color',color)

axis(hax,'equal');

end
