function meqplotQ(L,varargin)
%MEQPLOTQ: compare profiles and geometry for static equilibria with the same L structure
% meqplotQ(L,LY1,LY2)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% post processing was run for at least one LY
ipost = any(cellfun(@(x)isfield(x,'iqQ'),varargin)); 

if ipost, nr = 4; else, nr=2; end; % rows for plots
hs(1) = subplot(1 ,3, 1);
hs(2) = subplot(nr,3, 2);
hs(3) = subplot(nr,3, 3);
hs(4) = subplot(nr,3, 5);
hs(5) = subplot(nr,3, 6);
if ipost
  hs(6) = subplot(nr,3, 8);
  hs(7) = subplot(nr,3, 9);
  hs(8) = subplot(nr,3,11);
  hs(9) = subplot(nr,3,12);
end

meqgplot(L.G,hs(1),'vl'); hold(hs(1),'on');

for ii=1:numel(varargin)
  LY = varargin{ii};
  
  % postprocessing is available for this LY
  ipost = isfield(LY,'iqQ'); 
  
  %% flux plots
  if LY.nA>0 && LY.nB>0
    F = linspace(LY.F0(1),2*LY.F1(LY.nB)-LY.FA(1),21);
    plotLCFS = true;
  else
    F = linspace(min(LY.Fx(:)),max(LY.Fx(:)),21);
    plotLCFS = false;
  end

  ax=hs(1);
  col = ax.ColorOrder(ii,:);
  
  contour(ax,L.G.rx,L.G.zx,LY.Fx,F,'linecolor',col);
  if plotLCFS
    contour(ax,L.G.rx,L.G.zx,LY.Fx,[1 1]*LY.F1(LY.nB),'linecolor',col,'linewidth',2);
  end

  plot(ax,LY.rA,LY.zA,'ow');
  plot(ax,LY.rB,LY.zB,'gs');
  
  axis(ax,'equal','tight');
  
  if LY.nB<=2 % 1 or 2 domains
    FN = L.pQ.^2;
  elseif LY.nB==3 % doublet
    
    sIp = sign(LY.Ip);
    FLobes = sIp*(LY.F0(1:2)-LY.F1(1:2))*(L.pQ.^2-1);
    Fmantle = sIp*(LY.F0(3)-LY.F1(3))*L.pQ.^2;
    FN = ([FLobes;Fmantle]' - min(FLobes(:)))/(max(Fmantle)-min(FLobes(:)));
  end
  
  %% profiles
  ax = hs(2);
  hp{1} = plot(ax,FN,mu0*LY.PpQ,'color',col);
  hold(ax,'on');
  ylabel(ax,'\mu_0p'''); xlabel(ax,'\psi_N');
  
  ax = hs(3);
  hp{2} = plot(ax,FN,LY.PQ/1e3,'color',col);
  hold(ax,'on');
  ylabel(ax,'p [kPa]'); xlabel(ax,'\psi_N');
  
  ax = hs(4);
  hp{3} = plot(ax,FN,LY.TTpQ,'color',col);
  ylabel(ax,'TT''');xlabel(ax,'\psi_N');
  hold(ax,'on');
  
  ax = hs(5);
  hp{4} = plot(ax,FN,LY.TQ,'color',col);
  ylabel(ax,'T [Wb]');xlabel(ax,'\psi_N');
  hold(ax,'on');
  
  if ipost
    ax = hs(6);
    hp{5} = plot(ax,FN,1./LY.iqQ,'color',col);
    ylabel(ax,'q'); xlabel(ax,'\psi_N');
    hold(ax,'on');
    
    ax = hs(7);
    hp{6} = plot(ax,FN,LY.kappa,'color',col);
    ylabel(ax,'\kappa'); xlabel(ax,'\psi_N');
    hold(ax,'on');
    
    ax = hs(8);
    hp{7} = plot(ax,FN,LY.jtorQ*1e-6,'color',col);
    ylabel(ax,'j_{tor} [MA/m^2]');  xlabel(ax,'\psi_N');
    hold(ax,'on');
    
    ax = hs(9);
    hp{8} = plot(ax,FN,LY.delta,'color',col);
    ylabel(ax,'\delta');  xlabel(ax,'\psi_N');
    hold(ax,'on');
  end
 
    
  % adjust line styles
  for ip=1:numel(hp) % loop over plots
    hpp = hp{ip};
    for jj=1:numel(hpp) % loop over lines
      switch jj
        case 1, hpp(jj).LineStyle = '-';
        case 2, hpp(jj).LineStyle = '--';
        case 3, hpp(jj).LineStyle = '-.';
      end
    end
  end
end
