function [rA,zA,FA,dr2FA,dz2FA,drzFA,rX,zX,FX,dr2FX,dz2FX,drzFX,...
  rB,zB,FB,lB,lX,Opy,F0,F1,status,msg,id,dF0dFx,dF1dFx,ixI] = meqpdom(Fx,Ip,isaddl,L)
%MEQPDOM  Plasma domain parameters
% [RA,ZA,FA,DR2FA,DZ2FA,DRZFA,RX,ZX,FX,DR2FX,DZ2FX,DRZFX,
% RB,ZB,FB,LB,LX,OPY,F0,F1,STATUS,MSG,ID,DF0DFX,DF1DFX,IXI] = MEQPDOM(FX,IP,ISADDL,L)
%
% MEQPDOM returns position, flux gradients of magnetic axes, x points,
% boundary points, as well as plasma domain parameters.
%
% FX(Z,R) is the flux map 
% IP plasma current for its sign, IP=0 corresponding to a vacuum case
% ISADDL selects the use of X-points 0: exclude x-points 1: use x-points
%
% RA,ZA,FA position and flux of the magnetic axes
% DR2FA,DZ2FA,DRZFA flux 2nd derivatives on axes, 
% RX,ZX,FX position and flux of the X-points on the X-points polygon,
% DR2FX,DZ2FX,DRZFX flux 2nd derivatives at X-point,
% RB,ZB,FB point and flux defining LCFS or internal separatrices
% LB is FALSE if no valid point defines the LCFS 
% LX is TRUE if an X-point defines the LCFS or separatrix
% OPY(Z,R) indices indicate different plasma domain
% F0,F1 limiting flux values for each domain.
% STATUS is TRUE if the domain identification is successful
% MSG,ID contain a message and identifier describing the issue when STATUS
%   is FALSE
% dF0dFx,dF1dFx (optional outputs) derivatives of F0 and F1 w.r.t the Fx
%   (sparse matrices if P.icsint=false)
% ixI (optional output) index of grid cells where central points were found
%
% For details, see: [J-M. Moret et al. Fus.Eng.Des 2015], Section 2.3
% Multi-domain case for doublets extension added later
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

dogradient = nargout>23;

sIp = sign(Ip);
Opy = zeros(L.nzy,L.nry,'int8');
rx = L.G.rx; zx = L.G.zx;

FN = L.FN*sIp;

if all(Fx(:)==Fx(1)) % all the same flux, happens for vacuum case with Ia=0
  zA=zeros(0,1); rA=zA; FA=zA; dz2FA=zA; dr2FA=zA; drzFA=zA; ixA=zA; 
  zX=zA;         rX=zA; FX=zA; dz2FX=zA; dr2FX=zA; drzFX=zA; ixX=zA; stat=1;
else
  % find all x/o points
  [zA,rA,FA,dz2FA,dr2FA,drzFA,ixA,zX,rX,FX,dz2FX,dr2FX,drzFX,ixX,stat] = ...
    asxymex(Fx,L.G.zx,L.G.rx,L.P.dasm,L.dzx,L.drx,L.idzx,L.idrx,L.Oasx,L.dimw);
end
if (~stat)
  % Issue warning only, errors handled if nA~=1.
  % Note: should never happen if itert=0 since dimw=numel(L.Oly)
  s=warning('off','backtrace');
  warning('asxymex:workingSpace','Working space (%d) too small, increase DIMW',L.dimw);
  warning(s);
end

nA = numel(FA); nX = numel(FX);

if L.P.icsint
  [zA,rA,FA,dz2FA,dr2FA,drzFA,zX,rX,FX,dz2FX,dr2FX,drzFX] = ...
    asxycs(Fx,zA,rA,zX,rX,L);
end

% Check for vacuum case
isvacuum = nA == 0 || sIp ==0;
% When this condition is met, no domains can be found hence FN will not be used

%% LCFS flux
% Limiter flux
if L.P.ilim == 0 || isvacuum
  Fl = zeros(0,1);
  drFl = Fl;
  dzFl = Fl;
  rl = Fl;
  zl = Fl;
elseif L.P.ilim == 1 % default limiter treatment - only on rl,zl points
  % interpolate and keep only compatible sign of extrema
  [Fl,drFl,dzFl] = fl4pmex(Fx,int32(L.kxl-1),L.clx,FN);
  rl = L.G.rl;
  zl = L.G.zl;
elseif L.P.ilim == 2 % interpolated flux extrema on limiter
  % interpolate on linear segments and keep only compatible sign of extrema
  [rl,zl,Fl,drFl,dzFl] = fl4pinterp(Fx,L.kxl,L.clx,L.kxlh,L.clhx,FN,L.G.rl,L.G.zl);
elseif L.P.ilim == 3 % Use cubic spline for Fx interpolation and pp interpolation for the limiter
  [Fl,rl,zl,drFl,dzFl] = flcs(Fx,FN,L);
else
  error('unknown ilim value %s',L.P.ilim)
end

% Keep only valid limiter candidates
%   NB: The rest of the code works without this but this saves some computation.
%   In Simulink (where only ilim=1 is implemented), the whole array is kept
%   since rl,zl are passed as parameters to fbndmex.
if L.P.ilim~=0 && ~isvacuum
  maskl = (Fl ~= FN);
  rl    = rl(maskl);
  zl    = zl(maskl);
  Fl    = Fl(maskl);
  drFl  = drFl(maskl);
  dzFl  = dzFl(maskl);
  if dogradient && L.P.ilim==1
    % 4-point interpolation data
    kxl = L.kxl(maskl);
    clx = L.clx(:,maskl);
  end
end

if ~(L.P.ihole || isvacuum)
  % Eliminate axes with wrong hessian, assuming all axes should have similar signs for current.
  iAok = (-sign(dr2FA+dz2FA) == sIp);
  FA=FA(iAok); rA=rA(iAok); zA=zA(iAok); dr2FA=dr2FA(iAok); dz2FA=dz2FA(iAok); drzFA=drzFA(iAok); ixA = ixA(iAok);
  nA = numel(FA);
end
    
% order FA by vertical position, FX by flux value
[zA,k] = sort(zA,'descend'); FA = FA(k); rA = rA(k); dr2FA = dr2FA(k); dz2FA = dz2FA(k); drzFA = drzFA(k); ixA = ixA(k);
    
% Use X point hessians to compute directions of X point
% (towards positive gradient direction)
H1 = sIp*(dr2FX - dz2FX) * 0.5;
H0 = sqrt( H1.*H1 + drzFX.*drzFX);
vrX0 = sqrt(H0 + H1);
vzX0 = sqrt(H0 - H1) .* ((drzFX >= 0)*2 -1)*sIp;
        
%% Determine domain for each of the regions
np = nA+nX;                % Number of potential domains
if isvacuum
  np = 0;
end
kB = zeros(np,1);          % Index of X-point or limiter point for domain
FB = kB; rB = kB; zB = kB; % Flux and coordinates of boundary point
FI = kB; rI = kB; zI = kB; % Flux and coordinates of center point
lX = false(np,1);          % Flag for diverted domain
lB = false;                % Flag for successful domain identification
kX = zeros(nX,1);          % Number of domains connected to X-point
IX = true(nX,1);           % Flag for X-points yet to be treated
ixI = kB;                  % Index of grid point linked to center point (6-point interpolation)

for kp = 1:np
  FX_ = FX; % Restore all X-points as candidates
  Fl_ = Fl; % Restore all limiter point candidates
  if kp <= nA
    % First treat domains that contain an extremum (axis)
    rI(kp) = rA(kp); zI(kp) = zA(kp); FI(kp) = FA(kp); ixI(kp) = ixA(kp);
 
    vrX = rI(kp)-rX;
    vzX = zI(kp)-zX;
  else
    % Select first X-point untreated and connected to 2 already identified domains
    mask = IX & (kX > 1);
    k = find(mask,1);
    if isempty(k), break, end % If none is found, all remaining X-points are connected to a single domain and their separatix will intersect the wall. So our job is done.
    FX_(kX>1) = FN; % Eliminate X-points already connected to at least 2 domains
    IX(k) = false; % X-point has been treated
    rI(kp) = rX(k); zI(kp) = zX(k); FI(kp) = FX(k); ixI(kp) = ixX(k);

    vrX = vrX0;
    vzX = vzX0;
  end

  % use bavxmex to 'cut' regions outside of X-point polygon
  vsX = sign( vrX.*(rI(kp)-rX) + vzX.*(zI(kp)-zX));
  vrX = vrX .* vsX;
  vzX = vzX .* vsX;
  % Eliminate X-points outside of X-point polygon
  selX = FX_ ~= FN;
  k = bavxmex(rX(selX),zX(selX),vrX(selX),vzX(selX),0,rX,zX);
  FX_(~k | ~isaddl) = FN; % isaddl=0 excludes all X-points
  % cut regions to be excluded on (r,z)
  OX = bavxmex(rX(selX),zX(selX),vrX(selX),vzX(selX),0,rx,zx,[]);
  
  % Exclude limiter points with wrong sign of derivatives
  dAFl = (drFl.*(rI(kp)-rl) + dzFl.*(zI(kp)-zl));
  Fl_(dAFl*FN > 0) = FN;
  % Exclude candidate limiter points outside x-point polygon  
  Fl_(~bavxmex(rX(selX),zX(selX),vrX(selX),vzX(selX),L.P.xdoma,rl,zl)) = FN;
  
  % Boundary flux from limiter points or X-points
  [FB(kp),rB(kp),zB(kp),lB,lX(kp),kB(kp)] = fbndmex(Fl_,rl,zl,FX_,rX,zX,FN);
      
  if ~lB, break; end

  % Increase number of domains associated to X-point
  if lX(kp), kX(kB(kp)) = kX(kB(kp))+1; end
   
  Oxyk = OX & (Fx -FI(kp)).*(Fx - FB(kp)) <= 0;

  Opy(Oxyk(L.lxy)) = kp; % indicized Opy
end

% truncate to effective number of domains
nB = find(rB>0,1,'last');
if isempty(nB), nB = 0; end
FB = FB(1:nB,1); rB = rB(1:nB,1); zB = zB(1:nB,1); lX = lX(1:nB,1); kB = kB(1:nB,1);
% mask outside limiter
Opy(~L.Oly) = 0;

if L.nD == 1 && nB == 1
  % Order X-points by proximity to FB
  [~,k] = sort(abs(FX-FB));
else
  % Order X-points by flux value
  [~,k] = sort(FX,'descend');
end
ixX = ixX(k);
FX = FX(k); zX = zX(k); rX = rX(k);
dr2FX = dr2FX(k); dz2FX = dz2FX(k); drzFX = drzFX(k);
% Inverse of k permutation
ik(k) = 1:nX; kB(lX) = ik(kB(lX));

% Limiting values for iD=1..L.nD
F0 = zeros(L.nD,1); F1 = F0; nBeff=min(L.nD,nB);
F0(1:nBeff) = FI(1:nBeff);
F1(1:nBeff) = FB(1:nBeff);

%% Checks
pla = sIp~=0;
status = false;
if pla && nA==0
  msg = 'No magnetic axis';
  id  = 'NoMagAx';
elseif pla && ~lB
  msg = 'could not find LCFS';
  id  = 'NoLCFS';
elseif any(dr2FA.*dz2FA<drzFA.^2)
  msg = 'Hessian at magnetic axis has det<0';
  id  = 'MagAxHess';
elseif nB > L.nD
  msg = sprintf('number of found domains (%d) exceeds maximum number (%d)',nB,L.nD);
  id  = 'TooManyDomains';
else
  msg = '';
  id = '';
  status = true;
end

%% Gradients
if dogradient
  % Allocate
  % When using cubic spline interpolation the flux at the axis or boundary
  % point depends on the value of all nodes in the grid, but without it a
  % local interpolation scheme is used which warrants the use of sparse
  % matrices to represent the jacobians
  if L.P.icsint
    % Use full matrices
    dF0dFx = flcsJac(rI(1:nBeff),zI(1:nBeff),L);
    dF1dFx = flcsJac(rB(1:nBeff),zB(1:nBeff),L);
    if nBeff<L.nD
      % Pad if needed
      dF0dFx(nBeff+1:L.nD,:) = 0;
      dF1dFx(nBeff+1:L.nD,:) = 0;
    end
  else
    % Use sparse matrices
    nstencil = 6; % 6-point interpolation
    dF0dFx = sparse([],[],[],L.nD,L.nx,nstencil*nBeff);
    dF1dFx = sparse([],[],[],L.nD,L.nx,nstencil*nBeff);
    %
    % Center points and X-points which are also boundary points
    maskX = kB(lX(1:nBeff));
    ixs = [ixI(1:nBeff,1);ixX(maskX,1)];
    dFsdFx = asxyJac(Fx,L.dzx,L.drx,L.idzx,L.idrx,ixs);
    dF0dFx(   1:nBeff ,:) = dFsdFx(1:nBeff,:);
    dF1dFx(lX(1:nBeff),:) = dFsdFx(nBeff+1:end,:);

    % Limiter points
    maskl = kB(~lX(1:nBeff));
    if ~isempty(maskl)
      if L.P.ilim == 0 % No limiter points
      elseif L.P.ilim == 1 % default limiter treatment - only on rl,zl points
        iB = repmat(find(~lX(1:nBeff)),1,4);
        ix = kxl(maskl)+[0,L.nzx,L.nzx+1,1];
        dF1dFx = dF1dFx + sparse(iB,ix,clx(:,maskl).',L.nD,L.nx);
      elseif L.P.ilim == 2 % interpolated flux extrema on limiter
        % TODO
      elseif L.P.ilim == 3 % Use cubic spline for Fx interpolation and pp interpolation for the limiter
        % This case cannot be reached as it requires icsint=true
      end
    end
  end
end
end
