function meqpath(code,tok)
%MEQPATH Checks and adds genlib and device-specific folders
%
% meqpath(code,tok) does the following:
% * Checks whether genlib is on the path by checking for presence of greenem.m
%   If it is not, it looks for a local genlib folder and adds that to the path.
% * Checks whether solver is a subpath path of the main meq path by checking for
%   presence of solveF.m. If not, adds the local solver/ folder to the path
% * Checks presence of meq[p,g]tok.m, <code>[p,g,x]tok.m files
%   If they are not on the path, attempts to add a local folder <tok> to the
%   path.
% 
% code can be any of the MEQ codes: fbt,liu, fgs etc. 
% tok is the short tokamak description string
% 
% This function is typically called at the beginning of the high-level MEQ
% functions so users usually do not need to call it separately.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

meqdir = fileparts(mfilename('fullpath')); % folder containing this file

% Check genlib
if isempty(which('greenem'))
  genlibpath = fullfile(meqdir,'genlib');
  path_to_add  = {genlibpath};
  fprintf('adding genlib directory %s to path\n',genlibpath)
else
  path_to_add = {};
end

% Check solver
if isempty(which('solveF')) || ~isequal(fileparts(fileparts(which('solveF'))),meqdir)
  solverpath = fullfile(meqdir,'solver');
  path_to_add=[path_to_add,solverpath];
  fprintf('adding solver directory %s to path\n',solverpath)
end

% look for meqp<tok>.m file
meqptok = strcat('meqp',tok);
if isempty(which(meqptok))
  % try to add a device folder in the default location
  dirname = fullfile(meqdir,tok);
  if ~exist(dirname,'dir')
    error('%s not found and no directory %s found either. Please add MEQ device-specific files for tok=%s to the path, or check spelling.',meqptok,dirname,tok)
  else
    fprintf('adding device directory %s to path\n',dirname)
    path_to_add = [path_to_add,dirname];
  end
end

% add all paths in one call
if ~isempty(path_to_add)
  addpath(path_to_add{:});
end

% now check that genlib is there
assert(~isempty(which('greenem')),...
  'The greenem function from the genlib toolbox was not found, check that the submodule was initialized properly (see README.md) or add your custom genlib path to the MATLAB path');

% now check that solver is there
assert(~isempty(which('solveF')),...
  'The solveF function from the solver folder was not found, check that the folder exists or add your custom solver path to the MATLAB path');

% now check that all files are there
reqfiles = strcat('meq',{'p','g'},tok);
if any(strcmp(code,{'fge','rzp'}))
  pgx = {'p','g'}; % no fge/rzpx<tok>
else
  pgx = {'p','g','x'};
end
reqfiles = [reqfiles,strcat(code,pgx,tok)];

for file = reqfiles
  assert(~isempty(which(file{:})),'could not find %s in the MATLAB path, while it is needed to run %s for tok=%s',file{:},code,tok)
end
  
end


