function [xs,us,zs,LYs,dxs_,conv,dzg] = meqopt(L,LXs,...
  xs,us,zs,sinit,init_response_matrix,RaUt,ReUt,RiUt,...
  Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c,...
  IDs,ws,TusU,TzsU,d1dt,d2dt,...
  idts,picard_Precs,dxdus,dxdxps,dxs_,Uscale,masku,...
  msgtype,qpsolve,qpopts,ctol,zfct)
% MEQOPT Grad-Shafranov optimizer based on FGS/FGE
%
%   [xs,us,zs,LYs,dxs_,conv] = meqopt(L,LXs,...
%     xs,us,zs,sinit,lambda,sinit,Ra,RaUt,Re,ReUt,Ri,RiUt,dxs,dus,...
%     Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c,...
%     IDs,ws,bbs,TusU,TzsU,d1dt,d2dt,...
%     idts,picard_Precs,dxdus,dxdxps,dxs_,Uscale,...
%     msgtype,qpsolve,qpopts,ctol)
%
% Inputs:
%  * L: FBT/LIU L structure
%  * LXs: FBT/LIU LX structure, possibly multi-slice
%  * xs,us,zs: cell arrays with initial estimates of the state (xs) inputs
%    (us) and optimization variables (zs)
%  * sinit: status of initialization phase
%  * init_response_matrix: if true, RaUt,ReUt,RiUt values passed as
%    arguments have been initialized and can be used
%  * RaUt,ReUt,RiUt: initial values of reduced residual jacobians
%  * Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c: cell
%    arrays with optimization problem matrices for each time slice
%  * IDs,ws: cell arrays of ID, w (see FBTOPT for details)
%  * TusU,TzsU: Transfer matrices between us, zs and U
%  * d1dt,d2dt: Stencils for 1st and 2nd order time derivatives
%  * idts: inverse of each time step in the interval (0 for static  solution)
%  * picard_Precs: cell array of Picard 'preconditioner'
%  * dxdus, dxdxps: cell arrays containing for each time the state
%    dependence over the inputs and the state dependence over the previous
%    state
%  * dxs_: Initial guess for inversion of Jacobian
%  * Uscale: typical scale of U (optimization parameters)
%  * masku: Logical array indicating which FGS/FGE inputs are optimized
%  * msgtype: type of exception handling, 'e'/'w' for errors or warnings
%  * qpsolve: function handle to QP solver
%  * qpopts: cell array with options for QP solver
%  * ctol: constraint tolerance
%  * zfun: vertical position stabilisation function handle (LIU only)
% Outputs
%  * xs,us,zs: cell arrays with final estimates of the state (xs) inputs
%    (us) and optimization variables (zs)
%  * LYs: LY structure corresponding to final state
%  * dxs_: Final estimate for inversion of Jacobian
%  * conv: convergence flag
%  * dzg: Fitted vertical shift (LIU only)
%
% Details:
%
% This is a general formulation based on reduced-hessian SQP method.
% The approach is similar to the one in NICE [B. Faugeras et al. FED 2020].
% If the FGE jacobian is computed using the approximation dIydFx=0 and
% dCodFx=0 we recover the Picard iteration scheme from both the original
% FBT algorithm [Hofmann CPC 1988] as well as the GSPD algorithm [J. Wai 
% et al, arXiv 2023].
%
% Given a cost function W(x,u,z) and some non-linear constraints F(x,u)=0
% such that Jx=dF/dx is non-singular representing the physics model, the
% optimal solution is found by the following iteration scheme:
%   x(k+1) - x(k) = -Jx\F(k) - (Jx\Ju)*(u(k+1) - u(k))
%                 = dx0      + (dxdu) *(du)
%   du = u(k+1) - u(k) 
%   dz = z(k+1) - z(k) 
%   where du,dz are the solution of the optimization of the reduced
%   cost function Wt(du,dz) = W(x(k)+dx0+dxdu*du,u(k)+du,z(k)+dz)
%
% Since no assumptions were made to the structure of W, it also applies to
% constrained problems.
%
% This is further extended to evolutive problems by considering the
% dependence of F on the previous state xp. Considering the cost function
% for N time slices W(x[1],x[2],...,x[N],u[1],u[2],...,u[N],z[1]..z[N]) and our
% non-linear constraints F(x[i],u,x[i-1])=0
%  - the first time slice is treated in the same way as the
%  time-independent problem:
%   x[1](k+1) - x[1](k) = -Jx\F[1](k) - (Jx\Ju)  *(u[1](k+1) - u[1](k))
%                       = dx0[1]      + (dxdu[1])*(du[1])
%  - For the subsequent ones:
%   x[i](k+1) - x[i](k) = -Jx\F[i](k) - (Jx\Ju)  *(u[i](k+1) - u[i](k)) - (Jx\Jxp)*(x[i-1](k+1)-x[i-1](k))
%                       = dx0[i]      + (dxdu[i])*(du[i])               + (dxdxp[i])*(dx[i-1])
% The reduced cost function is then formed by expressing dx[i] as a
% function of dx0[k] and du[k] for k=1...i:
%   dx[i] = dx0[i]        + dxdxp[i]*(dx0[i-1]           + dxdxp[i-1]*(... + dxdxp[2]*dx0[1]       )...)
%         + dxdu[i]*du[i] + dxdxp[i]*(dxdu[i-1]*(du[i-1] + dxdxp[i-1]*(... + dxdxp[2]*dxdu[1]*du[1])...)
% which gives
%   Wt(du[1],du[2],...,du[N],dz[1]...dz[N]) = 
%         W(x[1]+dx[1],x[2]+dx[2],...,x[N]+dx[N],u[1]+du[1],u[2]+du[2],...,u[N]+du[N],z[1]+dz[1],..,dz[N]+dz[N])
% And we optimize for all dus at once.
%
% Note: In MEQ we define Jxdot = -dt*dF/dxp = -dt*Jxp.
%
% Code nomenclature:
%   - x is the FGE nonlinear state vector
%   - xp is the FGE nonlinear state vector from the previous time slice
%   - u is a subset of the FGE external inputs that are part of the
%   optimization variables (u=Ia for the static case, u=Va for the
%   evolutive case)
%   - z is an additional optimization variable that only affects the cost function
%   - U is the complete set of optimization variables for the reduced
%   optimization problem. For FBT, these are the union of u and Fb). For FBT evolutive problems,
%   U=(Va[1],Va[2],...,Va[N],Fb[1],Fb[2],...Fb[N]). For LIU these are u and dz
%   - the suffix t is used for quantities related to the reduced
%   optimization problem (e.g. Rat = Ra0 + Rax*(x + dx0) + Rau*u,
%   Raut = Rau + Rax*dxdu)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Debug options
% debug>0: text output for each time slice
% debug>1: text output for each iteration
% debugplot=1: Only plot at end of time iteration or on error,
% debugplot=2; Plot all iterations
dotxt = L.P.debug>0;
dotxi = L.P.debug>1;
doplt = L.P.debugplot>0;
dopli = L.P.debugplot>1;
dopost = dotxt || dopli; % post-processing at fgeF call after iteration

%% NL method
xHasIy = strcmpi(L.P.algoNL,'all-nl');
% FGS/FGE operator options
%  - picard iterations: use function handle
opts_pic = optsF;
opts_pic.dojacF    = true;
opts_pic.dopost    = dopost;
%  - newton iterations
opts_sqp = optsF;
opts_sqp.dojacx    = ~L.P.jacobian_handle;
opts_sqp.dojacF    =  L.P.jacobian_handle;
opts_sqp.dojacu    = true;
opts_sqp.dojacxdot = true;
opts_sqp.dopost    = dopost;

% solveF parameters for jacobian inversion
Psol = struct();
Psol.algoGMRES   = L.P.algoGMRES;
Psol.mkryl       = L.P.mkryl;
Psol.epsilon_res = L.P.epsilon_res;
Psol.userowmask  = L.P.userowmask;

% dresdCo (needed to compute dxdCo)
if all(masku(L.ind.iuC))
  drC = zeros(L.nN,L.nC);
  drC(L.ind.irC,:) = diag(-1./L.xscal(L.ind.ixg));
  iusC = sum(masku(1:L.ind.iuC(1)-1)) + (1:L.nC); % us(iusC) = Co
elseif any(masku(L.ind.iuC))
  error('Either include all or none of the ag constraints in the optimization problem')
else
  drC = [];
  iusC = [];
end

% Init
nts = numel(LXs.t);
resFs = cell(nts,1);
LYs = resFs;
Opys = resFs;

LXts = resFs;
for its = 1:nts
  LXts{its} = meqxk(LXs,its);
end

% Tolerance and number of iterations
tol = LXs.tol(1);
if ~L.P.itert
  niter = LXs.niter(1);
else
  niter = L.P.itert;
end

kit = 0;
doSQP = false;
conv = false;
shot = LXs.shot(1);
t = LXs.t;

hasdz = strcmp(L.code,'liu') && L.ndz>0;
hasNL = strcmp(L.code,'liu');

if hasdz
  dz = zs{1};
else
  dz = 0;
end
dzg = 0;

% Evaluate residuals
[Ra,Re,Ri] = meqlsqeval(L,...
  Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c,...
  d1dt,d2dt,xs,us,zs);

alpha_step = 1-L.P.relax;
frozeOpy = false;

while isempty(sinit) % If initialization failed, go to post-processing

  doSQP = L.P.useSQP && kit >= L.P.wsSQP;
  
  % Deactivate dz stabilization for SQP
  if hasdz && doSQP
    % LIU single slice is assumed
    dz(:)   = 0;
    zs{1} = dz;
  end

  %% Evaluate GS residual and jacobians
  if doSQP, opts = opts_sqp;
  else,     opts = opts_pic;
  end
  [resFs,LYs,fgeF_failed,Opys,dx0s,dxdus,dxdxps,dxs_] = ...
    evaluate_fgeF(L,LXts,xs,us,zs,frozeOpy,hasdz,Opys,opts,xHasIy,doSQP,masku,...
    idts,picard_Precs,drC,iusC,TusU,Uscale,Psol,dxdus,dxdxps,dxs_);
  % Stop if one fgeF call failed
  if fgeF_failed
    meqmsge('w',mfilename,L.P.tokamak,t,kit,shot,...
      sprintf('fgeF call failed for t=%f',LXts{its}.t),...
      'fgeFfailed');
    break
  end

  %% Update NL residual values and jacobians
  if hasNL
    [Ra,Re,Ri,Raxc,Rexc,Rixc,Rauc,Reuc,Riuc,Razc,Rezc,Rizc] = ...
      update_NL_residuals(L,LXs,LYs,doSQP,hasdz,kit,...
      Ra,Re,Ri,Raxc,Rexc,Rixc,Rauc,Reuc,Riuc,Razc,Rezc,Rizc);
  end

  %% Update reduced optimization problem
  % Form reduced optimization problem
  % Identify which rows of dxdu are zeros (outside of plasma domain for
  % algoNL=all-nl)
  mask = false(L.nN,1);
  if ~L.isEvolutive && xHasIy
    mask(Opys{1}(:)==0) = true;
  end
  if ~doSQP && ~hasNL && init_response_matrix
    % Picard step, keep RaUt/ReUt/RiUt from previous iteration
    [Rat,   ~,Ret,   ~,Rit,   ~] = meqlsqprep(L,...
      Raxc,Rauc,Razc,Ra,Rexc,Reuc,Rezc,Re,Rixc,Riuc,Rizc,Ri,...
      TusU,TzsU,d1dt,d2dt,xs,us,zs,dx0s,dxdus,dxdxps,mask,true);
  else
    % Non-linear step
    [Rat,RaUt,Ret,ReUt,Rit,RiUt] = meqlsqprep(L,...
      Raxc,Rauc,Razc,Ra,Rexc,Reuc,Rezc,Re,Rixc,Riuc,Rizc,Ri,...
      TusU,TzsU,d1dt,d2dt,xs,us,zs,dx0s,dxdus,dxdxps,mask,false);
    init_response_matrix = true;
  end

  %% Debug
  meqoptdebug(dopli,dotxi,kit==0);

  %% Solve reduced optimization problem
  % Solve for u/Fb update, compute x update
  [dxs,dus,dzs,~,~,s] = meqlsqsolve(L,qpsolve,qpopts,...
    Rat,RaUt,Ret,ReUt,Rit,RiUt,...
    dx0s,dxdus,dxdxps,TusU,TzsU,...
    Uscale,false,ctol);

  if ~isempty(s)
    meqmsge(msgtype,'meqlsqsolve',L.P.tokamak,t,kit,shot,s.msg,s.msgid); break;
  end

  %% Update state and inputs
  for its = 1:nts
    xs{its} = xs{its} + alpha_step*dxs{its};
    us{its} = us{its} + alpha_step*dus{its};
    zs{its} = zs{its} + alpha_step*dzs{its};
  end

  if hasdz
    % LIU single-slice is assumed
    dzg = zs{1};
    dz = zfct(dzg);
  end

  %% Update iteration counter
  kit = kit+1;

  %% Evaluate residuals
  % Compute fgeF residuals
  %   This call should be equivalent to the following iteration, so we do
  %   not update Opys (or xs,us,zs)
  [resFs,LYs,fgeF_failed] = ...
    evaluate_fgeF(L,LXts,xs,us,zs,frozeOpy,hasdz,Opys,optsF('dopost',dopost),xHasIy,doSQP,masku);
  % Stop if one fgeF call failed
  if fgeF_failed
    meqmsge('w',mfilename,L.P.tokamak,t,kit,shot,...
      sprintf('fgeF call failed for t=%f',LXts{its}.t),...
      'fgeFfailed');
    break
  end

  % Compute residuals
  [Ra,Re,Ri] = meqlsqeval(L,...
    Raxc,Rauc,Razc,Ra0c,Rexc,Reuc,Rezc,Re0c,Rixc,Riuc,Rizc,Ri0c,...
    d1dt,d2dt,xs,us,zs);

  % Update NL residuals
  if hasNL
    [Ra,Re,Ri] = ...
      update_NL_residuals(L,LXs,LYs,doSQP,hasdz,kit,...
      Ra,Re,Ri);
  end

  % Prepare some for evaluation of convergence
  Re_      = vertcat(Re  {:});
  resmaxF  = max(abs([resFs{:}]));                       % Maximum FGE residual
  resmaxRe = max(abs(Re_));                              % Maximum equality constraint residual
  dxmax = max(abs(vertcat(dxs{:})));
  dumax = max(abs(vertcat(dus{:}))./(vertcat(TusU{:})*Uscale));

  %% Convergence check
  if L.P.itert
    if kit >= niter, conv = true; break; end
  elseif ~L.P.useSQP || doSQP
    % If all residuals are lower than tolerance, stop.
    %   NB: Do not check first order optimality condition of reduced
    %   optimization problem since scaling is quite different
    if max([resmaxF,resmaxRe]) < tol, conv = true; break; end
    % When using Picard iterations, only check FGE residual
    if ~L.P.useSQP && resmaxF < tol, conv = true; break; end

    % Also stop if change in state/inputs is small enough
    if max([dxmax,dumax]) < L.P.tolstep, conv = true; break; end
    % Maximum number of iterations exceeded
    if kit >= niter
      meqmsge('w',mfilename,L.P.tokamak,t,kit,shot,'iterations exceeded','nIterMax');
      break
    elseif kit == L.P.iterfrz
      % Freeze plasma domain to ease convergence
      meqmsge('w',mfilename,L.P.tokamak,t,kit,shot,sprintf('No convergence after %d iterations - freezing plasma domain',L.P.iterfrz),'fixOpy');
      % LY always contains Opy for fbt/liu even if dopost is false
      frozeOpy = true;
    end
  end
end % while 1 (iterations over kit=[1:niter])

% Debug final iteration
if isempty(sinit)
  meqoptdebug(dopli,dotxi);
end

if ~isempty(sinit)
  meqmsge(msgtype,[L.code,'t'],L.P.tokamak,t,kit,shot,sinit.msg,sinit.msgid);
end

%% Post-processing

% indices of z variables that don't appear in the optimization problem
if L.isEvolutive
  for its = 1:nts
    mask = ~any([Razc{its};Rezc{its}],1);
    zs{its}(mask) = NaN; % Mark these as NaN explicitly
  end
end

% Recompute post-processing if needed
if ~dopost || ~isempty(sinit) || frozeOpy
  % Compute post-processing without frozen Opy
  [resFs,LYs] = ...
    evaluate_fgeF(L,LXts,xs,us,zs,false,hasdz,Opys,optsF('dopost',true),xHasIy,doSQP,masku);
end
  
for its = 1:nts
  res = resFs{its};
  LYt = LYs{its};

  % Opt specific
  LYt.isconverged = conv;
  LYt.niter = kit;
  LYt.res = res;
  LYt.ID   = vertcat(IDs{:,its});
  Ra_  = vertcat(Ra{:,its});
  ws_  = vertcat(ws{:,its});
  Re_  = vertcat(Re{:,its});
  LYt.werr = -Ra_;                                        % weighted cost function error
  LYt.uerr = -Ra_./ws_;                                   % unweighted cost function error
  LYt.chi  = sqrt(sum(Ra_.^2)/numel(Ra_));                % From approximate FBT constraints
  LYt.chie = sqrt(sum(Re_.^2)/numel(Re_));                % From exact FBT constraints

  % Additional output arguments
  for k = 1:numel(L.argoutc)
    if isempty(L.argoutc{k}.file)
      LYt.(L.argoutc{k}.fld) = eval(L.argoutc{k}.exp);
    end
  end

  LYs{its} = LYt;
end

%% Debugging each time slice or interval
meqoptdebug(doplt && ~dopli, dotxt && ~dotxi , true);

end

%% Evaluate fgeF compute dx0/dxdu/dxdxp
function [resFs,LYs,fgeF_failed,Opys,dx0s,dxdus,dxdxps,dxs_] = ...
  evaluate_fgeF(L,LXts,xs,us,zs,frozeOpy,hasdz,Opys,opts,xHasIy,doSQP,masku,...
  idts,picard_Precs,drC,iusC,TusU,Uscale,Psol,dxdus,dxdxps,dxs_)

fgeF_failed = false; % Indicator for fgeF failure

nts = numel(LXts);
resFs = cell(1,nts);
LYs = resFs;
dx0s = resFs;
aux = cell(1,6);

LYt = []; % init
xp = []; tp = LXts{1}.t;
for its = 1:nts
  x = xs{its};
  u = us{its};
  LXt = LXts{its};
  % Set up L, LXt, LYp
  [L,LXt,LYp] = prepare_fgeF_call(L,u,masku,xp,tp,LXt,LYt,xHasIy,opts.dopost);

  % Set Opy if needed
  if frozeOpy
    aux{3} = Opys{its};
  end

  % Set dz if needed
  if hasdz
    aux{4} = zs{its};
    aux{5} = Opys{its};
  end
  aux{6} = doSQP;

  % Evaluate fgeF
  [resF,LYt,Jx,Ju,Jxdot,Jxdiag] = fgeF(x,L,LXt,LYp,opts,aux);
  mask = logical(Jxdiag);
  % Prepare next slice
  xp = x; tp = LXt.t;
  % Store residual LY and Opy in cell arrays
  resFs{its} = norm(resF);
  LYs{its}   = LYt;
  Opys{its}  = LYt.Opy;
  
  % Handle failures (always process all slices anyway)
  if any(isnan(resF)), fgeF_failed=true; continue; end

  % If jacobians aren't needed skip the rest
  if ~(opts.dojacF || opts.dojacx)
    continue
  end

  % Jacobian part
  if ~doSQP || L.P.usepreconditioner
    if isa(Jx,'function_handle')
      % Evaluate jacobian in ag directions
      dx = zeros(L.nN,L.ng);
      dx(L.ind.ixg+(0:L.ng-1)*L.nN) = 1;
      Jxg = Jx(dx);
    else
      Jxg = Jx(:,L.ind.ixg);
    end
    % Compute Picard 'preconditioner' - inverse of approximate Jacobian
    Pinv = picard_Precs{its}(Jxg);
  else
    Pinv = [];
  end
  %
  if ~doSQP
    % Compute Picard state update step
    dx0s{its} = -Pinv(resF);
    if any(masku(L.ind.iuC))
      % Compute dxdCo for Co=ag
      dxdus{its}(:,iusC) = -Pinv(drC);
    end
  else
    Ju_ = Ju(:,masku);
    if L.isEvolutive && idts(its)
      % For Ju(Va),Jxdot only rows corresponding to circuit residuals are
      % non-zero, so solve only a reduced number of systems
      M = [zeros(L.nN-L.ne,L.ne);eye(L.ne)];
      dx_ = solve_linear_problem(Jx,-[resF,M   ],dxs_{its},Pinv,mask,Jxdiag,Psol);
      dx0s{its}   = dx_(:,1);
      dxdus{its}  = dx_(:,2:end)*Ju_(L.ind.irD,:);
      dxdxps{its} = dx_(:,2:end)*(Jxdot*(-idts(its)));
    else
      uscale = TusU{its}*Uscale;
      Ju_ = Ju_.*uscale(masku).';
      dx_ = solve_linear_problem(Jx,-[resF,Ju_],dxs_{its},Pinv,mask,Jxdiag,Psol);
      dx0s{its}   = dx_(:,1);
      dxdus{its}  = dx_(:,2:end)./uscale(masku).';
    end
    % Store solution for use as initial guess in next iteration
    dxs_{its} = dx_;
  end
end
end

%% Update LXt/LYp for fgeF call
function [L,LXt,LYp] = prepare_fgeF_call(L,u,masku,xp,tp,LXt,LYt,xHasIy,useLY)
% This function sets up the fgeF call for a given time slice by updating
% LX, LYp and setting time-dependent options in L

if isempty(xp)
  % First slice is independent of LYp
  LYp = struct('t',tp,'aq',[],'aW',[]);
  % Add required fields for fgeF
  LYp.Ia = zeros(L.G.na,1);
  LYp.Iu = zeros(L.G.nu,1);
  LYp.Iy = zeros(L.nzy,L.nry);
elseif useLY
  % Use previous LY from debug output
  LYp = LYt;
else
  % Construct LYp from previous state
  LYp = struct('t',tp,'aq',[],'aW',[]);
  % Add required fields for fgeF
  xSIp = xp.*L.xscal;
  LYp.Ia = xSIp(L.ind.ixa);
  LYp.Iu = xSIp(L.ind.ixu);
  if xHasIy
    LYp.Iy = reshape(xSIp(L.ind.ixGS),L.nzy,L.nry);
  else
    Fx = reshape(xSIp(L.ind.ixGS),L.nzx,L.nrx);
    LYp.Iy = -(L.dlst*Fx(:))./L.rhsf - L.Tye*[LYp.Ia;LYp.Iu];
  end
end

% Modify inputs if needed
u_ = zeros(L.nuN,1);
u_(masku) = u;
if L.isEvolutive % u = [Va;Co;InID]
  % Adjust voltages
  ua = u_(L.ind.iua);
  LXt.Va(masku(L.ind.iua)) = ua(masku(L.ind.iua));
  % Adjust Ini
  uni = u_(L.ind.iua);
  LXt.IniD(masku(L.ind.iuni)) = uni(masku(L.ind.iuni));
else % u = [Ia;Iu;Co]
  % Adjust coil currents
  ua = u_(L.ind.iua);
  LXt.Ia(masku(L.ind.iua)) = ua(masku(L.ind.iua));
  % Adjust vessel currents
  uu = u_(L.ind.iuu);
  LXt.Iu(masku(L.ind.iuu)) = uu(masku(L.ind.iuu));
end
% Adjust constraints
Co = LX2Co(L,LXt);
uC = u_(L.ind.iuC);
Co(masku(L.ind.iuC)) = uC(masku(L.ind.iuC));
LXt = Co2LX(L,LXt,Co);

% Time-varying parameters
%   bfp
isbffbt = strcmp(func2str(L.P.bfct),'bffbt');
if isbffbt
  L.bfp = LXt.bfp;
end
%   isaddl
L.P.isaddl = LXt.isaddl;
%   idoublet
L.P.idoublet = LXt.idoublet;
end

%% Update NL residuals and their jacobians
function [Ra,Re,Ri,Raxc,Rexc,Rixc,Rauc,Reuc,Riuc,Razc,Rezc,Rizc] = ...
  update_NL_residuals(L,LXs,LYs,doSQP,hasdz,kit,...
  Ra,Re,Ri,Raxc,Rexc,Rixc,Rauc,Reuc,Riuc,Razc,Rezc,Rizc)

dojac = nargout > 3;

% LIU single-slice is assumed

% DML measurement
if L.P.idml
  Wt = L.Wd(L.kdt)*LXs.rBt;
  kat = L.kdt;
  % Compute Ra
  Ra{1}(kat) = Wt.*(LXs.Xt-LYs{1}.Xt);
  if dojac
    % Jacobian
    %   Apply weight
    Raxc{1}(kat,:) = (Wt.*LYs{1}.dXtdx);
    Rauc{1}(kat,:) = (Wt.*LYs{1}.dXtdu);
  end
end
% Regularisation constraints
if any(L.P.wreg) && L.nq
  kaq = L.nd+(1:L.nq);
  Wq = L.Wq;
  % Adapt regularisation with elongation - domain specific
  if L.nD == 1 % Deactivate adaptive regularization for multiple domains (temp.)
    b0 = bboxmex(LYs{1}.Opy>0,L.zy,L.ry);
    ie0 = (b0(4) - b0(2)) / (b0(3) - b0(1));
    Wq = Wq.*min(exp(-(1./reshape(repmat(ie0, 1, L.nq/L.nD)',L.nq,1) -L.P.elomin).*L.P.wregadapt),1);
  end
  % Compute Ra
  Ra{1}(kaq) = Wq.*(-LYs{1}.Xq);
  if dojac
    % Jacobian
    %   Apply weight
    Raxc{1}(kaq,:) = (Wq.*LYs{1}.dXqdx);
    Rauc{1}(kaq,:) = (Wq.*LYs{1}.dXqdu);
  end
end

% Inequality constraints
if L.P.ipm && L.nc
  % Compute Ri
  Ri{1} = -LYs{1}.Xc;
  if dojac
    % Jacobian
    Rixc{1} = LYs{1}.dXcdx;
    Riuc{1} = LYs{1}.dXcdu;
  end
end

% Vertical stabilization parameter
if hasdz
  % Compute Ra
  kar = L.kdr;
  if ~doSQP
    Ra{1}(kar) = L.Wr.*([LXs.Ff;LXs.Bm]-LYs{1}.Xr);
  end
  if dojac
    if ~doSQP && (L.nD == 1 || kit>0)
      dXrddz = LYs{1}.dXrddz;
    else
      dXrddz = zeros(L.nr,L.ndz);
    end
    Razc{1}(kar,:) = L.Wr.*dXrddz;
  end
end
end
