function fname_out = meqmovie(L,LY,varargin)
%MEQMOVIE create matlab movie based on meq LY structure
%
% meqmovie(L,LY,'par','val')
%
%   example: [L,LX,LY] = liuqe(61400,[0.1:1e-1:2]);
%            moviefile = meqmovie(L,LY);
%            moviefile = meqmovie(L,meqxk(LY,1:2:end),'style','fancy','dosave',true,'filename','mymovie')
%
% parameters:
%    'style': { 'plain','fancy','full'}
%           plain: only flux on computational grid
%           fancy: add coils and flux outside grid
%           full:  fancy + evolving time traces
%           custom: use custom function via plotfun (see below)
%    'allslices': { true,false } plot all slices or only converged ones
%    'extraquantities': cell array with extra quantities to plot. Supported: {'Fx','Bpx'}
%    'dosave': { true,false} % save movie file or not
%    'framerate': movie framerate. Default=24
%    'titlefun': custom title function handle, signature str = titlefun(L,LY);
%    'videoprofile': Video profile (see help writeVideo, or choose 'gif' for animated gif
%    'filename': string of target movie file (default: something based on shot/time)
%    'plotfun' : function handle of custom plotting function, 
%                    signature matching either hax=plotfun(hax,L,LY) or hax=plotfun(hax,L,LX,LY)
%                    with hax being axis handles for possible re-use of
%                    existing objects.
%    'LX'   : optional LX structure for custom plots with signature plotfun(ax,L,LX,LY)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if isempty(LY), warning('can not plot empty LY'); return; end
if L.P.izgrid
  defaultstyle = 'fancy';
else
  defaultstyle = 'plain';
end

% multiple LY handling
if ~iscell(LY)
  LYs = {LY}; % single-LY case
else
  LYs = LY; % multi-LY case
  LY = LYs{1}; % first one
end
nLYs = numel(LYs);

if isnumeric(LY.shot)
  shot= LY.shot(1);
else
  shot = [];
end
defaultfname = sprintf('%s_%d_%02.4f',L.P.tokamak,shot,LY.t(1));

p=inputParser;
p.addParameter('style',defaultstyle,@(x) ischar(x));
p.addParameter('allslices',true,@(x) islogical(x));
p.addParameter('dosave',false,@(x) islogical(x));
p.addParameter('filename',defaultfname,@(x) ischar(x)); 
p.addParameter('decimate',1,@(x) isscalar(x) && isnumeric(x)); % decimate every n frames
p.addParameter('extraquantities',{})
p.addParameter('force',false); % force ploting even for long movies
p.addParameter('framerate',24);
p.addParameter('videoprofile','Motion JPEG AVI');
p.addParameter('LX',struct.empty,@(x) isstruct(x));
p.addParameter('titlefun', function_handle.empty,...
  @(x) isa(x,'function_handle') || (isa(x,'cell') && numel(x)==nLYs))
p.addParameter('plotfun', function_handle.empty,...
  @(x) isa(x,'function_handle'))

p.KeepUnmatched = true;

parse(p,varargin{:}); P = p.Results; LX = P.LX;

assert(~isequal(P.style,'full') || nLYs==1,...
  'style=''full'' not supported for multiple LYs');

if strcmp(P.style,'custom')
  assert(~isempty(P.plotfun),'when using ''custom'' style, plotfun must not be empty')
else
  assert(isempty(P.plotfun),'use ''custom'' style to enable plotfun');
end
assert(isstruct(L),'L must be a structure');

% check consistency of all LY structures
tt = LYs{1}.t;
for iLY = 1:nLYs
  assert(isstruct(LYs{iLY}),'all LYs must be structures');
  assert(all(tt==LYs{iLY}.t),'all LYs must have the same time grid');
end
assert(isempty(LX) || numel(LX.t) == numel(LY.t),'number of LX time slices must match LY')

% decimate
nt = numel(tt);
if nt>240 && ~P.force
  fprintf('Warning! Asking more than %d frames (%f second movie), this may take time... are you sure?',nt,nt/P.framerate)
  if ~contains(lower(input('Y/[N]','s')),'y')
    return
  end
end

if P.dosave
  if strcmp(P.videoprofile,'gif')
    P.filename = [P.filename,'.gif'];
  else
    % prepare writerObj unless writing gif directly
    [writerObj,moviefile] = init_recording(P.filename,P.framerate,P.videoprofile);
    if isempty(writerObj)
      fprintf('could not init movie writing, exiting\n')
      return
    else
      fprintf('writing movie file: %s\n',moviefile);
    end
  end
else
  moviefile = '';
end

slices_to_plot = 1:P.decimate:nt;
n_slices = numel(slices_to_plot);

for it=1:n_slices
  for iLY = 1:nLYs
    if nLYs == 1
      h = gca;
    else
      h = subplot(1,nLYs,iLY);
    end
    LYt = meqxk(LYs{iLY},slices_to_plot(it));

    if ~P.allslices && ~LYt.isconverged, continue; end
    
    fprintf('plotting t=%f\n',LYt.t);
    switch P.style
      case 'fancy'
        h = meqplotfancy(L,LYt,'parent',h,...
          p.Unmatched);
        h.Position = [0.08 0.03 0.92 0.90];
      case 'plain'
        h = meqplott(L,LYt,'parent',h);
      case 'full'
        if it==1, hax=[]; end
        hax = meqplotfull(L,LYt,'hax',hax);
        h=hax(1);
      case 'custom'
        hax = h;
        if isempty(LX)
          hax = P.plotfun(hax,L,LYt);
        else
          LXt = meqxk(LX,it); % slice
          hax = P.plotfun(hax,L,LXt,LYt); % call with LX
        end
        h=hax(1);
      otherwise
        error('unknown style %s',P.style)
    end
    
    if ~isempty(P.extraquantities)
      if any(contains(P.extraquantities,'Bpx'))
        if exist('hBpx','var') && any(ishandle(hBpx)), delete(hBpx); end
        [~,hBpx1] = contour(h,L.G.rx,L.G.zx,LYt.Bpx,0:1e-3:0.01,'b-');
        [~,hBpx2] = contour(h,L.G.rx,L.G.zx,LYt.Bpx,0:1e-2:0.1 ,'b--');
        hBpx = [hBpx1;hBpx2];
      end
      if any(contains(P.extraquantities,'Fx'))
        if exist('hFx','var') && any(ishandle(hFx)), delete(hFx); end
        if exist('hFX','var') && any(ishandle(hFX)), delete(hFX); end
        [~,hFx] = contour(h,L.G.rx,L.G.zx,LYt.Fx,-2:1e-2:2,'k--');
        FX = LYt.FX; FX = FX(~isnan(FX)); if numel(FX)==1, FX=FX*[1 1]; end
        [~,hFX] = contour(h,L.G.rx,L.G.zx,LYt.Fx,FX,'k:');
      end
    end
    
    %% Text
    if ~isempty(P.titlefun)
      if iscell(P.titlefun)
        titfun = P.titlefun{iLY};
      else
        titfun = P.titlefun;
      end
      titstr = titfun(L,LYt); % evaluate user-specified function
      title(h,titstr)
    elseif ~strcmp(P.style,'custom') % assume custom has own title setting ability
      if isfield(LYt,'Vloop')
        titstr = sprintf('t=%06.4f,Vloop=%4.2fV',LYt.t,LYt.Vloop);
      elseif isfield(LY,'Ip')
        titstr = sprintf('%s%d,t=%06.4f,Ip=%3.1fkA',L.P.tokamak,shot,LYt.t,LYt.Ip/1e3);
      end
      title(h,titstr)
    end
    %%
  end
  
  if it==1
    axis(h,'off');
    set(gcf,'Color','white')
    set(gcf,'Position',[100,100,400,700])
  end
  
  drawnow; pause(0.01);
  
  if P.dosave
    fprintf('Writing movie frame: %02d/%02d\n',it,nt)
    frame = getframe(gcf);
    
    if strcmp(P.videoprofile,'gif')
      writeGif(P.filename,frame,it,P.framerate)
    else
      writeVideo(writerObj,frame);
    end
  end
end

if P.dosave
  if strcmp(P.videoprofile,'gif')
    fprintf('wrote animated gif to %s\n',P.filename)
    fname_out = P.filename;
  else
    close(writerObj);
    fprintf('Finished writing movie to: %s\n',moviefile);
    fname_out = moviefile;
  end
else
  fname_out = '';
end
end

function writeGif(filename,frame,kt,fps)
im = frame2im(frame);
[imind,cm] = rgb2ind(im,256);
% Write to the GIF File
if kt == 1
  imwrite(imind,cm,filename,'gif', 'Loopcount',inf);
else
  imwrite(imind,cm,filename,'gif','WriteMode','append','DelayTime',1/fps);
end
end

function [writerObj,moviefile] = init_recording(moviefile,framerate,video_profile)
% initialize recording

if isempty(moviefile)
  [filename,pathname] = uiputfile(moviepath);
else
  filename = moviefile;
  pathname = pwd;
end

% if user aborts here, or other problem, return empty and abort recording
if ~ischar(filename)
  writerObj = [];
  moviefile = '';
  return
else
  % prepare movie object
  moviefile = fullfile(pathname,[filename,'.avi']);
  writerObj = VideoWriter(moviefile,video_profile);
  set(writerObj,'FrameRate',framerate);
  open(writerObj);
end

end
