function [dxs,dus,dzs,lambda,c_ok,s] = meqlsqsolve(L,qpsolve,qpopts,...
  Rat,RaUt,Ret,ReUt,Rit,RiUt,...
  dx0s,dxdus,dxdxps,TusU,TzsU,...
  Uscale,no_ineq,ctol)
% MEQLSQSOLVE solves the reduced least-squares problem for MEQOPT
%
% A detailed description of the inputs and outputs is available in MEQOPT.
%
% This function assembles the linear least squares problem, solves it using
% MEQLSQLIN and processes the output to produce the updates for all states
% (dxs), inputs (dus) and optimization variables (dzs).
%
% SEE ALSO: MEQOPT, MEQLSQPREP, MEQLSQLIN, FBTT, FBTOPT
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% sizes
nt = size(Rat,2);

%% Assemble LSQ components
%   RHS
Rat_ = vertcat(Rat{:});
Ret_ = vertcat(Ret{:});
Rit_ = vertcat(Rit{:});
%   Response matrices
RaUt_ = vertcat(RaUt{:});
ReUt_ = vertcat(ReUt{:});
RiUt_ = vertcat(RiUt{:});

%% convert inequality constraints involving only one variable into bounds
[RiUt_,Rit_,lb,ub] = ineq_to_bounds(RiUt_,Rit_);

%% SOLVE LSQ PROBLEM
dU0 = []; % No initial guess
[dU,lambda,c_ok,s] = meqlsqlin(qpsolve,qpopts,RaUt_,Rat_,RiUt_,Rit_,ReUt_,Ret_,lb,ub,Uscale,dU0,ctol,no_ineq);

%% Post-processing
dxs = cell(nt,1); dus = dxs; dzs = dxs;
dxp = zeros(L.nN,1);
for it = 1:nt
  dus{it} = TusU{it}*dU;
  dzs{it} = TzsU{it}*dU;
  if it == 1
    dxp = dx0s{it} + dxdus{it}*dus{it};
  else
    dxp = dx0s{it} + dxdus{it}*dus{it} + dxdxps{it}*dxp;
  end
  dxs{it} = dxp;
end

end

function [Aiq,biq,lb,ub] = ineq_to_bounds(Aiq,biq)
% given inequality constraints Aiq * x < biq
% convert constraints involving only one variable to bounds.
% Returns reduced Aiq,biq where these bounds have been
% discarded.

nx = size(Aiq,2);                 % size of x (unknowns)
ub = Inf(nx,1); lb = -ub;         % init ub,lb
l = (sum(Aiq ~= 0,2) == 1);       % limit involves just one x value
Cl = Aiq(l,:); bl = biq(l);       % take these Cl*x < bl
cscal = sum(Cl,2);                % cscal.*x < b
iub = (cscal>0) ; ilb = ~iub;     % lower or upper bound
[j,~] = find(Cl');                % corresponding x indices (row-wise)
ub(j(iub)) = bl(iub)./cscal(iub); % x < b/cscal for upper bound
lb(j(ilb)) = bl(ilb)./cscal(ilb); % x > b/cscal for lower bound

Aiq = Aiq(~l,:); biq = biq(~l);   % eliminate these from inequality constraints
end
