function [Rat,RaUt,Ret,ReUt,Rit,RiUt] = meqlsqprep(L,...
  Raxc,Rauc,Razc,Ra,Rexc,Reuc,Rezc,Re,Rixc,Riuc,Rizc,Ri,...
  TusU,TzsU,d1dt,d2dt,xs,us,zs,dx0s,dxdus,dxdxps,mask,skip_response_matrix)
% MEQLSQPREP constructs the reduced least-squares problem for MEQOPT
%
% A detailed description of the inputs and outputs is available in MEQOPT.
%
% This function computes the reduced residuals Rat/Ret/Rit and reduced
% residual jacobians RaUt/ReUt/RiUt such that:
%   Ra = Ra(xs,us,zs)
%   Rat = Ra + Rax*dx0
%   RaUt = (Rax*dxdu + Rau)*TusU + Raz*TzsU
% This is part of the reduced hessian SQP algorithm from MEQOPT.
%
% Eventually all the different matrices for each time are combined
% in MEQLSQSOLVE to yield a single constrained least-squares problem for U:
%
%  min_U |RaUt*U - Rat|_2^2
%   ReUt*U = Ret
%   RiUt*U < Rit
%
% Time derivatives in the optimization problem:
%   For any function F(t), we use the approximations:
%     dFdt  (t(i)) = sum_j d1dt(i,j)*F(t(j))
%     d2Fdt2(t(i)) = sum_j d2dt(i,j)*F(t(j))
%   For cost function terms or constraints of the form
%     R(t(i)) = R0 + Rx*dxdt  (t(i)) + Ru*dudt  (t(i))
%   we substitute dxdt, dudt with the approximations above to get the
%   reduced optimization problem formulation
%     R(t(i)) = Rt + RtU*U
%   with Rt  = R0 + Rx*(sum_j d1dt(i,j)*(x(t(j))+dx0(t(j)))) + Ru*(sum_j d1dt(i,j)*u(t(j)))
%        RtU =      Rx*(sum_j d1dt(i,j)*dxdU(t(j)))          + Ru*(sum_j d1dt(i,j)*dudU(t(j)))
%   And similarly for second order time derivatives.
%
% SEE ALSO: FBTHELP, FBTLSQINIT, MEQLSQSOLVE, FBTT, MEQOPT
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% sizes
[nd,nt] = size(Raxc);
nU = size(TusU{1},2);
nx = L.nN;

if nd>3
  error('Only up to 2nd order derivative constraints implemented')
end

% Init
Rat  = cell(nd,nt); Ret  = Ra; Rit  = Ra;
RaUt = Ra;          ReUt = Ra; RiUt = Ra;

dx0 = Ra; dxdU = Ra;

%% Compute right-hand sides and response matrices
% - Static case
if ~L.isEvolutive
  % Reduced residual
  Rat{1} = Ra{1} - Raxc{1}*dx0s{1};
  Ret{1} = Re{1} - Rexc{1}*dx0s{1};
  Rit{1} = Ri{1} - Rixc{1}*dx0s{1};
  
  % Allow skipping computation of RaUt, ReUt, RiUt
  if ~skip_response_matrix
    % Reduced response matrices
    Rax  =  Raxc{1};
    Rex  =  Rexc{1};
    Rix  =  Rixc{1};
    dxdu = dxdus{1};
    if L.P.userowmask && any(mask)
      Rax  =  Rax(:,~mask);
      Rex  =  Rex(:,~mask);
      Rix  =  Rix(:,~mask);
      dxdu = dxdu(~mask,:);
    end
    RaUt{1} = (Rauc{1} + Rax*dxdu)*TusU{1} + Razc{1}*TzsU{1};
    ReUt{1} = (Reuc{1} + Rex*dxdu)*TusU{1} + Rezc{1}*TzsU{1};
    RiUt{1} = (Riuc{1} + Rix*dxdu)*TusU{1} + Rizc{1}*TzsU{1};
  end
  return
end

% - Evolutive case
% We first need to compute dx0 and dxdU for all time slices
for it = 1:nt
  % State update dx0
  if it == 1
    dx0{it}  = dx0s{it};
  else
    dx0{it}  = dx0s{it} + dxdxps{it}*dx0{it-1};
  end
  
  % Allow skipping computation of RaUt, ReUt, RiUt
  if skip_response_matrix, continue; end
  
  % variation of xs{it} with respect to U dxdU
  if it == 1
    dxdU{it} = dxdus{it}*TusU{it};
  else
    dxdU{it} = dxdus{it}*TusU{it} + dxdxps{it}*dxdU{it-1};
  end
end

% Now assemble all including time derivatives
for it = 1:nt
  nu = numel(us{it});
  nz = numel(zs{it});
  for id = 1:nd
    if id == 1
      %   Value constraints
      dx0_ = dx0{it};
      % Allow skipping computation of RaUt, ReUt, RiUt
      if ~skip_response_matrix
        dudU_ = TusU{it};
        dzdU_ = TzsU{it};
        dxdU_ = dxdU{it};
      end
    else
      %   1st/2nd order time derivative constraints
      x_   = zeros(nx,1);
      u_   = zeros(nu,1);
      z_   = zeros(nz,1);
      dx0_ = zeros(nx,1);
      % Allow skipping computation of RaUt, ReUt, RiUt
      if ~skip_response_matrix
        dudU_ = zeros(nu,nU);
        dzdU_ = zeros(nz,nU);
        dxdU_ = zeros(nx,nU);
      end
      % Loop over all slices that influence the time derivative at slice it
      for itt = 1:nt
        if id == 2
          fac = d1dt(it,itt);
        else
          fac = d2dt(it,itt);
        end
        if fac == 0, continue; end % Skip points not in the stencil
        x_   = x_   + fac*xs {itt};
        u_   = u_   + fac*us {itt};
        z_   = z_   + fac*zs {itt};
        dx0_ = dx0_ + fac*dx0{itt};
        % Allow skipping computation of RaUt, ReUt, RiUt
        if ~skip_response_matrix
          dudU_ = dudU_ + fac*TusU{itt};
          dzdU_ = dzdU_ + fac*TzsU{itt};
          dxdU_ = dxdU_ + fac*dxdU{itt};
        end
      end
    end

    % Reduced residual
    Rat{id,it} = Ra{id,it} - Raxc{id,it}*dx0_;
    Ret{id,it} = Re{id,it} - Rexc{id,it}*dx0_;
    Rit{id,it} = Ri{id,it} - Rixc{id,it}*dx0_;

    % Allow skipping computation of RaUt, ReUt, RiUt
    if ~skip_response_matrix
      % Reduced response matrices
      RaUt{id,it} = Rauc{id,it}*dudU_ + Razc{id,it}*dzdU_ + Raxc{id,it}*dxdU_;
      ReUt{id,it} = Reuc{id,it}*dudU_ + Rezc{id,it}*dzdU_ + Rexc{id,it}*dxdU_;
      RiUt{id,it} = Riuc{id,it}*dudU_ + Rizc{id,it}*dzdU_ + Rixc{id,it}*dxdU_;
    end
  end
end

end
