function Iy = meqinitIy(L,t,IpD,bbs,capaj)
% MEQINITIY Plasma current distribution initial guess
%
% Iy = meqinitIy(L,t,IpD,bbs,capaj)
%
% Initializes parabolic Iy using bounding box info bb
% and specified current profile peakedness.
% 
% Inputs:
%   L: L structure
%   t: vector of times [1,nt]
%   IpD: Plasma current per domain [nD,nt]
%   bbs: Cell array with nt cells of bounding boxes {[nD,4]}
%   capaj: parameter for current profile peakedness per time (j=(1-rho.^2).^capaj)
%
% Outputs: 
%   IyD: array with initialized plasma current array [nzy,nry,nt]
%
% NB: Iy for time points where |Ip|>0 and bbs contains NaNs are generated 
%     by linearly interpolating neighbouring time points.
%     This requires the first and last time points to be valid.
%
% SEE ALSO: BBOXMEX
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

nt = numel(t);
Iy = zeros(L.nzy,L.nry,nt);
iok = false(size(t)); % flag whether Iy was initialized at this step

for it = 1:numel(iok)
  if all(IpD(it)==0), iok(it) = true; continue; end % plasmaless slice
  bb = bbs{1,it}; % this bounding box
  if ~any(isnan(bb(:)))
    % these can be initialized from bounding boxes
    rh = 0.5*(bb(:,1)+bb(:,2)); zh = 0.5*(bb(:,3)+bb(:,4));
    wh = 0.5*(bb(:,2)-bb(:,1)); hh = 0.5*(bb(:,4)-bb(:,3));

    for iD=1:numel(rh) % loop over domains
      IyD = max(1-((L.rry-rh(iD))/wh(iD)).^2-((L.zzy-zh(iD))/hh(iD)).^2,0).^capaj(it);
      IyD = IpD(iD,it)/sum(IyD(:))*IyD;
      Iy(:,:,it) = Iy(:,:,it) + IyD;
    end
    iok(it) = true;
  end
end

% any missing Iy are interpolated from initialized ones
if any(~iok([1,end]))
  error('first and last Iy must be initialized to perform interpolation');
end
p = [3,1,2]; % Set time as first dimension for interp1
if ~all(iok)
  Iy = ipermute(interp1(t(iok)',permute(Iy(:,:,iok),p),t','linear'),p);
end
end
