classdef meqgui < handle
  % MEQGUI simple MEQ GUI, customizable for specific applications
  %
  % Usage example: [L,LX,LY] = fbt('ana',101);
  % meqgui(L,LY);
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    timetraces meqplottraces % time traces aggregating object
    eqslider meqplotslider % equilibrium plots with and slider object
  end
    
  properties (Hidden)
    parent % parent figure object
    HLUpdateTime event.listener % listener object
    HLUpdateEq event.listener % listener object
  end

  methods
    function obj = meqgui(L,LY,varargin)
      % parse inputs and assign to properties
      for ii=1:2:numel(varargin)
        mypar = varargin{ii};
        myval = varargin{ii+1};
        if isprop(obj,mypar)
          obj.(mypar) = myval;
        else
          error('no property corresponding to input %s',mypar)
        end
      end

      if isempty(obj.parent) || ~isgraphics(obj.parent)
        obj.parent = gcf(); % use gcf as parent if not given
      end
      
      % generate equilibrium slider objects
      obj.eqslider = meqplotslider(L,LY,'parent',obj.parent);
      
      % set object positions
      obj.eqslider.ax.Position     = [0.05 0.2 0.3 0.6];
      obj.eqslider.slider.Position = [0.05 0.10 0.3 0.05];
      obj.eqslider.text.Position   = [0.05 0.05 0.3 0.03];
      obj.eqslider.btn.Position    = [0.25 0.05 0.03 0.03];

      % make time trace axis layout
      nr = 9; nc = 2;
      ci = 0.45; co = 0.08; % column inner/outer gap
      ru = 0.05; rl = 0.05; % row upper/lower gap
      fr = 0.01;
      fc = 0.08; % gap

      % Call static method to generate axes
      hax = meqplottraces.create_axes(obj.parent,...
        nr,nc,ru,rl,ci,co,fr,fc);

      % Construct time trace object using these axes
      obj.timetraces = meqplottraces(L,LY,'hax',hax,...
        'traceinfo',{'Ip','Ia','Wk','Va'});

      % add listeners
      % update time trace display when changing eq. slider
      obj.HLUpdateTime = addlistener(obj.eqslider,'updated_time',...
        @(src,evnt)listenUpdateTimeTraces(obj,src,evnt));

      % update eq. display when selecting a time from the time traces
      obj.HLUpdateEq = addlistener(obj.timetraces,'select_time_slice',...
        @(src,evnt)listenUpdateEqSlice(obj,src,evnt));
    end

    function listenUpdateEqSlice(obj,src,~)
      % update equilibrium slice to match selected time index
      obj.eqslider.set_time_slice(src.time_index);
    end

    function listenUpdateTimeTraces(obj,src,~)
      % update timetraces time index with equilibrium plot's time slice
      obj.timetraces.time_index = src.get_current_time_slice;
    end
  end
end
