function varargout = meqgsave(G,P,opts,fname,verbosity)
%MEQGSAVE Save meq geometry quantities for easy re-loading with MEQGLOAD
%
% meqgsave(G,P[,opts,fname,verbosity])
%   G   : Geometry structure
%   P   : Parameter structure
%   opts: string of objects to save
%            all possible options: 'awxzuvfm'. 
%            default: 'awxzvfm' because loading u is not supported yet
%  fname: file name (default: meqG.mat)
%  verbosity: display level. default: 1, set 0 to silence
% 
% Calling meqgsave with an output argument returns the full path of the stored file
% 
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% Input handling
allopts = 'awxzuvfm';
if nargin<3 || isempty(opts)
  opts = 'awxzvfm';
  if (~P.izgrid), opts(opts == 'z') = [];end % Remove z if P.izgrid is false
end
if nargin<4 || isempty(fname)
  fname = fullfile(pwd,'meqG.mat');
end
if nargin<5 || isempty(verbosity)
  verbosity = 1;
end

% Checks
assert(ischar(opts),'opts must be a string');
assert(all(contains(num2cell(opts),num2cell(allopts))),...
  'opts must contain only characters from the set ''%s''',allopts)
assert(~contains(opts,'z') || P.izgrid , 'must set izgrid=true to store z')
%%
% list of all potential things to save and compute
fulllist = {...
  'Mvv',            'Mxv','Mzv','Mbv','dzMvx','Brxv','Bzxv','Brzv','Bzzv',...
  'Mav',      'Maa','Mxa','Mza',      'dzMax','Brxa','Bzxa','Brza','Bzza',...
  'Mau',      'Muu','Mxu','Mzu',      'dzMux','Brxu','Bzxu','Brzu','Bzzu',...
  'Mfv','Mfw','Mfa','Mfx',            'dzMfx',                            ...
  'Bmv','Bmw','Bmx','Bma',            'dzBmx',                            ...
                          'Mzy',                                          ...
  };
% Matrices for w are not saved since these are intermediate results
%  'Mwv','Mww',      'Mxw','Mzw','Mbw','dzMwx','Brxw','Bzxw','Brzw','Bzzw',...

excludeopts = setdiff(num2cell(allopts),num2cell(opts)); % excluded options
yy = cellfun(@(x) x(end-1:end),fulllist,'UniformOutput',false); % last two letters of each in fulllist
savelist = fulllist(~contains(yy,excludeopts)); % save those not being excluded

if isempty(savelist), warning('MEQGSAVE:emptylist','empty savelist, not saving anything'); return; end

%% Compute geometry quantities including from list of objects to be saved
if verbosity, fprintf('computing geometry information to be saved... '); tic; end
P.gfile = ''; % Avoids loading out-of-date file
G = meqg(G,P,savelist{:});
if verbosity, fprintf('done in %fs\n',toc); end
fields = fieldnames(G)';

% remove *u,*m,*f,*.s fields from structure to be saved, as well as excluded ones
rmfields = fields(endsWith(fields,excludeopts)|endsWith(fields,{'s','c','W'})); 
G = rmfield(G,rmfields);

%% Save to file
if verbosity, fprintf('saving to %s...',fname); tic; end
save(fname,'-struct','G');
if verbosity,  fprintf('done in %fs \n',toc); end

if nargout
  varargout{1} = which(fname); % optional output;
end
