function meqgplot(G,ax,opts)
%MEQGPLOT: Generic geometry structure plotting
% meqplot(G) % plot G structure
% meqplot(G,ax); % pass axis handle
% meqplot(G,ax,options); % pass options to plot
%   options: v: Vessel
%            l: limiter contour
%            m: B probles
%            f: flux loops
%            W: wall gaps
%            w: coil windings
%            x: grid box
%
% typing 'legend show' after the plotting will show a legend
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if nargin<3
  opts = 'tvlmfWwx';
end

if nargin<2 || isempty(ax)
  ax = gca;
end
 
assert(ishandle(ax),'invalid handle supplied');
 
if contains(opts,'x')
  % grid box
  [rrx,zzx] = meshgrid(G.rx,G.zx);
  mesh(rrx,zzx,zeros(size(rrx)),'edgecolor',0.8*[1 1 1],'displayname','grid')
end
hold(ax,'on');

% vessel
if contains(opts,'v')
  % plot per component
  for in = 1:size(G.Tnv,1)
    seln = G.Tnv(in,:)~=0;
    plot(ax,G.rv(seln),G.zv(seln),'.b','displayname',G.dimn{in});
  end
end
 
% limiter
if contains(opts,'l')
  rl = [G.rl;G.rl(1)]; zl = [G.zl;G.zl(1)];
  plot(ax,rl,zl,'color',0.3*[1 1 1],'linewidth',1.5,'displayname','limiter');
end
 
if contains(opts,'w')
  % windings and coils
  plot(ax,G.rw,G.zw,'rs','displayname','coil winding');
end
 
if contains(opts,'f')
  plot(ax,G.rf,G.zf,'xm','displayname','flux loops');
end

if contains(opts,'m')
  plot(ax,G.rm,G.zm,'om','displayname','B probes');
end
 
% gaps
if contains(opts,'W') && G.nW
  plot(ax,[G.rW G.rW-G.aW.*cos(G.oW)]',...
          [G.zW G.zW+G.aW.*sin(G.oW)]','-o','displayname','wall gap');
end
 
view(2);
axis(ax,'equal','tight');
 
end
