function meqcompare(varargin)
%MEQCOMPARE graphically compare static equilibria
% example:
%    meqcompare(L,LY1,LY2) % compare 2 cases with same L structure
%    meqcompare(L1,LY1,L2,LY2,...) % compare 2 (or more) cases with different L structures
%    meqcompare(L1,LY1) % just plot one case
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% parse inputs
iL = 0; nLY = 0;
Ls = {}; % init

for ii=1:numel(varargin)
  argin = varargin{ii};
  if isempty(argin), continue; end % empty LY
  assert(isstruct(argin),'all non-empty input arguments must be structures');
  if isfield(argin,'P') % L structure
    iL = iL + 1; 
    Ls{iL} = argin; %#ok<AGROW> % append new L
    iLY = 0;
  elseif isfield(argin,'Fx') % LY structure
    LY = argin;
    assert(numel(LY.t)==1,'Can only compare scalar LY structures, use meqxk() to slice')

    % check compatibility with corresponding L structure
    check_compatibility(Ls{iL},LY);

    iLY = iLY + 1; % LY counter for this L
    nLY = nLY + 1; % total LY counter
    LYs{iL}{iLY} = LY; %#ok<AGROW> % append
  else
    error('input does not look like an L or LY structure')
  end
end
assert(nLY<5,'meqcompare supports comparing up to a maximum of 4 LY');

% setup handles
hs(1) = subplot(1,2,1);
hs(2) = subplot(6,2,2);
hs(3) = subplot(6,2,4);
hs(4) = subplot(6,2,6);
hs(5) = subplot(6,2,8);
hs(6) = subplot(6,2,10);
hs(7) = subplot(6,2,12);


wbar=0.8/nLY; % bar graph widths

LineStyleOrder = {'-','--','-.',':'};
ii = 0; % total LY counter

for iL = 1:numel(Ls) % for L structures
  L = Ls{iL};
  if iL == 1
    % plot vessel, limiter, wall
    meqgplot(L.G,hs(1),'vlw'); hold(hs(1),'on');
  end

  nLY = numel(LYs{iL});
  for iLY = 1:nLY
    LY = LYs{iL}{iLY};
    ii = ii+1;

    %% flux plots
    ax=hs(1); hold(ax,'on');
    col = ax.ColorOrder(ii,:);
    sty = LineStyleOrder{ii};

    FFB = repmat(LY.FB,1,max(3-numel(LY.FB),1));
    if ii==1; F=linspace(min(LY.Fx(:)),max(LY.Fx(:)),21); end % same contour values for all cases

    contour(ax,L.G.rx,L.G.zx,LY.Fx,F  ,'linecolor',col              ,'LineStyle',sty);
    contour(ax,L.G.rx,L.G.zx,LY.Fx,FFB,'linecolor',col,'linewidth',2,'LineStyle',sty);

    plot(ax,LY.rA,LY.zA,'ow');
    plot(ax,LY.rB,LY.zB,'gs');

    axis(ax,'equal','tight');

    %% profiles
    ax = hs(2);
    plot(ax,L.pQ,mu0*LY.PpQ,'color',col,'LineStyle',sty);
    hold(ax,'on');
    title(ax,'\mu_0p''')

    ax = hs(3);
    plot(ax,L.pQ,LY.TTpQ,'color',col,'LineStyle',sty);
    title(ax,'TT''')
    hold(ax,'on');

    %% Currents
    ax = hs(4);
    bar(ax,(1:L.G.na)+(ii-1)*wbar+0.1,LY.Ia,wbar,'FaceColor',col)
    hold(ax,'on');
    title(ax,'Ia');

    ax = hs(5);
    plot(ax,1:L.G.nv,LY.Iv,'Color',col,'LineStyle',sty)
    hold(ax,'on');
    title(ax,'Iv');

    %% measurements
    ax = hs(6);
    bar(ax,(1:L.G.nm)+(ii-1)*wbar+0.1,LY.Bm,wbar,'FaceColor',col)
    hold(ax,'on');
    title(ax,'Bm');

    ax = hs(7);
    bar(ax,(1:L.G.nf)+(ii-1)*wbar+0.1,LY.Ff,wbar,'FaceColor',col)
    hold(ax,'on');
    title(ax,'Ff');
  end
end
end

function check_compatibility(L,LY)
  for fld = {'Fx','Ia','Iv','Ff','Bm','TTpQ','PpQ'}
    myfld = fld{1};
    sz_expect = meqsize(L,myfld);
    sz_true = size(LY.(myfld));
    msg = sprintf('size of %s is not compatible - do the LYs have different L structures?',myfld);
    errID = 'MEQCOMPARE:IncompatibleLandLY';
    assert(isequal(sz_expect,sz_true),errID,msg)
  end
end