function S = meqcdefun(h)
% meqcdefun container for cde constraint functions
% Function signature:
% [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%      dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%   mycde(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,iD)
%
% Function jacobians are only computed and returned if nargout>1
%
% cde constraint functions Arguments:
%   L:        struct, Standard L structure
%   LX:       struct, Standard LX structure. Should contain LX.signeo
%   LY:       struct, Standard LY structure containing information from
%             last time step
%   Fx:       double(nzx, nrx), Flux field on X-grid
%   ag:       double(ng, 1), ag values
%   Iy:       double(nzy, nry), Plasma current field on Y-grid
%   F0:       double(nD, 1), Axis flux value per domain
%   F1:       double(nD, 1), Separatrix flux value per domain
%   Ie:       double(ne, 1), Active coil and vessel component currents
%   Opy:      int8(nzy, nry), Plasma domain segmentation map on Y-grid
%   idt:      double, Inverse of time-step. Could be 0. in certain cases.
%   iD:       int, Flag indicating for which domain the cde should be
%             evaluated
% cde constraint functions Returns:
%   res:      double(npD, 1), Residual of CDE for domain specified by iD
%   dresdFx:  double(npD, nx), jacobian of residual w.r.t. Fx input
%   dresdag:  double(npD, ng), jacobian of residual w.r.t. ag input
%   dresdIy:  double(npD, ny), jacobian of residual w.r.t. Iy input
%   dresdF0:  double(npD, 1),  jacobian of residual w.r.t. F0(iD) input
%   dresdF1:  double(npD, 1),  jacobian of residual w.r.t. F1(iD) input
%   dresdIe:  double(npD, ne), jacobian of residual w.r.t. Ie input
%   dresdIni: double(npD, nD), jacobian of residual w.r.t. non-inductive currents
%   dresdFxdot: double(npD, nx), jacobian of residual w.r.t. change in Fx
%   dresdagdot: double(npD, ng), jacobian of residual w.r.t. change in ag
%   dresdIydot: double(npD, ny), jacobian of residual w.r.t. change in Iy
%   dresdF0dot: double(npD, 1),  jacobian of residual w.r.t. change in F0(iD)
%   dresdF1dot: double(npD, 1),  jacobian of residual w.r.t. change in F1(iD)
%   dresdIedot: double(npD, ne), Derivatives of residual w.r.t. change in Ie
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

fun   = localfunctions;
names = cellfun(@func2str,fun,'UniformOutput',false);
S = cell2struct(fun,names);
if nargin==0, return; end

if ischar(h) && isequal(h,'help')
  help(mfilename);
  fprintf('\nCatalog of available %s CDE options\n',mfilename)
  for ii=1:numel(fun)
    myfun = names{ii};
    fprintf('%s:\n',myfun)
    help(sprintf('%s>%s',mfilename,myfun));
  end
end
end


function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
              dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
         OhmTor_rigid_0D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,iD)
% OHMTOR_RIGID_0D CDE of plasma current under rigid plasma assumption
% function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%             dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%           OhmTor_rigid_0D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,iD)
%
% CDE residual under rigid plasma assumption.
% Formulation is derived from circuit equations for the whole Y-grid under
% the projection to the IyD / Ip component.
% Hence, residual is formulated as
%     res = Lp(iD) * IpD_dot + Rp(iD) * IpD + Fext_dot*(IyD/IpD),
% where Rp is given,
% Fext is the induced change in flux from all sources outside domain iD
% and Lp can either be provided through LX or computed as
%     Lp = IyD^T Myy IyD / IpD^2,
% depending on L.P.iLpext
%
% For inputs / outputs, see meqcdefun
%
% [+MEQ MatlabEQuilibrium Toolbox+]

IyD  =    Iy .* (   Opy==iD);
zIyD = LY.Iy .* (LY.Opy==iD);
IpD  = sum(IyD( :));
IpDz = sum(zIyD(:));

% Compute loop voltage felt by plasma domain from changes in all external currents
if L.nD == 1 % For single domain plasmas, only external coils change the flux
  zFye = L.Mye * [LY.Ia; LY.Iu];
  Fye = L.Mye * Ie;
  Fydot = (Fye - zFye) * idt;
  if ~L.P.iLpext % Plasma flux contribution is needed for Lp computation
    FyD = reshape(Fx(2:end-1,2:end-1),L.ny,1) - Fye;
  end
else % For multi domain plasmas, changes in other plasma domains have to be considered
  % flux contribution from iD domain for current and last timestep
  FyD  = meqFx(L, IyD);  FyD =  FyD(L.lxy);
  zFyD = meqFx(L,zIyD); zFyD = zFyD(L.lxy);
  % external change in flux by total change in flux minus contribution from iD domain
  Fydot = Fx(2:end-1, 2:end-1) - LY.Fx(2:end-1, 2:end-1);
  Fydot = (Fydot(:) - FyD + zFyD) * idt; 
end
  
if ~L.P.iLpext
  % equivalent to IyD' Myy IyD / IpD^2
  Lp = IyD(:)' * FyD / IpD^2;
else
  Lp = LX.Lp(iD);
end

% Multi domain IniD
if L.nD > 1, IniD = LX.IniD; else, IniD = LX.Ini; end

res = Lp*(IpD - IpDz)*idt + LX.Rp(iD)*(IpD-IniD(iD)) + Fydot'*IyD(:)/IpD;

if nargout>1
  % plasma domain mask derivatives
  dIyDdIy = Opy(:)'==iD;
  dIpDdIy = dIyDdIy;
  
  % Fydot term derivatives, depending on how Fydot is computed
  if L.nD == 1
    dFydotdFy = 0;
    IyDdFydotdIy = zeros(1, L.ny);
    dFydotdIe = L.Mye * idt;
    dFydotdFydot = 0;
    IyDdFydotdIydot = zeros(1, L.ny);
    dFydotdIedot = L.Mye;
  else
    % to only multiply with Myy once
    IyDdFydIy = (IyD(:)' * L.Myy) .* dIyDdIy; 
    
    dFydotdFy = idt;
    IyDdFydotdIy = -IyDdFydIy * idt;
    dFydotdIe = zeros(L.ny, L.ne);
    dFydotdFydot = 1;
    IyDdFydotdIydot = -IyDdFydIy;
    dFydotdIedot = zeros(L.ny, L.ne);
  end
  
  % Lp derivatives, depending on how Fydot is computed
  if ~L.P.iLpext
    if L.nD == 1
      dLpdFy = IyD(:)'/IpD^2;
      dLpdIy = FyD(:)'.*dIyDdIy/IpD^2 - (IyD(:)'*FyD/IpD^3)*dIpDdIy;
      dLpdIe = -IyD(:)' * L.Mye / IpD^2;
    else
      dLpdFy = zeros(1, L.ny);
      dLpdIy = FyD(:)'.*dIyDdIy/IpD^2 - (IyD(:)'*FyD/IpD^3)*dIpDdIy + IyDdFydIy/IpD^2;
      dLpdIe = zeros(1, L.ne);
    end
  else
    dLpdFy = zeros(1, L.ny);
    dLpdIy = zeros(1, L.ny);
    dLpdIe = zeros(1, L.ne);
  end
  
  % filling residual jacobians
  dresdFx = zeros(1, L.nx);
  dresdFx(:, L.lxy) = (IpD - IpDz)*idt*dLpdFy + IyD(:)'*dFydotdFy/IpD;
  dresdag = zeros(1, L.ng);
  dresdIy = (IpD - IpDz)*idt*dLpdIy + (Lp*idt+LX.Rp(iD)-Fydot'*IyD(:)/IpD^2)*dIpDdIy + ...
            (Fydot'.*dIyDdIy)/IpD + IyDdFydotdIy/IpD;
  dresdF0 = 0;
  dresdF1 = 0;
  dresdIe = (IpD - IpDz)*idt*dLpdIe + IyD(:)'*dFydotdIe/IpD;
  
  dresdIni = -LX.Rp(iD);
  
  dresdFxdot = zeros(1, L.nx);
  dresdFxdot(:, L.lxy) = IyD(:)'*dFydotdFydot/IpD;
  dresdagdot = zeros(1, L.ng);
  dresdIydot = Lp*dIyDdIy + IyDdFydotdIydot/IpD;
  dresdF0dot = 0;
  dresdF1dot = 0;
  dresdIedot = IyD(:)'*dFydotdIedot/IpD;
end
end


function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
              dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
         OhmTor_0D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,iD)
% OHMTOR_0D CDE of plasma current by toroidal projection of Ohm's law
% function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%             dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%           OhmTor_0D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,iD)
%
% Formulation is derived from circuit equations where loop voltage is given
% by the change in toroidal flux and using the projection onto the Iy/Ip
% component.
% Hence, residual is formulated as
%     res = Iy/Ip * Fx_dot + Rp * Ip,
% where Rp is given in LX.
%
% For inputs / outputs, see meqcdefun
%
% [+MEQ MatlabEQuilibrium Toolbox+]

IyD = Iy(:) .* (Opy(:)==iD);

dFx = (Fx - LY.Fx);
dFy = dFx(2:end-1, 2:end-1);
IpD = sum(IyD(:));
if IpD ~= 0
  IyN = IyD / IpD;
else
  IyN = (Opy(:) == iD) / sum((Opy(:) == iD));
end

% Multi domain IniD
if L.nD > 1, IniD = LX.IniD; else, IniD = LX.Ini; end

res = dFy(:)'*IyN*idt + LX.Rp(iD)*(IpD - IniD(iD));

if nargout>1
  dresdF0 = 0; dresdF1 = 0;
  dresdag = zeros(1, L.ng);
  dresdIe = zeros(1, L.ne);
  
  dresdFx = zeros(1, L.nx);
  dresdFx(:, L.lxy) = IyN * idt;
  
  dIpDdIy = (Opy(:)==iD).';
  if IpD ~= 0
    dIyNdIy = (Opy(:) == iD) / IpD;
    dIyNdIpD = -IyD / (IpD * IpD);
  else
    dIyNdIy = zeros(L.ny, 1);
    dIyNdIpD = zeros(L.ny, 1);
  end

  dresdIy =  dFy(:)' .* dIyNdIy.' * idt + ...
            (dFy(:)' * dIyNdIpD) * dIpDdIy * idt + ...
             LX.Rp(iD) * dIpDdIy;
  
  dresdIni = -LX.Rp(iD);
  
  dresdFxdot = zeros(1, L.nx);
  dresdFxdot(:, L.lxy) = IyN;
  
  dresdagdot = zeros(1, L.ng);
  dresdIydot = zeros(1, L.ny);
  dresdF0dot = zeros(1, 1);
  dresdF1dot = zeros(1, 1);
  dresdIedot = zeros(1, L.ne);
end
end


function [res, dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
               dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
         cde_ss_0D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,iD)
% CDE_SS_0D steady state plasma current equation derived from RAPTOR CDE formulation
% function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%             dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%           cde_ss_0D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,iD)
%
% Formulation is derived from RAPTOR CDE by integrating it from rho=0 to rho=1
% and assuming steady state, i.e. dPhi/dt = 0
% Residual reads
%     res = Ip - Ini + Tb * Phib / (2 pi) * int [sigma / T^2] dPhi * d(FA + FB)/2dt
% Profile computation is used to compute things like Phi, T etc. for the
% integral.
% For derivation, see equations G.17, G.20, G.21, G.24 in Frascesco
% Carpanese's thesis. [https://doi.org/10.5075/epfl-thesis-7914]
%
% For inputs / outputs, see meqcdefun
%
% [+MEQ MatlabEQuilibrium Toolbox+]

[~,~,~,~,FtQ,Ft0Q] = meqintQ(L,F0,F1,LX.rBt,ag,Fx,Opy);
FtPQ = FtQ(:, iD)+Ft0Q(:, iD); % total Plasma toroidal flux on pQ grid
[~,~,~,TQ] = meqprof(L.fPg,L.fTg,ag,L.pQ(:).^2,F0,F1,LX.rBt,L.bfct,L.bfp,L.idsx);

FAdot = (F0(iD) - LY.F0(iD)) * idt;
FBdot = (F1(iD) - LY.F1(iD)) * idt;
FABdot = (FBdot + FAdot)/2;

iTsQ = LX.signeo(:, iD) ./ TQ(:, iD).^2; % sigma/F^2 profile appearing in various places
SQ =  (LX.rBt ./ (2 .* pi)) * trapz(FtPQ, iTsQ);
A1Q = SQ*FABdot;

% Multi domain IniD
if L.nD > 1, IniD = LX.IniD; else, IniD = LX.Ini; end

scal = 1e-6;
res = scal*(sum(Iy(:) .* (Opy(:) == iD)) - IniD(iD) + A1Q);

if nargout>1
  dresdIe = zeros(1, L.ne);
  
  [~,~,~,dFtQdag,~,~,~,dFtQdFx,...
    ~,~,~,dFtQdF0,~,~,~,dFtQdF1] = ...
    meqintQJac(L,F0,F1,LX.rBt,ag,Fx,Opy);
  
  dFtQdag = squeeze(dFtQdag(:, iD, :));
  dFtQdFx = squeeze(dFtQdFx(:, iD, :));
  dFtQdF0 = squeeze(dFtQdF0(:, iD, iD));
  dFtQdF1 = squeeze(dFtQdF1(:, iD, iD));
  
  [~,~,~,dTQdag,~,~,~,~,dTQdF0,~,~,~,~,dTQdF1,~] = ...
    meqprofJac(L.fPg,L.fTg,ag,L.pQ(:).^2,F0,F1,LX.rBt,L.bfct,L.bfp,L.idsx);
  
  dTQdag = squeeze(dTQdag(:, iD, :));
  dTQdF0 = squeeze(dTQdF0(:, iD, iD));
  dTQdF1 = squeeze(dTQdF1(:, iD, iD));
  
  dresdIy = scal*double((Opy(:) == iD).');
  
  diTsQdF0 = -2 .* LX.signeo(:, iD) ./ TQ(:, iD).^3 .* dTQdF0;
  diTsQdF1 = -2 .* LX.signeo(:, iD) ./ TQ(:, iD).^3 .* dTQdF1;
  diTsQdFag = -2 .* LX.signeo(:, iD) ./ TQ(:, iD).^3 .* dTQdag;
  
  dSQdFtPQ = zeros(1, L.nQ);
  dSQdFtPQ(1:end-1) = -0.5 * (iTsQ(1:end-1) + iTsQ(2:end)).';
  dSQdFtPQ(2:end) = dSQdFtPQ(2:end) + 0.5 * (iTsQ(1:end-1) + iTsQ(2:end)).';
  dSQdFtPQ = dSQdFtPQ .* (LX.rBt ./ (2 .* pi));
  
  dSQdiTsQ = zeros(1, L.nQ);
  dSQdiTsQ(1:end-1) = 0.5 * (FtPQ(2:end) - FtPQ(1:end-1)).';
  dSQdiTsQ(2:end) = dSQdiTsQ(2:end) + 0.5 * (FtPQ(2:end) - FtPQ(1:end-1)).';
  dSQdiTsQ = dSQdiTsQ .* (LX.rBt ./ (2 .* pi));
  
  dSQdF0 = dSQdFtPQ * dFtQdF0  + dSQdiTsQ * diTsQdF0;
  dSQdF1 = dSQdFtPQ * dFtQdF1  + dSQdiTsQ * diTsQdF1;
  dSQdag = dSQdFtPQ * dFtQdag  + dSQdiTsQ * diTsQdFag;
  dSQdFx = dSQdFtPQ * dFtQdFx; % diTsQdFx = 0, therefore omitted
  
  dresdF0 = scal*(dSQdF0 * FABdot + SQ * 0.5 * idt);
  dresdF1 = scal*(dSQdF1 * FABdot + SQ * 0.5 * idt);
  
  dresdag = scal*(dSQdag * FABdot);
  dresdFx = scal*(dSQdFx * FABdot);
  
  dresdIni = -scal;
  
  dresdFxdot = zeros(1, L.nx);
  dresdagdot = zeros(1, L.ng);
  dresdIydot = zeros(1, L.ny);
  dresdF0dot = 0.5 * scal * SQ;
  dresdF1dot = 0.5 * scal * SQ;
  dresdIedot = zeros(1, L.ne);
end
end


function [res, dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
               dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
         static_Iy_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
% static_Iy_1D CDE for constraining plasma current against reference under test functions
% function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%             dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%           static_Iy_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
%
% Technically not a CDE. This is a static CDE-like residual to constrain the 
% plasma current under test functions. igD indicates what test functions to
% use. Mathematically, this constrains for all test functions g:
%   int LX.Iy * g dR dZ - int Iy * g dR dZ
% This represents a natural extension of the Ip constraint to constraining
% the multiple degrees of freedom of the jtor profile.
%
% This residual is mainly used for initializing a simulation using another
% 1D CDE. In general, a 1D CDE leaves too many degrees of freedom open to
% be constrained by classical plasma constraints, hence this one can be
% used for any input LX.Iy current distribution
%
% For inputs / outputs, see meqcdefun
%
% [+MEQ MatlabEQuilibrium Toolbox+]

% mask of basis function supports
Opy_ag = zeros(L.ny, L.ng);
for iD=1:L.nD
  Opy_ag(:, logical(L.TDg(iD, :))) = Opy_ag(:, logical(L.TDg(iD, :))) | (Opy(:) == iD);
end

% get basis function fields
gQg = L.bfct(91, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
gQg = gQg(:, igD) .* Opy_ag(:, igD);

% mask of basis function supports
LX_Opy_ag = zeros(L.ny, L.ng);
for iD=1:L.nD
  LX_Opy_ag(:, logical(L.TDg(iD, :))) = LX_Opy_ag(:, logical(L.TDg(iD, :))) | (LX.Opy(:) == iD);
end
% get basis function fields evaluated on reference flux map
% this ensures that the constraint behaves well even if the plasma is in a
% different position
gQg_ref = L.bfct(91, L.bfp, LX.Fx(2:end-1, 2:end-1), LX.F0, LX.F1);
gQg_ref = gQg_ref(:, igD) .* LX_Opy_ag(:, igD);

% scale by Ip
scal = 1 ./ (L.Ip0 * L.intG0(igD));
res = scal .* gQg.' * Iy(:) - scal .* gQg_ref.' * LX.Iy(:);

if nargout >= 2
  dresdIe = 0; dresdIni = 0; dresdag = 0;
  dresdFxdot = 0; dresdagdot = 0; dresdIydot = 0; dresdF0dot = 0; dresdF1dot = 0; dresdIedot = 0;
  
  % fill jacobians of gQg with mode 92
  [dgdFx,dgdF0,dgdF1] = L.bfct(92, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
  dgQgdFx = dgdFx(:, igD   ) .* Opy_ag(:, igD);
  dgQgdF0 = dgdF0(:, igD, :) .* Opy_ag(:, igD);
  dgQgdF1 = dgdF1(:, igD, :) .* Opy_ag(:, igD);
  
  dresdFx = zeros(sum(igD), L.nx);
  dresdFx(:, L.lxy) = scal .* (Iy(:) .* dgQgdFx).';
  dresdF0 = scal .* reshape(sum(reshape(Iy, L.ny, 1, 1) .* dgQgdF0, 1), [], L.nD);
  dresdF1 = scal .* reshape(sum(reshape(Iy, L.ny, 1, 1) .* dgQgdF1, 1), [], L.nD);
  
  dresdIy = scal .* gQg.';
end    
end


function [res, dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
               dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
         cde_OhmTor_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
% cde_OhmTor_1D toroidal projection of Ohm's law under test functions
% function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%             dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%           cde_OhmTor_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
%
% 1D Formulation of toroidal projection of Ohm's law. The residual given by:
%   int psi_dot / r * g dRdZ + int 2 pi / sigma * (j_phi - j_ni) * g dRdZ
% The gQg are used as test functions, where igD indicates which
% test functions should be used.
%
% For inputs / outputs, see meqcdefun
%
% [+MEQ MatlabEQuilibrium Toolbox+]

TDg = logical(L.TDg); % casting to avoid errors if L.TDg is double
% compute j_phi
j_phi = Iy .* L.idsx;

% make sigma array
FNy = reshape(Fx(2:end-1, 2:end-1), L.nzy, L.nry);
FNy = (FNy - reshape(F0, 1, 1, L.nD)) ./ reshape(F1 - F0, 1, 1, L.nD);
act_iDs = find(F0~=F1).';
sigma = zeros(L.nzy, L.nry);
for iD=act_iDs
  sigma = sigma + interp1(L.pQ.^2, LX.signeo(:, iD), FNy(:, :, iD), ...
                          'linear', 0) .* (Opy == iD);
end
sigma = sigma + LX.signeo(end, end) .* (Opy == 0); % last value outside plasma

% make jni_parallel array
% This CDE needs a jni profile which is currently not supported in MEQ
% For now, Ini is expected to be 0
if isscalar(LX.Ini)
  jni_parallel = 0;
  if LX.Ini ~= 0
    warning('cde_OhmTor_1D only works with 1D non-inductive current')
  end
else
  jni_parallel = zeros(L.nzy, L.nry);
  for iD=act_iDs
    jni_parallel = jni_parallel + interp1(L.pQ.^2, LX.Ini(:, iD), FNy(:, :, iD), ...
                     'linear', 0) .* (Opy == iD);
  end
end

% mask of basis function supports
Opy_ag = false(L.ny, L.ng);
for iD=1:L.nD
  Opy_ag(:, TDg(iD, :)) = Opy_ag(:, TDg(iD, :)) | (Opy(:) == iD);
end
% get basis function fields
gQg = L.bfct(91, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
gQg = gQg(:, igD) .* Opy_ag(:, igD);

% integrand of weak formulation
integrand = (Fx(2:end-1, 2:end-1) - LY.Fx(2:end-1, 2:end-1)) ./ L.rry .* idt;
integrand = integrand + 2 .* pi .* (j_phi - jni_parallel) ./ sigma;

integrand = reshape(integrand, 1, []);

% approximation of integral against test functions
scal = L.intG0(igD);
res = integrand * gQg .* L.dsx;
res = res.' ./ scal;

if nargout >= 2
  dresdintegrand = gQg.' .* L.dsx;
  dresdgQg = integrand .* L.dsx; % dintegranddgQg = 0 in this case
                    
  % sigma field jacobian
  dsigmadFy = zeros(L.nzy, L.nry);
  dsigmadF0 = zeros(L.nzy, L.nry, L.nD);
  dsigmadF1 = zeros(L.nzy, L.nry, L.nD);
  for iD=act_iDs
    % using stepped difference as approximation of interpolation derivative
    signeo_grads = diff(LX.signeo(:, iD)) ./ diff(L.pQ.^2).';
    dsdFy_iD = interp1(L.pQ.^2, [signeo_grads; 0], FNy(:, :, iD), 'previous', 0) .* (Opy == iD);
    dsigmadFy = dsigmadFy + dsdFy_iD ./ (F1(iD) - F0(iD));
    dsigmadF0(:, :, iD) = dsdFy_iD .* (Fx(2:end-1, 2:end-1) - F1(iD)) ./ (F1(iD) - F0(iD)).^2;
    dsigmadF1(:, :, iD) = dsdFy_iD .* (F0(iD) - Fx(2:end-1, 2:end-1)) ./ (F1(iD) - F0(iD)).^2;
  end
  dsigmadFy = reshape(dsigmadFy, L.ny, 1);
  dsigmadF0 = reshape(dsigmadF0, L.ny, L.nD);
  dsigmadF1 = reshape(dsigmadF1, L.ny, L.nD);
  
  % fill jacobians with mode 92
  [dgdFx,dgdF0,dgdF1] = L.bfct(92, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
  dgQgdFx = dgdFx(:, igD   ) .* Opy_ag(:, igD);
  dgQgdF0 = dgdF0(:, igD, :) .* Opy_ag(:, igD);
  dgQgdF1 = dgdF1(:, igD, :) .* Opy_ag(:, igD);
  
  tmp = -2 .* pi .* (j_phi(:) - jni_parallel(:)) ./ (sigma(:) .* sigma(:));
  dintegranddFy = idt ./ L.rry(:) + dsigmadFy .* tmp;
  dintegranddF0 = dsigmadF0 .* tmp;
  dintegranddF1 = dsigmadF1 .* tmp;
  
  dresdFx = zeros(sum(igD), L.nx);
  dresdFx(:, L.lxy) = (dresdgQg .* dgQgdFx.' + dresdintegrand .* dintegranddFy.') ./ scal;
  dresdag = 0; % no dependence for ags
  dresdIy = (2 .* pi ./ sigma(:)).' .* gQg.' ./ scal; % dgQgdIy=0
  dresdIe = 0;
  dresdIni = 0; % until 1D Ini is supported
  dresdF0 = (reshape(sum(dresdgQg.' .* dgQgdF0, 1), sum(igD), L.nD) + dresdintegrand * dintegranddF0) ./ scal;
  dresdF1 = (reshape(sum(dresdgQg.' .* dgQgdF1, 1), sum(igD), L.nD) + dresdintegrand * dintegranddF1) ./ scal;
  
  
  dresdFxdot = zeros(sum(igD), L.nx);
  dresdFxdot(:, L.lxy) = (1 ./ L.rry(:)).' .* gQg.' .* L.dsx ./ scal;
  dresdagdot = 0;
  dresdIydot = 0;
  dresdF0dot = 0;
  dresdF1dot = 0;
  dresdIedot = 0;
end
end


function [res, dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
               dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
         cde_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
% cde_1D Full weak formulation of 1D CDE with parallel conductivity
% function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%             dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%           cde_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
%
% 1D weak formulation of Ohm's law. The residual in terms of the parallel
% conductivity is derived by treating the toroidal part as in cde_OhmTor_1D
% and the poloidal part by considering changes in the TT' profile.
% The residual given by:
%   int Tdot / r * g dRdZ + int psi_dot / r * T * g' dRdZ + int 2 pi / sigma * (j_phi - j_ni) * T * g' dRdZ + ...
%   int |grad psi|^2 / (sigma * mu0 * r) * T' * g' dRdZ
% The gQg are used as test functions, where igD indicates which
% test functions should be used.
%
% For inputs / outputs, see meqcdefun
%
% [+MEQ MatlabEQuilibrium Toolbox+]

% some inits
TDg = logical(L.TDg);
irry = 1 ./ L.rry;
j_phi = Iy .* L.idsx;

% make sigma array
FNy = reshape(Fx(2:end-1, 2:end-1), L.nzy, L.nry);
FNy = (FNy - reshape(F0, 1, 1, L.nD)) ./ reshape(F1 - F0, 1, 1, L.nD);
act_iDs = find(F0~=F1).';
sigma = zeros(L.nzy, L.nry);
for iD=act_iDs
  sigma = sigma + interp1(L.pQ.^2, LX.signeo(:, iD), FNy(:, :, iD), 'linear', 0) .* (Opy == iD);
end
sigma = sigma + LX.signeo(end, end) .* (Opy == 0); % last value outside plasma
isigma = 1 ./ sigma;

% make jni_parallel array
% This CDE needs a jni profile which is currently not supported in MEQ
% For now, Ini is expected to be 0
if isscalar(LX.Ini)
  jni_parallel = 0;
  if LX.Ini ~= 0
    warning('cde_1D only works with 1D non-inductive current')
  end
else
  jni_parallel = zeros(L.nzy, L.nry);
  for iD=act_iDs
    jni_parallel = jni_parallel + interp1(L.pQ.^2, LX.Ini(:, iD), FNy(:, :, iD), ...
                     'linear', 0) .* (Opy == iD);
  end
end

% mask of basis function supports
Opy_ag = false(L.ny, L.ng);
zOpy_ag = false(L.ny, L.ng);
for iD=1:L.nD
  Opy_ag(:, TDg(iD, :)) = Opy_ag(:, TDg(iD, :)) | (Opy(:) == iD);
  zOpy_ag(:, TDg(iD, :)) = zOpy_ag(:, TDg(iD, :)) | (LY.Opy(:) == iD);
end
% get basis function fields
[G, IG] = L.bfct(91, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
[~, zIG] = L.bfct(91, L.bfp, LY.Fx(2:end-1, 2:end-1), LY.F0, LY.F1);
G = G .* Opy_ag; IG = IG .* Opy_ag;
zIG = zIG .* zOpy_ag;

[~,aTTpg,~,ahqTg] = L.bfct(3,L.bfp,ag,zeros(L.nD, 1),ones(L.nD, 1),L.fPg,L.fTg,L.idsx);
[~,~,~,zahqTg]    = L.bfct(3,L.bfp,LY.ag,zeros(L.nD, 1),ones(L.nD, 1),L.fPg,L.fTg,L.idsx);

% T^2 and TT' fields
Tsq_field = IG * ahqTg;
Tsq_field_old = zIG * zahqTg;
TTp_field = G * aTTpg;

% get T fields
T_field = sqrt(2 * reshape(Tsq_field, L.nzy, L.nry) + LX.rBt.^2) .* sign(LX.rBt);
T_field_old = sqrt(2 * reshape(Tsq_field_old, L.nzy, L.nry) + LX.rBt.^2) .* sign(LX.rBt);
TTp_field = reshape(TTp_field, L.nzy, L.nry);
Tp_field = TTp_field ./ T_field;

% stuff being test against test function
integrand_IG = irry .* (T_field - T_field_old) .* idt;

% stuff being test against derivative of test function
integrand_G = 2 .* pi .* j_phi .* T_field .* isigma;
integrand_G = integrand_G + T_field .* irry .* (Fx(2:end-1, 2:end-1) - LY.Fx(2:end-1, 2:end-1)) .* idt;
integrand_G = integrand_G - 2 .* pi .* L.rry .* jni_parallel .* isigma;

% grad psi squared part
gradr_Fx = 0.5 .* (Fx(2:end-1, 3:end) - Fx(2:end-1, 1:end-2)) .* L.idrx;
gradz_Fx = 0.5 .* (Fx(3:end, 2:end-1) - Fx(1:end-2, 2:end-1)) .* L.idzx;
gradFx_sq = gradr_Fx .* gradr_Fx + gradz_Fx .* gradz_Fx;

integrand_G = integrand_G + gradFx_sq .* Tp_field .* isigma .* irry .* (1 / mu0);

integrand_G = reshape(integrand_G, 1, []);
integrand_IG = reshape(integrand_IG, 1, []);

scal = L.intG0(igD);
res = (integrand_IG * IG(:, igD) + integrand_G * G(:, igD)) * L.dsx;
res = res.' ./ scal;

if nargout >= 2
  % G, IG derivatives
  [dGdFy,dGdF0,dGdF1,dIGdFy,dIGdF0,dIGdF1] = L.bfct(92, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
  dGdFy = dGdFy .* Opy_ag; dGdF0 = dGdF0 .* Opy_ag; dGdF1 = dGdF1 .* Opy_ag;
  dIGdFy = dIGdFy .* Opy_ag; dIGdF0 = dIGdF0 .* Opy_ag; dIGdF1 = dIGdF1 .* Opy_ag;
  
  % aTTpg, ahqTg jacobians, bfct 3 mode is linear in ag coefficients
  [~,daTTpgdag,~,dahqTgdag] = L.bfct(3,L.bfp,ones(L.ng, 1),zeros(L.nD, 1),ones(L.nD, 1),L.fPg,L.fTg,L.idsx);
  
  % Tsq, TTp jacobians
  dTsqdFy = dIGdFy * ahqTg;
  dTsqdF0 = reshape(sum(dIGdF0 .* ahqTg', 2), L.ny, L.nD); % matrix mult in second dimension for doublets
  dTsqdF1 = reshape(sum(dIGdF1 .* ahqTg', 2), L.ny, L.nD);
  dTsqdag = IG .* dahqTgdag';
  dTTpdFy = dGdFy * aTTpg;
  dTTpdF0 = reshape(sum(dGdF0 .* aTTpg', 2), L.ny, L.nD);
  dTTpdF1 = reshape(sum(dGdF1 .* aTTpg', 2), L.ny, L.nD);
  dTTpdag = G .* daTTpgdag';
  
  % T, Tp jacobians
  iT_field = 1 ./ T_field(:);
  dTdFy = dTsqdFy .* iT_field;
  dTdF0 = dTsqdF0 .* iT_field;
  dTdF1 = dTsqdF1 .* iT_field;
  dTdag = dTsqdag .* iT_field;
  iT_field_cubed = iT_field .* iT_field .* iT_field;
  dTpdFy = dTTpdFy .* iT_field - TTp_field(:) .* dTsqdFy .* iT_field_cubed;
  dTpdF0 = dTTpdF0 .* iT_field - TTp_field(:) .* dTsqdF0 .* iT_field_cubed;
  dTpdF1 = dTTpdF1 .* iT_field - TTp_field(:) .* dTsqdF1 .* iT_field_cubed;
  dTpdag = dTTpdag .* iT_field - TTp_field(:) .* dTsqdag .* iT_field_cubed;
  
  % sigma field jacobian
  dsigmadFy = zeros(L.nzy, L.nry);
  dsigmadF0 = zeros(L.nzy, L.nry, L.nD);
  dsigmadF1 = zeros(L.nzy, L.nry, L.nD);
  for iD=act_iDs
    % using stepped difference as approximation of interpolation derivative
    signeo_grads = diff(LX.signeo(:, iD)) ./ diff(L.pQ.^2).';
    dsdFy_iD = interp1(L.pQ.^2, [signeo_grads; 0], FNy(:, :, iD), 'previous', 0) .* (Opy == iD);
    dsigmadFy = dsigmadFy + dsdFy_iD ./ (F1(iD) - F0(iD));
    dsigmadF0(:, :, iD) = dsdFy_iD .* (Fx(2:end-1, 2:end-1) - F1(iD)) ./ (F1(iD) - F0(iD)).^2;
    dsigmadF1(:, :, iD) = dsdFy_iD .* (F0(iD) - Fx(2:end-1, 2:end-1)) ./ (F1(iD) - F0(iD)).^2;
  end
  dsigmadFy = reshape(dsigmadFy, L.ny, 1);
  dsigmadF0 = reshape(dsigmadF0, L.ny, L.nD);
  dsigmadF1 = reshape(dsigmadF1, L.ny, L.nD);
  
  % more dFx stuff
  dintegrand_GdT = 2 .* pi .* j_phi .* isigma + ...
                   (Fx(2:end-1, 2:end-1) - LY.Fx(2:end-1, 2:end-1)) .* idt .* irry;
  dintegrand_GdTp = gradFx_sq .* irry .* isigma .* (1 / mu0);
  dintegrand_Gdsigma = -2 .* pi .* j_phi .* T_field .* isigma .* isigma - ...
                       gradFx_sq .* Tp_field .* irry .* isigma .* isigma .* (1 / mu0) + ...
                       2 .* pi .* L.rry .* jni_parallel .* isigma .* isigma;
  
  dintegrand_IGdT = idt .* irry(:);
  
  % dFx stuff
  dintegrand_GdFy = dintegrand_GdT(:) .* dTdFy + ...
                    dintegrand_GdTp(:) .* dTpdFy + ...
                    dintegrand_Gdsigma(:) .* dsigmadFy + ...
                    T_field(:) .* irry(:) .* idt;
  dintegrand_GdF0 = dintegrand_GdT(:) .* dTdF0 + ...
                    dintegrand_GdTp(:) .* dTpdF0 + ...
                    dintegrand_Gdsigma(:) .* dsigmadF0;
  dintegrand_GdF1 = dintegrand_GdT(:) .* dTdF1 + ...
                    dintegrand_GdTp(:) .* dTpdF1 + ...
                    dintegrand_Gdsigma(:) .* dsigmadF1;
  dintegrand_Gdag = dintegrand_GdT(:) .* dTdag + ...
                    dintegrand_GdTp(:) .* dTpdag;
  
  dintegrand_IGdFy = dintegrand_IGdT .* dTdFy;
  dintegrand_IGdF0 = dintegrand_IGdT .* dTdF0;
  dintegrand_IGdF1 = dintegrand_IGdT .* dTdF1;
  dintegrand_IGdag = dintegrand_IGdT .* dTdag;
  
  %dFxdot stuff
  dintegrand_GdFydot = T_field(:) .* irry(:);
  dintegrand_IGdFydot = dTdFy .* irry(:);
  dintegrand_IGdF0dot = dTdF0 .* irry(:);
  dintegrand_IGdF1dot = dTdF1 .* irry(:);
  dintegrand_IGdagdot = dTdag .* irry(:);
  
  % gradpsisq part
  dresdgradFysq = Tp_field(:) .* irry(:) .* isigma(:) .* (1 / mu0) .* G(:, igD);
  dresdgradFysq = reshape(dresdgradFysq, L.nzy, L.nry, sum(igD));
  dresdFx = zeros(L.nzx, L.nrx, sum(igD));
  dresdFx(2:end-1, 3:end  , :) = dresdFx(2:end-1, 3:end  , :) + gradr_Fx .* dresdgradFysq .* L.idrx;
  dresdFx(2:end-1, 1:end-2, :) = dresdFx(2:end-1, 1:end-2, :) - gradr_Fx .* dresdgradFysq .* L.idrx;
  dresdFx(3:end  , 2:end-1, :) = dresdFx(3:end  , 2:end-1, :) + gradz_Fx .* dresdgradFysq .* L.idzx;
  dresdFx(1:end-2, 2:end-1, :) = dresdFx(1:end-2, 2:end-1, :) - gradz_Fx .* dresdgradFysq .* L.idzx;
  dresdFx = reshape(dresdFx, L.nx, sum(igD))' * L.dsx ./ scal;
  
  % dresd[] assembly
  dresdFx(:, L.lxy) = dresdFx(:, L.lxy) + ...
                      (dintegrand_GdFy .* G(:, igD) + dintegrand_IGdFy .* IG(:, igD) + ...
                       integrand_G(:) .* dGdFy(:, igD) + integrand_IG(:) .* dIGdFy(:, igD))' * L.dsx ./ scal;
  dresdF0 = (dintegrand_GdF0' * G(:, igD) + dintegrand_IGdF0' * IG(:, igD))' * L.dsx ./ scal;
  % matrix mult per domain for doublets
  dresdF0 = dresdF0 + reshape(sum(integrand_G(:) .* dGdF0(:, igD, :), 1) + ...
                              sum(integrand_IG(:) .* dIGdF0(:, igD, :), 1), sum(igD), L.nD) * L.dsx ./ scal;
  dresdF1 = (dintegrand_GdF1' * G(:, igD) + dintegrand_IGdF1' * IG(:, igD))' * L.dsx ./ scal;
  dresdF1 = dresdF1 + reshape(sum(integrand_G(:) .* dGdF1(:, igD, :), 1) + ...
                              sum(integrand_IG(:) .* dIGdF1(:, igD, :), 1), sum(igD), L.nD) * L.dsx ./ scal;
  dresdag = (dintegrand_Gdag' * G(:, igD) + dintegrand_IGdag' * IG(:, igD))' * L.dsx ./ scal;
  dresdIy = (2 .* pi .* T_field(:) .* isigma(:) .* G(:, igD))' ./ scal; %  * L.idsx * L.dsx
  dresdIe = 0;
  
  dresdIni = 0; % until 1D Ini is supported
  % dresd[]dot assembly
  dresdFxdot = zeros(sum(igD), L.nx);
  dresdFxdot(:, L.lxy) = (dintegrand_GdFydot .* G(:, igD) + dintegrand_IGdFydot .* IG(:, igD))' * L.dsx ./ scal;
  dresdagdot =                                   (dintegrand_IGdagdot' * IG(:, igD))' * L.dsx ./ scal;
  dresdIydot = 0;
  dresdF0dot =                                   (dintegrand_IGdF0dot' * IG(:, igD))' * L.dsx ./ scal;
  dresdF1dot =                                   (dintegrand_IGdF1dot' * IG(:, igD))' * L.dsx ./ scal;
  dresdIedot = 0;
end
end


function [res, dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
               dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
         static_cde_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
% static_cde_1D steady state formulation 1D CDE with parallel conductivity
% function [res,dresdFx, dresdag, dresdIy, dresdF0, dresdF1, dresdIe, dresdIni, ...
%             dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%           static_cde_1D(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD)
%
% Steady state formulation of cde_1D. Does not depend on time
% derivatives. The residual is built around forcing the loop voltage tested
% against each test function g_i to be the same. For
%   int1_i = int 1 / r * T * g_i' dRdZ,
%   int2_i = int 2 pi / sigma * (j_phi - j_ni) * T * g_i' dRdZ + ...
%            int |grad psi|^2 / (sigma * mu0 * r) * T' * g_i' dRdZ,
% the residual given by
%   0 = int1_i int2_{i+1} - int2_i int1_{i+1},   i=1,...,sum(igD)-1
%   0 = (sum(Iy) - LX.Ip) / L.Ip0.
% The gQg are used as test functions, where igD indicates which
% test functions should be used.
%
% For inputs / outputs, see meqcdefun
%
% [+MEQ MatlabEQuilibrium Toolbox+]

% some inits
TDg = logical(L.TDg);
ngD = sum(igD);
assert(ngD>=2, 'static_cde_1D needs more than one degree of freedom. Consider using an Ip constraint instead.');
irry = 1 ./ L.rry;
j_phi = Iy .* L.idsx;

% make sigma array
FNy = reshape(Fx(2:end-1, 2:end-1), L.nzy, L.nry);
FNy = (FNy - reshape(F0, 1, 1, L.nD)) ./ reshape(F1 - F0, 1, 1, L.nD);
act_iDs = find(F0~=F1).';
sigma = zeros(L.nzy, L.nry);
for iD=act_iDs
  sigma = sigma + interp1(L.pQ.^2, LX.signeo(:, iD), FNy(:, :, iD), 'linear', 0) .* (Opy == iD);
end
sigma = sigma + LX.signeo(end, end) .* (Opy == 0); % last value outside plasma
isigma = 1 ./ sigma;

% make jni_parallel array
% This CDE needs a jni profile which is currently not supported in MEQ
% For now, Ini is expected to be 0
if isscalar(LX.Ini)
  jni_parallel = 0;
  if LX.Ini ~= 0
    warning('cde_OhmTor_1D only works with 1D non-inductive current')
  end
else
  jni_parallel = zeros(L.nzy, L.nry);
  for iD=act_iDs
    jni_parallel = jni_parallel + interp1(L.pQ.^2, LX.Ini(:, iD), FNy(:, :, iD), ...
                     'linear', 0) .* (Opy == iD);
  end
end

% mask of basis function supports
Opy_ag = false(L.ny, L.ng);
for iD=1:L.nD
  Opy_ag(:, TDg(iD, :)) = Opy_ag(:, TDg(iD, :)) | (Opy(:) == iD);
end
% get basis function fields
[G, IG] = L.bfct(91, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
G = G .* Opy_ag; IG = IG .* Opy_ag;

[~,aTTpg,~,ahqTg] = L.bfct(3,L.bfp,ag,zeros(L.nD, 1),ones(L.nD, 1),L.fPg,L.fTg,L.idsx);

% T^2 and TT' fields
Tsq_field = IG * ahqTg;
TTp_field = G * aTTpg;

% get T and T' fields
T_field = sqrt(2 * reshape(Tsq_field, L.nzy, L.nry) + LX.rBt.^2) .* sign(LX.rBt);
TTp_field = reshape(TTp_field, L.nzy, L.nry);
Tp_field = TTp_field ./ T_field;

% stuff being tested against derivative of test function
integrand_G = 2 .* pi .* j_phi .* T_field .* isigma;
integrand_G = integrand_G - 2 .* pi .* L.rry .* jni_parallel .* isigma;

% grad psi squared part
gradr_Fx = 0.5 .* (Fx(2:end-1, 3:end) - Fx(2:end-1, 1:end-2)) .* L.idrx;
gradz_Fx = 0.5 .* (Fx(3:end, 2:end-1) - Fx(1:end-2, 2:end-1)) .* L.idzx;
gradFx_sq = gradr_Fx .* gradr_Fx + gradz_Fx .* gradz_Fx;

integrand_G = integrand_G + gradFx_sq .* Tp_field .* isigma .* irry .* (1 / mu0);

integrand_G = reshape(integrand_G, 1, L.ny);
integrand_G_Vloop = reshape(T_field .* irry, 1, L.ny);

integrals1 = (integrand_G       * G(:, igD))' * L.dsx;
integrals2 = (integrand_G_Vloop * G(:, igD))' * L.dsx;

res = zeros(ngD, 1);
scal = L.intG0(igD); scal = scal(1:end-1) .* scal(2:end);
res(2:end) = (integrals1(2:end) .* integrals2(1:end-1) - integrals2(2:end) .* integrals1(1:end-1)) ./ scal;
res(1) = (sum(Iy(:)) - LX.Ip) / L.Ip0;

if nargout >= 2
  % G, IG derivatives
  [dGdFy,dGdF0,dGdF1,dIGdFy,dIGdF0,dIGdF1] = L.bfct(92, L.bfp, Fx(2:end-1, 2:end-1), F0, F1);
  dGdFy = dGdFy .* Opy_ag; dGdF0 = dGdF0 .* Opy_ag; dGdF1 = dGdF1 .* Opy_ag;
  dIGdFy = dIGdFy .* Opy_ag; dIGdF0 = dIGdF0 .* Opy_ag; dIGdF1 = dIGdF1 .* Opy_ag;
  
  % aTTpg, ahqTg jacobians, bfct 3 mode is linear in ag coefficients
  [~,daTTpgdag,~,dahqTgdag] = L.bfct(3,L.bfp,ones(L.ng, 1),zeros(L.nD, 1),ones(L.nD, 1),L.fPg,L.fTg,L.idsx);
  
  % Tsq, TTp jacobians
  dTsqdFy = dIGdFy * ahqTg;
  dTsqdF0 = reshape(sum(dIGdF0 .* ahqTg', 2), L.ny, L.nD); % matrix mult in second dimension for doublets
  dTsqdF1 = reshape(sum(dIGdF1 .* ahqTg', 2), L.ny, L.nD);
  dTsqdag = IG .* dahqTgdag';
  dTTpdFy = dGdFy * aTTpg;
  dTTpdF0 = reshape(sum(dGdF0 .* aTTpg', 2), L.ny, L.nD);
  dTTpdF1 = reshape(sum(dGdF1 .* aTTpg', 2), L.ny, L.nD);
  dTTpdag = G .* daTTpgdag';
  
  % T, Tp jacobians
  iT_field = 1 ./ T_field(:);
  dTdFy = dTsqdFy .* iT_field;
  dTdF0 = dTsqdF0 .* iT_field;
  dTdF1 = dTsqdF1 .* iT_field;
  dTdag = dTsqdag .* iT_field;
  iT_field_cubed = iT_field .* iT_field .* iT_field;
  dTpdFy = dTTpdFy .* iT_field - TTp_field(:) .* dTsqdFy .* iT_field_cubed;
  dTpdF0 = dTTpdF0 .* iT_field - TTp_field(:) .* dTsqdF0 .* iT_field_cubed;
  dTpdF1 = dTTpdF1 .* iT_field - TTp_field(:) .* dTsqdF1 .* iT_field_cubed;
  dTpdag = dTTpdag .* iT_field - TTp_field(:) .* dTsqdag .* iT_field_cubed;
  
  % sigma field jacobian
  dsigmadFy = zeros(L.nzy, L.nry);
  dsigmadF0 = zeros(L.nzy, L.nry, L.nD);
  dsigmadF1 = zeros(L.nzy, L.nry, L.nD);
  for iD=act_iDs
    % using stepped difference as approximation of interpolation derivative
    signeo_grads = diff(LX.signeo(:, iD)) ./ diff(L.pQ.^2).';
    dsdFy_iD = interp1(L.pQ.^2, [signeo_grads; 0], FNy(:, :, iD), 'previous', 0) .* (Opy == iD);
    dsigmadFy = dsigmadFy + dsdFy_iD ./ (F1(iD) - F0(iD));
    dsigmadF0(:, :, iD) = dsdFy_iD .* (Fx(2:end-1, 2:end-1) - F1(iD)) ./ (F1(iD) - F0(iD)).^2;
    dsigmadF1(:, :, iD) = dsdFy_iD .* (F0(iD) - Fx(2:end-1, 2:end-1)) ./ (F1(iD) - F0(iD)).^2;
  end
  dsigmadFy = reshape(dsigmadFy, L.ny, 1);
  dsigmadF0 = reshape(dsigmadF0, L.ny, L.nD);
  dsigmadF1 = reshape(dsigmadF1, L.ny, L.nD);
  
  % more dFx stuff
  dintegrand_GdT = 2 .* pi .* j_phi .* isigma;
  dintegrand_GdTp = gradFx_sq .* irry .* isigma .* (1 / mu0);
  dintegrand_Gdsigma = -2 .* pi .* j_phi .* T_field .* isigma .* isigma - ...
                       gradFx_sq .* Tp_field .* irry .* isigma .* isigma .* (1 / mu0) + ...
                       2 .* pi .* L.rry .* jni_parallel .* isigma .* isigma;
  
  % dFx stuff
  dintegrand_GdFy = dintegrand_GdT(:) .* dTdFy + ...
                    dintegrand_GdTp(:) .* dTpdFy + ...
                    dintegrand_Gdsigma(:) .* dsigmadFy;
  dintegrand_GdF0 = dintegrand_GdT(:) .* dTdF0 + ...
                    dintegrand_GdTp(:) .* dTpdF0 + ...
                    dintegrand_Gdsigma(:) .* dsigmadF0;
  dintegrand_GdF1 = dintegrand_GdT(:) .* dTdF1 + ...
                    dintegrand_GdTp(:) .* dTpdF1 + ...
                    dintegrand_Gdsigma(:) .* dsigmadF1;
  dintegrand_Gdag = dintegrand_GdT(:) .* dTdag + ...
                    dintegrand_GdTp(:) .* dTpdag;
  
  % gradpsisq part
  dresdgradFysq = Tp_field(:) .* irry(:) .* isigma(:) .* (1 / mu0) .* G(:, igD);
  dresdgradFysq = reshape(dresdgradFysq, L.nzy, L.nry, ngD);
  dint1dFx = zeros(L.nzx, L.nrx, ngD);
  dint1dFx(2:end-1, 3:end  , :) = dint1dFx(2:end-1, 3:end  , :) + gradr_Fx .* dresdgradFysq .* L.idrx;
  dint1dFx(2:end-1, 1:end-2, :) = dint1dFx(2:end-1, 1:end-2, :) - gradr_Fx .* dresdgradFysq .* L.idrx;
  dint1dFx(3:end  , 2:end-1, :) = dint1dFx(3:end  , 2:end-1, :) + gradz_Fx .* dresdgradFysq .* L.idzx;
  dint1dFx(1:end-2, 2:end-1, :) = dint1dFx(1:end-2, 2:end-1, :) - gradz_Fx .* dresdgradFysq .* L.idzx;
  dint1dFx = reshape(dint1dFx, L.nx, ngD)' * L.dsx;
  
  % dint1d[] assembly
  dint1dFx(:, L.lxy) = dint1dFx(:, L.lxy) + ...
                      (dintegrand_GdFy .* G(:, igD) + integrand_G(:) .* dGdFy(:, igD))' * L.dsx;
  dint1dF0 = (dintegrand_GdF0' * G(:, igD))' * L.dsx;
  dint1dF0 = dint1dF0 + reshape(sum(integrand_G(:) .* dGdF0(:, igD, :), 1), ngD, L.nD) * L.dsx;
  dint1dF1 = (dintegrand_GdF1' * G(:, igD))' * L.dsx;
  dint1dF1 = dint1dF1 + reshape(sum(integrand_G(:) .* dGdF1(:, igD, :), 1), ngD, L.nD) * L.dsx;
  dint1dag = (dintegrand_Gdag' * G(:, igD))' * L.dsx;
  dint1dIy = (2 .* pi .* T_field(:) .* isigma(:) .* G(:, igD))'; %  * L.idsx * L.dsx
  
  % dint2d[] assembly
  dint2dFx = zeros(ngD, L.nx);
  dint2dFx(:, L.lxy) = (dTdFy .* irry(:) .* G(:, igD) + integrand_G_Vloop(:) .* dGdFy(:, igD))' * L.dsx;
  dint2dF0 = reshape(sum(integrand_G_Vloop(:) .* dGdF0(:, igD, :), 1), ngD, L.nD) * L.dsx + ...
             G(:, igD)' * (dTdF0 .* irry(:)) * L.dsx;
  dint2dF1 = reshape(sum(integrand_G_Vloop(:) .* dGdF1(:, igD, :), 1), ngD, L.nD) * L.dsx + ...
             G(:, igD)' * (dTdF1 .* irry(:)) * L.dsx;
  dint2dag = G(:, igD)' * (dTdag .* irry(:)) * L.dsx;
  
  % jacobian inits
  dresdFx = zeros(ngD, L.nx);
  dresdF0 = zeros(ngD, L.nD);
  dresdF1 = zeros(ngD, L.nD);
  dresdag = zeros(ngD, L.ng);
  dresdIy = zeros(ngD, L.ny);
  dresdIe = zeros(ngD, L.ne);
  dresdIni = 0; % until 1D Ini is supported
  % all dxdot terms are zero
  dresdFxdot = 0;
  dresdagdot = 0;
  dresdIydot = 0;
  dresdF0dot = 0;
  dresdF1dot = 0;
  dresdIedot = 0;
  
  % fill out jacobians from product rule
  dresdFx(2:end, :) = (dint1dFx(2:end, :) .* integrals2(1:end-1) - integrals2(2:end) .* dint1dFx(1:end-1, :) + ...
                       integrals1(2:end) .* dint2dFx(1:end-1, :) - dint2dFx(2:end, :) .* integrals1(1:end-1)) ./ scal;
  dresdF0(2:end, :) = (dint1dF0(2:end, :) .* integrals2(1:end-1) - integrals2(2:end) .* dint1dF0(1:end-1, :) + ...
                       integrals1(2:end) .* dint2dF0(1:end-1, :) - dint2dF0(2:end, :) .* integrals1(1:end-1)) ./ scal;
  dresdF1(2:end, :) = (dint1dF1(2:end, :) .* integrals2(1:end-1) - integrals2(2:end) .* dint1dF1(1:end-1, :) + ...
                       integrals1(2:end) .* dint2dF1(1:end-1, :) - dint2dF1(2:end, :) .* integrals1(1:end-1)) ./ scal;
  dresdag(2:end, :) = (dint1dag(2:end, :) .* integrals2(1:end-1) - integrals2(2:end) .* dint1dag(1:end-1, :) + ...
                       integrals1(2:end) .* dint2dag(1:end-1, :) - dint2dag(2:end, :) .* integrals1(1:end-1)) ./ scal;
  dresdIy(2:end, :) = (dint1dIy(2:end, :) .* integrals2(1:end-1) - integrals2(2:end) .* dint1dIy(1:end-1, :)) ./ scal;
  dresdIy(1,     :) = ones(1, L.ny) / L.Ip0;
end
end
