function [cdec, icde, np] = meqcdec(cde, agconc, fTg, TDg, nD, isEvolutive)
% MEQCDEC Parser for CDE specifications
% [cdec, icde, np] = meqcdec(cde, agconc, fTg, TDg, nD)
% takes in cde-string indicating which CDE to use and adds it for each
% domain as long as a suitable number of degrees of freedom are free
% the cde input string needs to contain '0D' or '1D' as keyword to decide
% if it constrains 1 or more degrees of freedom
% Arguments:
%   cde:    string, string indicating which cde to use, see meqcdefun for
%           available equations
%   agconc: cell(m, 4), cell array output of meqagconc indicating used ag
%           constraints
%   fTg:    array, mask array indicating which ag entries belong to the T
%           profile
%   TDg:    array, matrix indicating which ag entries belong to which domain
%   nD:     int, number of plasma domains
%   isEvolutive: bool, flag indicating if only static CDEs are considered
% returns:
%   cdec:   cell(k, 4), cell array describing which cde residual should be
%           used for which domain and how many degrees of freedom it constrains
%           each entry has the form
%           {fun_handle, domain_index, cde_name, num DOFs constrained}
%           for each domain, the number of DOFs constrained is either 1 for
%           0D CDEs or more for 1D CDEs
%   icde:   bool(1, nD), mask indicating which domain uses a CDE
%   np:     int, total number of degrees of freedom constrained by CDEs
%           in the end, it has to hold nC + np = ng
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% inits
S = meqcdefun(); % get catalog of functions
cdec = cell(0, 4);
icde = false(1, nD);
np = 0;

% no cde needed if empty
if isempty(cde)
  return;
elseif ~contains(cde, 'static') && ~isEvolutive
  % error when trying to run fgs with non-static CDE
  error('FGS cannot be run with a non-static CDE')
elseif contains(cde, 'static') && isEvolutive
  % warning when static cde is used in fge
  warning('Using static CDE for plasma evolution. Plasma response to Vloop will not be modeled correcly.')
end

% check that cde is implemented in S
if ~isfield(S, cde)
  error('CDE %s not found in meqcdefun.', cde)
end

if endsWith(cde, '0D')
  % this identifies how many degrees of freedom are missing for each domain
  missing_DOFs = missing_DOFs_per_domain(TDg, agconc);
  for iD=1:nD
    if missing_DOFs(iD) == 0 % it's okay when there is no cde for one domain
      continue;
    end
    assert(missing_DOFs(iD) == 1, 'CDE %s with 1 degree of freedom cannot be matched with %d free degrees of freedom in domain %i',cde,missing_DOFs(iD),iD)

    cdec = [cdec; {S.(cde), iD, cde, 1}];
    icde(iD) = true;
    np = np + 1;
  end
elseif endsWith(cde, '1D')
  % 1D cdes are global on all missing DOFs
  % need to get missing DOFs that are not constrained by ag constraints
  ag_constrained_dofs = [agconc{strcmp(agconc(:, 3), 'ag'), 4}];
  
  cde_dofs = logical(fTg);
  cde_dofs(ag_constrained_dofs) = 0;
  
  np = sum(cde_dofs);
  cdec = [cdec; {S.(cde), 0, cde, cde_dofs}];
  icde(:) = true;
else
  error('CDE signature must end with either "0D" or "1D" keyword.')
end

% at least some CDEs should have been added
if np == 0
  error('No domain has free degrees of freedom to add CDE');
end
end


function missing_DOFs = missing_DOFs_per_domain(TDg, agconc)
  [nD, ng] = size(TDg);  
  % identify direct "ag" constraint DOFs
  isag = strcmp(agconc(:,3), 'ag');
  ag_constrained_DOFs = [agconc{isag, 4}];
  
  % count non-"ag" constraints per domain
  agcon_domains = [agconc{~isag, 2}];
  constraints_per_domain = sum(agcon_domains(:) == 1:nD, 1);
  
  % restrict TDg to basis functions not controlled by "ag" constraints
  TDg_reduced = TDg(:, setdiff(1:ng, ag_constrained_DOFs)); 
  
  % this assigns each domain the basis functions which are not also acting
  % in another domain that encompasses the former
  ng_reduced = size(TDg_reduced, 2);
  Tig_unique = zeros(ng_reduced, 1);
  for ig=1:ng_reduced
    Tig_unique(ig) = find(TDg_reduced(:, ig), 1, 'last');
  end
  
  % counting how many basis functions are assigned to each domain and taking difference
  DOFs_per_domain = sum(Tig_unique == 1:nD, 1);
  missing_DOFs = DOFs_per_domain - constraints_per_domain;
end
