function [res,dresdFx,dresdag,dresdIy,dresdF0,dresdF1,dresdIe,dresdIni, ...
          dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
              meqcde(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy)
% MEQCDE CDE residual function
%[res,dresdFx,dresdag,dresdIy,dresdF0,dresdF1,dresdIe,dresdIni, ...
% dresdFxdot,dresdagdot,dresdIydot,dresdF0dot,dresdF1dot,dresdIedot] = ...
%     meqcde(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy)
% Takes in quantities of current state and outputs CDE residual and
% optional derivatives of this residual. If nD > 1, the residual can be
% multi-dimensional
% Arguments:
%   L:    struct, standard L structure
%   LX:   struct, standard LX structure used in fge solve
%   LY:   struct, LY structure of solution from previous timestep
%   Fx:   double(nzx, nrx), poloidal flux field
%   ag:   double(ng, 1), vector containing ag coefficients
%   Iy:   double(nzy, nry), plasma current map
%   F0:   double(1, nD), value of flux at magnetic axis for each domain
%   F1:   double(1, nD), value of flux at seperatrix for each domain
%   Ie:   double(ne, 1), vector containing external currents
%   Opy:  int8(nzy, nry), domain segmentation mask
% returns:
%   res:      double(np, 1), residual from the CDEs
%   dresdFx:  double(np, nx), jacobian of residual w.r.t. flux map
%   dresdag:  double(np, ng), jacobian of residual w.r.t. ag coeffs
%   dresdIy:  double(np, ny), jacobian of residual w.r.t. Iy
%   dresdF0:  double(np, nD), jacobian of residual w.r.t. flux at axis
%   dresdF1:  double(np, nD), jacobian of residual w.r.t. flux at seperatrix
%   dresdIe:  double(np, ne), jacobian of residual w.r.t. external currents
%   dresdIni: double(np, nD), jacobian of residual w.r.t. non-inductive currents
%   dresdFxdot: double(np, nx), jacobian of residual w.r.t. change in Fx
%   dresdagdot: double(np, ng), jacobian of residual w.r.t. change in ag
%   dresdIydot: double(np, ny), jacobian of residual w.r.t. change in Iy
%   dresdF0dot: double(np, nD), jacobian of residual w.r.t. change in F0
%   dresdF1dot: double(np, nD), jacobian of residual w.r.t. change in F1
%   dresdIedot: double(np, ne), jacobian of residual w.r.t. change in Ie
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% if jacobians are needed
dojac = nargout > 1;

% init
res = zeros(L.np, 1);
if dojac
  dresdFx = zeros(L.np, L.nx);
  dresdag = zeros(L.np, L.ng);
  dresdIy = zeros(L.np, L.ny);
  dresdF0 = zeros(L.np, L.nD);
  dresdF1 = zeros(L.np, L.nD);
  dresdIe = zeros(L.np, L.ne);
  
  dresdIni = zeros(L.np, L.nD);
  
  dresdFxdot = zeros(L.np, L.nx);
  dresdagdot = zeros(L.np, L.ng);
  dresdIydot = zeros(L.np, L.ny);
  dresdF0dot = zeros(L.np, L.nD);
  dresdF1dot = zeros(L.np, L.nD);
  dresdIedot = zeros(L.np, L.ne);
end

if L.np==0
  return
end

% if there is no dt (e.g. for the linearization), just set idt = 0
dt = LX.t - LY.t;
if dt == 0
  idt = 0;
else
  idt = 1./dt;
end

% active ag coefficients are all ag coefficients that don not control a
% basis function inside an inactive domain (F0==F1)
active_ags = ~any(L.TDg(F0 == F1, :), 1).';
% inactive ag coeffs that need to be replaced are cde specific
% they will later be filled in by ag=0 constraints
excluded_ags = false(L.ng, 1);

% very similar loop to the one in agconc
krr = 0;
for ii=1:size(L.cdec, 1)
  % get the domain index of cde
  % domain index = 0 means the cde acts globally
  iD = L.cdec{ii, 2};
  cde_func = L.cdec{ii, 1};
  
  if iD == 0 % global 1D cde
    % For 1D cde, L.cdec{ii, 4} indicates the basis functions used as test
    % functions. If domain for these basis functions is inactive, they are
    % not used as test functions anymore
    dof_info = L.cdec{ii, 4} & active_ags;
    % jacobian of 1D cde is for all F0 and all F1
    F01_dep_inds = 1:L.nD;
    
    % all non active degrees of freedom that would be constrained by cde
    excluded_ags = L.cdec{ii, 4} & ~active_ags;
  else % 0D cde
    dof_info = iD;
    % jacobian of 0D cde for domain iD only depends on F0(iD) and F1(iD)
    F01_dep_inds = iD;
    
    if F0(iD) == F1(iD) % inactive domain
      % for 0D cde, the last degree of freedom of an inactive domain needs
      % to be replaced by an ag constraint
      excluded_ags(find(L.TDg(iD, :), 1, 'last')) = 1;
      continue;
    end
  end
  
  % evaluate cde residual for this domain
  if dojac
    [resD, dresdFxD, dresdagD, dresdIyD, dresdF0D, dresdF1D, dresdIeD, dresdIniD, ...
           dresdFxdotD,dresdagdotD,dresdIydotD,dresdF0dotD,dresdF1dotD,dresdIedotD] = ...
      cde_func(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,dof_info);
  else
    resD = cde_func(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,dof_info);
  end
  
  % update reisudal indices
  irr = krr + (1:size(resD,1)); krr = krr + size(resD,1);
  res(irr) = resD;
  
  % jacobian updates if needed
  if dojac
    dresdFx(irr, :) = dresdFxD;
    dresdag(irr, :) = dresdagD;
    dresdIy(irr, :) = dresdIyD;
    dresdF0(irr, F01_dep_inds) = dresdF0D;
    dresdF1(irr, F01_dep_inds) = dresdF1D;
    dresdIe(irr, :) = dresdIeD;
    
    dresdIni(irr, F01_dep_inds) = dresdIniD;
  
    dresdFxdot(irr, :) = dresdFxdotD;
    dresdagdot(irr, :) = dresdagdotD;
    dresdIydot(irr, :) = dresdIydotD;
    dresdF0dot(irr, F01_dep_inds) = dresdF0dotD;
    dresdF1dot(irr, F01_dep_inds) = dresdF1dotD;
    dresdIedot(irr, :) = dresdIedotD;
  end
end

% add ag residuals for inactive ag coeffs
irr = krr + (1:sum(excluded_ags)); krr = krr + sum(excluded_ags);
scali = L.xscal(L.ind.ixg(excluded_ags));
res(irr) = ag(excluded_ags) ./ scali;
dresdag(irr, excluded_ags) = diag(1./scali);

if (krr ~= L.np)
  error('Number of residuals (%d) does not match total number of CDEs (%d)',krr,L.np)
end

end
