function agconc = meqagconc(agcon,TDg,idoublet)
% parser for ag constraint specifications.
%
% Reads constraint specifications from agcon, parses it, and outputs the
% result to be stored in L.agconc. Other inputs include the mapping matrix
% from basis functions to domains TDg and the parameter flag for doublets 
% idoublet.
%
% Specify constraint either as strings (e.g. 'Ip') which correspond to
% functions in meqagconfun.m
%
% Several ways of input:
%
% 1) Simple constraints, e.g. {'Ip','bp','li'}
%    This directly specifies the basis function constraints to be used.
%    For doublets, this defaults to having these constraints for the two
%    lobes, and fixed coefficients for mantle.
%
% 2) Constraints per domain, e.g.
%    {{'Ip','bp','li'},{'Ip','bt','qA'},'{'ag','ag','ag'}}
%    This explicitly specifies the constraints for each domain.
%
% 3) Direct specification in final format e.g.
%    {'Ip',1;'Ip',2,'bp',1,'bp',2,'Ip',3,'bp',1};
%    This specifies the constraint to be used and the corresponding domain.
%    Empty domain means the constraint applies to the whole plasma.
%
% outputs:
%  agconc is a cell array with 4 columns:
%     1) function handle for a given constraint
%     2) domains to which the constraint applies
%     3) corresponding quantity in LX entering into that constraint.
%     4) index to use in this LX quantity
%  This output is used by meqagcon.m to build the basis function constraints
%  
%
% See meqagcon.m, meqagconfun.m
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

S = meqagconfun(); % get catalog of functions
in = agcon;
assert(iscell(in),'agcon must be a cell');

[nD,ng] = size(TDg);
%#ok<*AGROW>
out = cell(0,2); % init
if size(in,2) ==2 && (isnumeric(in{1,2}) || isempty(in{1,2})) 
  % Input mode 3: direct input of cell of constraints
  % basic checks/conversions
  out = in;
  for ii=1:size(in,1)
    switch class(in{ii,1})
      case 'function_handle' % ok keep it
      case 'char',  out{ii,1} = myfun(S,in{ii,1}); % convert to func
      otherwise, error('don''t know how to handle this type')
    end
    nval = in{ii,2};
    assert(isempty(nval) || (isnumeric(nval) && isscalar(nval)),...
      'AGCONPARSE:DomainIndicatorType','domain indicator in second column must be empty or a numeric scalar')
    assert(isempty(nval) || (nval>0 && nval<=nD),...
      'AGCONPARSE:DomainIndicatorValue','domain indicator must be empty or between 0 and nD=%d',nD);
  end
elseif size(in,1)==1 % input a row of constraints
  if ~iscell(in{1}) % mode(1): single constraint set given: e.g. {'Ip','bp','qA'}
    if ~idoublet
      % singlet: only one set of constraints for one domain
      out = filliD(out,in,S,1);
    else
      % doublet: assume same for two lobes, then fix ag for mantle
      ng3 = sum(TDg(3,:)); % Number of basis functions for mantle.
      in = {in,in,repmat({'ag'},1,ng3)};
      for iD=1:numel(in)
        out = filliD(out,in{iD},S,iD);
      end
    end
  else % mode (2) multiple constraints are given
    for iD=1:numel(in)
      out = filliD(out,in{iD},S,iD);
    end
  end
end

% init LXfieldname and LXindex
LXfieldname = cell(0, 1); LXindex = cell(0, 1);
% find corresponding LX field and its index that enters constraint, if any
for icon=1:size(out,1)
  [fieldname] = meqagconq(out,icon,idoublet);
  iD = out{icon,2}; % domain index of this constraint
  switch fieldname
    case 'ag'
      % special case: fixed ag value. Need to find index of correct ag to
      % use for this constraint in meqagcon.
      
      igD = find(TDg(iD,:));                  % Indices of ag active in this domain
      ii = igD(find([out{:,2}]==iD) == icon); % index within constraints with this domain index
    otherwise
      ii = iD; % for other fields, take iDth index (for multidomain)
  end
  
  LXfieldname{icon,1} = fieldname;
  LXindex    {icon,1} = ii;
end

agconc = [out,LXfieldname,LXindex];

end

function out = filliD(out,in,S,iD)
% fill some rows of table for the constraints of this domain
for ii = 1:numel(in)
  out = [out;{myfun(S,in{ii}),iD}];
end
end

function fun = myfun(S,myfunstr)
% find function handle from list
if ~isfield(S,myfunstr)
  error('meqagconc:notfound','Constraint function %s not found, available functions are: %s',myfunstr,strjoin(fieldnames(S),','))
end
fun = S.(myfunstr);
end

function [agconq] = meqagconq(agconc,icon,idoublet)
% return string of field in LX, that enters in constraint function agcon
% with agcon in format {function_handle,domain_index}

qstr = func2str(agconc{icon,1});
iD   = agconc{icon,2};

% fields that can be computed for each separate domain, that have 'D'
% counterpart
fieldsD = {'Ip','Wk','Wp','Wt','Wt0','WN','Vp','Ft','Ft0','bp','bp','Ini','li'};

agconq = qstr;

if idoublet && ~isempty(iD) && contains(qstr,fieldsD)
  agconq = strcat(agconq,'D');   % fields like 'IpD'
end

end
