function [res, dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg,...
          dresdrA, dresddr2FA, dresddz2FA, dresddrzFA, dresdCo] = ...
          meqagcon(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg)
% MEQAGCON Compute residuals for basis function coefficient constraints in MEQ
%
% [res, dresdF0,dresdF1,dresdag,dresdFx,dresdTpDg,dresdITpDg,...
%  dresdrA, dresddr2FA, dresddz2FA, dresddrzFA, dresdCo] = ...
%    meqagcon(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg)
%
% Evaluates residual vector based on constraint specifications in L.agconc
%
% If nargout > 1, all Jacobians with respect to the continuously varying
% input quantities are also returned.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

nA = numel(rA);
nB = sum(F1 ~= F0);
irr = 0; % equation row index
res = zeros(L.nC,1);

% initialize all zero jacobians
dojac = nargout > 1;
if dojac
  dresdF0 = zeros(L.nC, L.nD); dresdF1 = dresdF0;
  dresdag = zeros(L.nC, L.ng);
  dresdTpDg = zeros(L.nC,L.nD*L.ng); dresdITpDg = dresdTpDg;
  dresdFx = zeros(L.nC, L.nx);
  dresdrA = zeros(L.nC, nA);
  dresddr2FA = dresdrA; dresddz2FA = dresdrA; dresddrzFA = dresdrA;
  dresdCo = zeros(L.nC, L.nC);
end

% for the Ipmin case, all domains are treated as inactive and res = ag
if abs(LX.Ip) < L.P.Ipmin
  res = ag./L.xscal(L.ind.ixg);
  res = res(1:L.nC);
  if dojac
    dresdag = diag(1./L.xscal(L.ind.ixg));
    dresdag = dresdag(1:L.nC,:); % Only keep first nC rows
  end
  return
end

for iC = 1:L.nC % for all constraints
  f = L.agconc{iC,1}; iD = L.agconc{iC,2};
  ii = L.agconc{iC,4}; % index for this constraint

  if iD>nB
    % Non-active domains will be treated after the loop
    irr = irr(end)+1;
    continue;
  end

  % call constraint function
  if nargout > 1
    [res_,dresdF0_,dresdF1_,dresdag_,dresdFx_,dresdTpDg_, ...
      dresdITpDg_,dresdrA_,dresddr2FA_,dresddz2FA_,dresddrzFA_,dresdCo_] = ...
      f(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,ii);
  else
    [res_] = ...
      f(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg,ii);
  end
  
  if L.P.debug>5
    fprintf('evaluated constraint %d: %s, size=%d, ires=%d, iD=%d, ii = %d, res=%+5.3e\n',...
      iC,func2str(f),size(res_,1),irr(end),iD,ii,res_);
  end
  
  % indices
  irr  = irr(end) + (1:(size(res_,1)));
  
  % build full array
  res(irr) = res_;
  if dojac
    dresdF0   (irr, : ) = dresdF0_;
    dresdF1   (irr, : ) = dresdF1_;
    dresdag   (irr, : ) = dresdag_;
    dresdFx   (irr, : ) = dresdFx_;
    dresdTpDg (irr, : ) = dresdTpDg_;
    dresdITpDg(irr, : ) = dresdITpDg_;
    dresdrA   (irr, : ) = dresdrA_;
    dresddr2FA(irr, : ) = dresddr2FA_;
    dresddz2FA(irr, : ) = dresddz2FA_;
    dresddrzFA(irr, : ) = dresddrzFA_;
    dresdCo   (irr, iC) = dresdCo_;
  end
end

if irr(end) ~= (L.nC)
  error('Number of residuals (%d) does not match number of ag constraints (%d)',irr(end),L.nC)
end

% Inactive domains
if nB<L.nD
  % Identify residuals associated to each domain
  iCi = [L.agconc{:,2}].'>nB; % Constraints for these domains
  nCi = sum(iCi);
  if nCi>0
    % Identify which bfs to set to 0
    igi = any(L.TDg(nB+1:end,:),1); % basis functions associated with these domains
    igi(cumsum(igi)>nCi) = false; % Only keep the first nCi
    % Set ag(ig) = 0
    scali = L.xscal(L.ind.ixg(igi));
    res(iCi) = ag(igi)./scali;
    if dojac
      % Only assign non-zero values
      dresdag(iCi,igi) = diag(1./scali);
    end
  end
end
end
