function Fx = meqFx(L,IyD,Ie,Fb_ext,dz,rst,Jh)
%MEQFX Evaluate flux (and optionnaly its derivatives) for given Iy, and optionally Ie
% FX = MEQFX(L,IYD[,IE,FB_EXT,DZ,RST,JH]) returns the
% solution of the poisson equation for given current density parametrization. 
% The L structure contains MEQ ancillary data, IYD contains the breakdown of 
% the plasma current per plasma domain , IE is the current in the external 
% conductors, assumed zero if not passed.
% FB_EXT is an optional argument with precomputed contributions from
% external currents.
% The last 3 arguments are optional and are typically used only in LIU. DZ
% is the vector of shifts of IY with respect to FX for each domain, if RST
% is true then JH contains the current in each FE for the LIU initial
% guess.
%
% The parameter L.P.gsxe determins how the computation is done
%  - if gsxe=1, then the boundary flux Fb is determined using meqfbp(Iy) and
%  Fbe*Ie then Iy is modified to include filaments from external currents
%  within the computational domain and the poisson equation is solved using
%  gszr with this modified Iy and the previously determined Fb (so the Iys
%  going into meqfbp and gszr are different)
%  - if gsxe=2, then the filaments from external currents within the
%  computational domain are excluded from the computation of Fb and their
%  contribution to the poisson solution is determined solely by analytical
%  green functions
%  - if gsxe=3, then the full solution is determined by analytical green
%  functions
%
% Note: outside of LIU or if L.ndz<=1 then IY can be used in place of IYD.
%
% See also MEQFXJAC
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if nargin<3
  Ie = zeros(L.ne,1);
end

nb = size(Ie,2);

% Flag to compute external current contribution
hasIe = any(Ie(:));
 
% Flag to compute plasma contribution
hasIy = any(IyD(:));

if ~(hasIe || hasIy)
  Fx = zeros(L.nzx,L.nrx,nb);
  return
end

% External current contributions
if hasIe
  if L.P.gsxe == 1 || L.P.gsxe == 2
    % Only gsxe=1/2 use Fbe/Iyie
    if nargin < 4 || isempty(Fb_ext)
      Fbe = L.Mbe*Ie;
      Iyie = reshape(L.Tye*Ie,L.nzy,L.nry,nb);
    else
      Fbe  = Fb_ext{1};
      Iyie = Fb_ext{2};
    end
  end
end

% Check size of Iy when multiple solutions are sought
if nb>1 && size(IyD,3) ~= nb
  error('meqFx:incompatibleInputs','size(Ie,2)=%d but size(IyD,3)=%d',nb,size(IyD,3));
end

% Effective number of domains
if nargin<7 || rst || ~any(dz)
  if nargin<7, rst = false;end
  if (rst && any(dz))
    error('meqFx:incompatibleInputs','If rst is true, all dzs should be 0');
  end
  dz = 0;
  dzMbe = zeros(1,L.ne);
  ndzeff = 1;
  if nb == 1 && size(IyD,3)>1
    IyD(:,:,1) = sum(IyD,3);
  end
else
  ndzeff = find(any(any(IyD,1),2),1,'last');
  if isempty(ndzeff), ndzeff = 1; end % No plasma current
  % Checks
  if any(dz) && nb>1
    error('meqFx:incompatibleInputs','Cannot compute multiple meqFx solutions with dz enabled');
  end
  if ndzeff > numel(dz)
    error('meqFx:incompatibleInputs','numel(dz)=%d, but ndzeff=%d',numel(dz),ndzeff);
  end
  dzMbe = L.dzMbe;
end

% Boundary flux
if nb==1
  Iy = IyD(:,:,1); % First domain
else
  Iy = IyD;
end
if L.P.gsxe < 3 % Fb not needed for gsxe=3
  if hasIy
    if rst % from FE
      Fb = L.Mbh*Jh;
    else % from Iy, as sheet current equivalent to normal flux derivative
      Fb = meqfbp(Iy,L); % boundary current contribution from plasma - Lackner's trick!
    end
  else
    Fb = zeros(L.nx-L.ny,1);
  end
end

% Full solution
switch L.P.gsxe
  case {1,2} % 1: Include external currents inside the grid in Iy - LIUQE mode
             % 2: Treat external currents inside the grid exactly - FBT mode
    if hasIe
      Fb = Fb +  Fbe;                           % Add effect of external currents and dz
      if dz(1), Fb = Fb - dz(1)*(dzMbe*Ie); end % Note: The mapping from e to y does not take dz into account
      Iy = Iy + Iyie;                           % Add external currents on computational grid
    end
    Fx = gszrmex(Fb,Iy,L.cx,L.cq,L.cr,L.cs,L.ci,L.co,L.idzx*dz(1));
  case 3 % Direct calculation of Fx from Green's functions (no gszrmex - slower)
    Fx = reshape(L.Mxy*reshape(Iy,L.ny,[]),L.nzx,L.nrx);
    if dz(1)
      dFxdzD = (Fx([2:end end],:,:)-Fx([1 1:end-1],:,:)).*[1;repmat(0.5,L.nzx-2,1);1]*L.idzx;
      Fx     =  Fx + dFxdzD*dz(1);
    end
    Fx =  Fx + reshape(L.Mxe*Ie,L.nzx,L.nrx,nb);
  otherwise
    error('meqFx:invalidmode','gsxe=%d is invalid',L.P.gsxe);
end

% Other domains (no external currents) - here nb==1 is assumed
for iD = 2:ndzeff
  Iy = IyD(:,:,iD);
  if L.P.gsxe < 3
    Fb = meqfbp(Iy,L);
    Fx = Fx + gszrmex(Fb,Iy,L.cx,L.cq,L.cr,L.cs,L.ci,L.co,L.idzx*dz(iD)); % Solve GS
  else
    FxD = reshape(L.Mxy*Iy(:),L.nzx,L.nrx); % new flux from this IyD
    if dz(iD)
      dFxdzD = (FxD([2:end end],:)-FxD([1 1:end-1],:)).*[1;repmat(0.5,L.nzx-2,1);1]*L.idzx;
      FxD = FxD + dFxdzD*dz(iD);
    end
    Fx = Fx + FxD;
  end
end
end
