function time = liuqe_times(shot,type,extra)
%LIUQE_TIMES  Determine time array for LIUQE anasrv call
%
% TIME = LIUQE_TIMES(SHOT[,TYPE,EXTRA])
%
% LIUQE_TIMES defines the time arrays for LIUQE reconstruction according 
% to TYPE. TIME is the list of times for which both LIUQE and the PSITBX
% will be run. EXTRA is an array containing extra times to be added
% to the default selection.
%
% TYPE can currently be either:
%    - refined: equally spaced points between start and end of shot with 1/1000 s separation
%    - thomson: use \results::thomson:times if not empty, otherwise 
%               equally spaced points between start and end of shot with 1/60 s separation
%    - mixed: union of fast and thomson [default]
%    - magnetics: all data points from the magnetics in the [0,3s] window
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

assert(nargin > 0,'First argument (shot) is not specified');
if nargin < 3, extra  =      [];end
if nargin < 2, type   = 'mixed';end
type = validatestring(lower(type) ,{'refined','thomson','mixed'},mfilename,'type');

% Open shot
meqmdsopen(shot,'tcv_shot');

% Select interval
tmin = 0;
[td,st] = mdsvalue('time_disrupted()');
assert(isodd(st),'Could not retrieve time of disruption for shot %05d',shot);
switch td
 case 512
  warning('TCV shot %05d was aborted and has no useful data, LIUQE time selection will be empty',shot);
  tmax = -Inf;
 case 256
  warning('TCV shot %05d was a PCS/Stray shot, LIUQE time selection will be empty',shot);
  tmax = -Inf;
 case 128
  [tf,st] = mdsvalue('fbte_time("FBTE")');
  assert(isodd(st),'Could not retrieve FBTE times for shot %05d',shot);
  tmax = tf(end) + 0.050; % Add 50ms after planned end of shot
 otherwise
  tmax = td + 0.050; % Add 50ms after time of disruption
end

mag_times = mdsvalue('\magnetics::bpol_avg:dim0').';

if strcmp(type,'magnetics')
 mask = mag_times >= tmin & mag_times <= tmax;
 time = mag_times(mask);
else
 has_refined = true;
 has_thomson = true;
 switch lower(type(1))
  case 'r'
   has_thomson = false;
  case 't'
   has_refined = false;
 end
 standard_times = (floor(tmin*60):ceil(tmax*60))/60;

 refined_times = [];
 if has_refined, refined_times = (floor(tmin*1000):ceil(tmax*1000))/1000;end

 thomson_times = [];
 if has_thomson
  % Get list of Thomson times ...
  [thomson_times,status] = mdsvalue('\results::thomson:times');
  if ~isodd(status), thomson_times = [];end
  % ... within interval of interest
  mask = thomson_times>=tmin & thomson_times<=tmax;
  thomson_times = thomson_times(mask);
  % Convert to row vector
  thomson_times = thomson_times(:).';
 end

 % Construct full time array
 time = unique([thomson_times,refined_times,extra(:).']);

 % If time_psi is empty, replace with the standard times (can only happen
 % for type_psi='thomson' and \results::thomson:times is empty)
 if isempty(time), time = standard_times;end

 % Round them to the nearest magnetics time
 k = iround(mag_times,time);
 k(diff(k) == 0) = []; % Remove duplicates (vector is already sorted).
 time = mag_times(k);
end

end