function [Rax,Rau,Raz,Ra0,...
          Rex,Reu,Rez,Re0,...
          Rix,Riu,Riz,Ri0,ID,w] = liuopt(L,LX)
% LIUOPT Compute LIU optimization problem for single time slice
%
% The FBT optimization problem aims at finding [x,u,Fb] that minimizes the
% cost function
%   W(x,u,dz) = 1/2*|Rax*x + Rau*u + Raz*dz - Ra0|^2
% under the exact constraints
%   Rex*x + Reu*u + Rez*dz - Re0 = 0
% and linear inequality constraints
%   Rix*x + Riu*u + Riz*dz - Ri0 <= 0
% x is the plasma state variable, u is a set of external inputs and dz is
% a fitting parameter. Additionally x and u must verify the FGS or FGE set
% of equations F(x,u)=0.
%
% The external inputs are [Ia;Iu;Co].
%
% Additional outputs include:
%   ID  ID per cost function term (see LIUC.M and L.dimID)
%   w   Weight for each cost function term
%
% SEE ALSO LIULSQINIT, FBTOPT, MEQOPT
% 
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

Ip = LX.Ip;
rBt = LX.rBt;

XCo = LX2Co(L,LX); % Co value in LX

Xr = [LX.Ff; LX.Bm];         % magnetic measurements
Xe = [LX.Ia; LX.Iu];         % external currents
Xi = [LX.Xt*rBt ; Ip];       % integral measurements
Xj = [XCo; LX.dz];           % constraints + dz

Yr = L.Wr.*Xr;
Ye = L.We.*Xe;
Yi = L.Wi.*Xi;
Yq = zeros(L.nq,1);          % Regularisation
Yj = L.Wj.*Xj;               % 0 for exact constraints (see liuc)

% Direct ag/dz constraints
Mjj = eye(L.nj); % equality constraint scaling is handled in LSQ solver so this is ok
% Direct Ie constraints
Mee = eye(L.ne);

WjC  = L.Wjj(:,1:L.nC);
Wjdz = L.Wjj(:,L.nC+(1:L.ndz));

MjC  = Mjj(:,1:L.nC);
Mjdz = Mjj(:,L.nC+(1:L.ndz));

% Approximate constraints
% AIe*Ie + AIy*Iy + Aag*ag + ACo*Co + Adz*dz = b
b = [Yr;Ye;Yi;Yq;Yj(L.kj)];
AIe = [L.Wr.*L.Mre;L.Wee           ;zeros(L.ni+L.nq+L.nja,L.ne)                           ];
AIy = [L.Wr.*L.Mry;zeros(L.ne,L.ny);zeros(1,L.ny);L.Wp*ones(1,L.ny);zeros(L.nq+L.nja,L.ny)];
Aag = zeros(L.nd+L.nq+L.nj-L.nje,L.ng);
ACo = [zeros(L.nd+L.nq,L.nC);             WjC( L.kj,:) ];
Adz = [zeros(L.nr+L.ne+L.ni+L.nq,L.ndz); Wjdz( L.kj,:) ];

w = [L.Wr;L.We;L.Wi;L.Wq;L.Wj(L.kj)];

IDf = @(s,n) repmat(L.dimID.(s),n,1); % function handle to get ID arrays
ID  = [...
  IDf('Loops',L.G.nf);
  IDf('Probes',L.G.nm);
  IDf('Coils',sum(L.ke(1:L.G.na)));
  IDf('Passives',sum(L.ke(L.G.na+(1:L.G.nu))));
  IDf('DML',1);
  IDf('Ip',1);
  IDf('Regularization',L.nq);
  IDf('Constraints',sum(L.kj));
  ];

% Equality constraints (direct Ie/ag/dz specification)
% For now these are only from Ie (~L.ke) and aj (~L.kj)
% CIe*Ie + CIy*Iy + Cag*ag +CCo*Co + Cdz*dz = d;
d = [Xe(~L.ke);Xj(~L.kj)];
CIe = [Mee(~L.ke,:);zeros(L.nje,L.ne)];
Cag = zeros(L.nee+L.nje,L.ng);
CIy = zeros(L.nee+L.nje,L.ny);
CCo = [zeros(L.nee,L.nC );  MjC(~L.kj,:)];
Cdz = [zeros(L.nee,L.ndz); Mjdz(~L.kj,:)];

% Parameter bounds
% c1 = -Inf(L.ne+L.nj,1);
% c2 =  Inf(L.ne+L.nj,1);

% Inequality constraints
CiIe = zeros(L.nc,L.ne );
CiIy = zeros(L.nc,L.ny );
Ciag = zeros(L.nc,L.ng );
CiCo = zeros(L.nc,L.nC );
Cidz = zeros(L.nc,L.ndz);
ci   = zeros(L.nc,1    );

% Handle different state definitions
if strcmpi(L.P.algoNL,'all-nl')
  % x=[Iy,ag[,Ie]]
   AGS = AIy;
  CiGS = CiIy;
   CGS = CIy;
else % all-nl-Fx, Newton-GS
  % x=[Fx,ag[,Ie]] and Iy = -dlst*Fx(:)./L.rhsf-Tye*Ie
   AGS =      - ( AIy./L.rhsf.')*L.dlst;
  CiGS =      - (CiIy./L.rhsf.')*L.dlst;
   CGS =      - ( CIy./L.rhsf.')*L.dlst;
   AIe =  AIe -   AIy*L.Tye;
  CiIe = CiIe -  CiIy*L.Tye;
   CIe =  CIe -   CIy*L.Tye;
end
  
%% Express in terms of FGS state and inputs
%  u = [Ie;Co] added dz as extra

% Approximate constraints
Ra0 =  b;
Rax = [AGS,Aag].*L.xscal.';
Rau = [AIe,ACo];
Raz =  Adz;
% Equality constraints
Re0 =  d;
Rex = [CGS,Cag].*L.xscal.';
Reu = [CIe,CCo];
Rez =  Cdz;
% Inequality constraints
Ri0 =  ci;
Rix = [CiGS,Ciag].*L.xscal.';
Riu = [CiIe,CiCo];
Riz =  Cidz;

end
