function L = liuc(P,G,varargin)
%LIUC LIUQE ancillary data
% L = LIUC(P,G) returns a structure with LIUQE ancillary data. P contains
% the configuration parameters as obtained by LIUP<TOK>. G contains the
% geometry as obtained by LIUG<TOK>. See also LIUP,LIUG.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Input arguments
for k = 1:2:numel(varargin), P.(varargin{k}) = varargin{k+1}; end

%% Generic MEQ ancillary data
L = meqc(P,G);

%% Checks
assert(all(size(P.wreg) == [L.nq,1]) || numel(P.wreg)==1   ,'wreg must have size 1 or [L.nq,1] = [%d,1]',L.nq)
assert(P.Ipmeas || ~isempty(G.Ipm),'if P.Ipmeas=0, G.Ipm needs to be non-empty');

%% Code
L.code = 'liu';

%% Update meqpdom parameters for rt
if P.itert, L.FN = -1000; L.dimw=20; L.raN = -1000; end

%% Plasma current estimator: Ip = Ipm*Bm + Ipa*Ia + Ipu*Iu
L.Ipa = -G.Ipm*G.Bma;
L.Ipu = -G.Ipm*G.Bmu;
L.Ips = -G.Ipm*G.Bms;

%% Finite elements
rip = ifc(isnan(P.rip),min(G.rl),P.rip);
rop = ifc(isnan(P.rop),max(G.rl),P.rop);
zlp = ifc(isnan(P.zlp),min(G.zl),P.zlp);
zup = ifc(isnan(P.zup),max(G.zl),P.zup);

nrh = min(P.nelem,2);
nzh = floor(P.nelem/nrh);
% Corners of FE domain bounding box
L.bh = [max(rip,G.rx(1)) max(zlp,G.zx(1)) min(rop,G.rx(end)) min(zup,G.zx(end))];

% Finite Elements for initial plasma current guess
if strcmp(P.fetype,'tri')
  [L.Tyh,L.rh,L.zh] = fetr(L); % triangular finite elements
  L.nh = size(L.Tyh,2);
else
  % pyramidal or bilinear elements - share same rh,zh
  rh = linspace(L.bh(1),L.bh(3),nrh+2); drh = rh(2)-rh(1); rh([1 end]) = [];
  zh = linspace(L.bh(2),L.bh(4),nzh+2); dzh = zh(2)-zh(1); zh([1 end]) = [];
  L.nh = nrh*nzh;
  [L.rh,L.zh] = meshgrid(rh,zh); L.rh = L.rh(:); L.zh = L.zh(:);
  hr = max(1-abs(reshape(L.ry,[L.nry,1])-reshape(rh,[1,nrh]))/drh,0);
  hz = max(1-abs(reshape(L.zy,[L.nzy,1])-reshape(zh,[1,nzh]))/dzh,0);
  
  if strcmp(P.fetype,'pyr')
    L.Tyh = reshape(min(reshape(hz,[L.nzy,1,nzh,1]), reshape(hr,[1,L.nry,1,nrh])),L.ny,L.nh) .* L.Oly(:);
  elseif strcmp(P.fetype,'bil')
    L.Tyh = reshape(    reshape(hz,[L.nzy,1,nzh,1]).*reshape(hr,[1,L.nry,1,nrh]) ,L.ny,L.nh) .* L.Oly(:);
  else
    error('LIUQE:liuc:fetype','unknown P.fetype');
  end
end
L.Tyh = L.Tyh./sum(L.Tyh);

%% Edge condition
L.Mbh = meqfbp(L.Tyh,L);

%% Vertical position feedback
L.Iph  = ones(1,L.nh); % equal to sum(L.Tyh,1)
L.Ipzh = sum(L.zzy(:).*L.Tyh,1);
L.ndz  = (P.stabz ~= 0)*L.nD;

%% liuagcon sets up constraints

if isempty(P.liuagcon)
  % set default
  agcon = repmat({'ag',1},L.ng,1);
  for ig = 1:L.ng
    agcon{ig,2} = find(L.TDg(:,ig),1,'first');
  end
else
  agcon = P.liuagcon;
end

L.agconc = meqagconc(agcon,L.TDg,P.idoublet); % consolidate constraint equations
L.nC     = size(L.agconc,1);                 % number of constraint equations
L.nj     = L.nC + L.ndz;

assert(L.nC==L.ng,'number of constraints (%d) not equal to number of ag (%d)',L.nC,L.ng)

%% Effect of dz on boundary flux
L.dzMbe = (-2*pi)*L.rb.*[G.Brxa(~L.lxy,:),G.Brxu(~L.lxy,:)];

%% Measurement and constraint indices
[L.kdf,L.kdm,L.kda,L.kdu,L.kdt,L.kdp,L.nd] = n2k(G.nf,G.nm,G.na,G.nu,1,1);
[                        L.kit,L.kip     ] = n2k(                    1,1);
L.kdr = [L.kdf L.kdm]; L.kde = [L.kda L.kdu]; L.kdi = [L.kdt L.kdp];
L.nr  = numel(L.kdr);  L.ne  = numel(L.kde);  L.ni  = numel(L.kdi);
% Group measurements/constraints linked to aj only
[L.kki,L.kkq,L.kkj,L.nk] = n2k(L.ni,L.nq,L.nj);

%% Measurement weighting
% Pad as necessary
wFf = P.wFf(:); wFf(end+1:G.nf,1) = 1;
wBm = P.wBm(:); wBm(end+1:G.nm,1) = 1;
wIa = P.wIa(:); wIa(end+1:G.na,1) = 1;
wIu = P.wIu(:); wIu(end+1:G.nu,1) = 1;

% overwrite wIu if no vessel current measurements
if ~P.ivesm, wIu(:) = 0; end

Wf = wFf(1:G.nf)/P.Fferr; Wm = wBm(1:G.nm)/P.Bmerr;
Wa = wIa(1:G.na)/P.Iaerr; Wu = wIu(1:G.nu)/P.Iuerr;
L.Wp = P.wIp/P.Iperr; L.Wt = P.idml/P.Fterr;
L.Wr = [Wf ; Wm]; L.We = [Wa ; Wu]; L.Wi = [L.Wt ; L.Wp];
L.Wd = [L.Wr ; L.We ; L.Wi];
L.Wq = P.wreg.*ones(L.nq,1);

%% Fitting variables weighting
% Pad as necessary
wCo = P.wCo(:); wCo(end+1:L.nC,1) = 0;
WCo = wCo./P.Coerr;

wdz = P.wdz(:); wdz(end+1:L.ndz,1) = 0;
Wdz = wdz(1:L.ndz)/P.dzerr;

Wj  = [WCo ; Wdz];
L.Wj  = Wj;
L.WCo = WCo;

%% Exact constraints
% aj
L.kj  = ~isinf(L.Wj); % Approximate constraints selector (Inf = exact constraint)
L.nja = sum(L.kj); % Number of approximate constraints
L.nje = L.nj - L.nja; % Number of exact constraints
% Approximate constraints will enter the LLS problem as Wjj*aj = Yj,
% lines corresponding to exact constraints are replaced with 0s, including
% Yj to avoid artificially increasing chi
L.Wj(~L.kj) = 0;
% Ie
L.ke  = ~isinf(L.We); % Approximate constraints selector
L.nea = sum(L.ke); % Number of approximate constraints
L.nee = L.ne - L.nea; % Number of exact constraints
% Approximate constraints will enter the LLS problem as Wee*Ie = Ye,
% lines corresponding to exact constraints are replaced with 0s, including
% Ye to avoid artificially increasing chi
L.We(~L.ke) = 0;

%% Response matrices
L.Mry = [G.Mfx(:,L.lxy);G.Bmx(:,L.lxy)];
L.Mre = [G.Mfa G.Mfu;G.Bma G.Bmu];
L.Mdzry = [G.dzMfx(:,L.lxy) ; G.dzBmx(:,L.lxy)];
L.Wre = L.Wr.*L.Mre;
L.Wee = diag(L.We);
L.Wjj = diag(L.Wj);
L.Mty = reshape(G.Mtx(L.lxy),1,[]);

%% FE fit
% response matrices
Wrh = L.Wr.*L.Mry*L.Tyh;
Wih = [zeros(1,L.nh) ; L.Wp*L.Iph];
% Remove columns for exactly constrained Ie
Wre = L.Wre(:,L.ke);
Wee = L.Wee(:,L.ke);
L.Wde = [Wre ; Wee              ; zeros(L.ni,L.nea)];
L.Wdh = [Wrh ; zeros(L.ne,L.nh) ; Wih              ];

if L.P.Ipvacuum>0
  % special case for Ip=0 forcing Jh=0, fit only Ie
  L.Aed0  = (L.Wde'*L.Wde) \ L.Wde';
end

if P.iterh>0 % ipm
  Ahh = [Wrh ; Wih]'*[Wrh ; Wih];
  L.uAhh = Ahh(triu(true(size(Ahh))));
  L.Ahd = zeros(L.nh,L.nd);
  L.Ahd(:,[L.kdr L.kdi]) = -[Wrh' Wih'];
  L.Aed = zeros(L.nea,L.nd);
  L.Aed(:,[L.kdr L.kde]) = (Wre'*Wre + Wee'*Wee) \ [Wre;Wee]';
  L.Ahe = Wrh'*Wre;
  L.Aeh = -L.Aed(:,L.kdr)*Wrh;
  L.Ahh = L.Ahe*L.Aeh;
else % direct solving of unconstrained lsq
  WdH   = [L.Wde L.Wdh];
  AHd   = (WdH'*WdH) \ WdH';
  L.Aed = AHd(      1:L.nea,:);
  L.Ahd = AHd(L.nea+1:end  ,:);
end

%% Fitting base function amplitudes
% Note: columns for exactly constrained Ie have been removed
% fixed parts
Wree = [Wre ; Wee];
iAee = (Wree'*Wree)\eye(L.nea);
Aere = iAee*Wree';
% for Simulink
L.Ae1d(:,[L.kdr L.kde L.kdi]) = [Aere zeros(L.nea,L.ni)];
% for bslv
L.Aer = Aere(:,     1:L.nr);
L.Aee = Aere(:,L.nr+1:end );

%% p and T2/2 profile, RT only
[L.TQg,L.ITQg] = L.bfct(2,L.bfp,L.pQ(:).^2,zeros(L.nD,1),ones(L.nD,1));

%% Miscellaneous
L.smalldia = P.itert>0;         % Enable small diamagnetism approximation when itert>0
L.liurtemu = P.itert && ~P.slx; % Flag for when liut is used to emulate Simulink/rt runs

%% NL model setup
% Use static equilibrium
circuit = false;
L.cdec = {};
L.np = 0;
[L.ind,L.nN,L.dimN,L.xscal,L.resscal,L.LX2x,L.isEvolutive,L.nuN,L.nrD] = meqalgoNLc(L,P.algoNL,circuit);

%% Setup for analytical jacobian
% Additional derivative functions
if P.gsxe~=3, [L.Mxy,L.Mxe] = meqFxjac(L); end

%% Linear part of jacobians
[L.Jx,L.Ju,L.Jxdot] = fgeFvacJac(L);

%% Identifier strings for cost function terms
IDnames = {'Loops','Probes','Coils','Passives','DML','Ip','Regularization','Constraints'};
L.dimID = cell2struct(num2cell(1:numel(IDnames))',IDnames);

end
