function liuanasrv(shot,~,analysis)
% LIUANASRV  Called by ANASRV to compute and write in RESULTS tree
% LIUANASRV(SHOT,VERSION,ANALYSIS)
%
% ANALYSIS can currently be either:
%  - THOMSON: LIUQE and PSITBX run at Thomson times only.
%  - FAST:    LIUQE and PSITBX run with millisecond time resolution and at
%      Thomson times. [default]
%  - MAX:     LIUQE and PSITBX run for the whole magnetics time base.
%
% The PSI and J_TOR nodes are stored only at PSITBX times.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if nargin < 3, analysis = 'fast'; end
analysis = validatestring(lower(analysis),{'thomson','fast','max'},mfilename,'analysis',3);

switch lower(analysis)
 case 'thomson'
  type  = 'thomson';
 case 'fast'
  type  = 'mixed';
 case 'max'
  type  = 'magnetics';
end

% Run comment
comment = sprintf('Standard LIUQE.M run with TYPE=%s',type);

% Handle extra times
psi_extra  = [];
fast_extra = [];
if ~strcmpi(analysis,'thomson')
 meqmdsopen(shot,'results');
 psi_extra  = mdsvalue('if_error(\TOP.EQUIL_1.TIME_REQUEST:PSI_EXTRA,*)').';
 fast_extra = mdsvalue('if_error(\TOP.EQUIL_1.TIME_REQUEST:FAST_EXTRA,*)').';
end

% Assemble time arrays
t = liuqe_times(shot,type,[psi_extra(:);fast_extra(:)]);
nt = numel(t);

% Parallel options
parneeded = nt>300;
paravail = ~isempty(ver('parallel')) || ~isempty(ver('distcomp'));

% Run LIUQE
[L,LX] = liuqe(shot,t,'LYall',false);
nt = numel(t);
if paravail && parneeded
  % Use parallel pool to speed up computation
  pool = gcp('nocreate');
  if isempty(pool)
    pool = parpool([2,4]); % At least 2 workers, at most 4 workers
    cleanup_pool = onCleanup(@() delete(pool));
  end
  np = pool.NumWorkers;
  % Split LX into np segments
  ns = ceil(nt/np);
  LX_ = cell(1,np);
  LY_ = LX_;
  for ii = 1:np
    LX_{ii} = meqxk(LX,(ii-1)*ns+1:min(ii*ns,nt));
  end
  % Run LIU in parallel loop
  parfor ii = 1:np
    LY_{ii} = liut(L,LX_{ii});
  end
  % Merge into single LY structure
  LY = meqcat(LY_{:});
else
  % Run sequentially
  LY = liut(L,LX);
end

% List of node dependencies
dep = {'\MAGNETICS::TOP.MEASUREMENTS.*:*:*',...
       '\RESULTS::DML:BO%%%.PARAM:*',...
       '\RESULTS::DML:CK%%%:*',...
       '\RESULTS::TOP.EQUIL_1.PARAMETERS:*',...
       '\RESULTS::TOP.EQUIL_1.TIME_REQUEST:*',...
      };
if ~L.P.idml
 dep(startsWith(dep,'\RESULTS::DML')) = [];
end

% Write results to RESULTS nodes
% Run psiwtcv first so that all the writing comes after the PSITBX comp.
psiwtcv(shot,'RESULTS','\TOP.EQUIL_1.RESULTS',L,LY,dep)
meqwtcv(shot,'RESULTS','\TOP.EQUIL_1.RESULTS',L,LY,dep)
liuwtcv(shot,'RESULTS','\TOP.EQUIL_1.RESULTS',L,LY,dep)
codwtcv(shot,'RESULTS','\TOP.EQUIL_1.CODE','LIUQE.M',comment);
if L.P.idml, dmlbo(shot,'BOSIG'), end
