function LY = liht(L,LX,varargin)
% LIUQE-like finite element plasma current distribution observer
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% optional parameters
for k = 1:2:numel(varargin)
  switch varargin{k}
    % Additional output arguments
    case 'argout', L.argoutc = meqargoutc(varargin{k+1},L.argoutc);
    % Additional parameters
    otherwise, L.P.(varargin{k}) = varargin{k+1};
  end
end

nt = numel(LX.t);

for kt = 1:nt
  t  = LX.t(kt);
  Ip = LX.Ip(kt); rBt = LX.rBt(kt);
  shot = LX.shot(min(kt,end));
  
  Xr = [LX.Ff(:,kt); LX.Bm(:,kt)]; % magnetic measurements
  Xe = [LX.Ia(:,kt); LX.Iu(:,kt)]; % external currents
  Xi = [LX.Ft(kt)*rBt ; Ip];       % integral measurements
  Yr = L.Wr.*Xr;
  Ye = L.We.*Xe;
  Yi = L.Wi.*Xi;
  Yd = [Yr ; Ye ; Yi];
  
  Jh0 = repmat(Ip/L.nh,L.nh,1);
  [Ie,Jh,Zd] = lihlsqsolve(L,Yd,Xe,Jh0,Ip);
  chih = sqrt(sum((Zd-Yd).^2)/L.nd); % chi of current fit

  Ip      = L.Iph  * Jh ;
  zIp     = L.Ipzh * Jh ;
  rIp     = L.Iprh * Jh ;
  rY      = rIp/Ip;
  zY      = zIp/Ip;

  if L.P.idoublet
    IpD     = [L.Iph_t  * Jh ; L.Iph_b  * Jh];
    rIpD    = [L.Iprh_t * Jh ; L.Iprh_b * Jh];
    zIpD    = [L.Ipzh_t * Jh ; L.Ipzh_b * Jh];
  else
    IpD = Ip; zIpD=zIp; rIpD=rIp;
  end
  rYD = rIpD./IpD;
  zYD = zIpD./IpD;
  
  Ia = Ie(1:L.G.na);
  Iu = Ie(L.G.na+(1:L.G.nu));
  Iy = reshape(L.Tyh*Jh,L.nzy,L.nry);
  
  %% Poisson solver for a coarse flux estimate
  Fx = meqFx(L,Iy,Ie,[],0,1,Jh);
  if L.P.ivacuum, end

  %% Fake meqpdom-related quantities
  [~,~,~,~,~,~,~,zX,rX,FX,dz2FX,dr2FX,drzFX,~,stat] = ...
    asxymex(Fx,L.G.zx,L.G.rx,L.P.dasm,L.dzx,L.drx,L.idzx,L.idrx,L.Oasx,L.dimw);

  if (~stat)
    % Issue warning only, errors handled if nA~=1.
    % Note: should never happen if itert=0 since dimw=numel(L.Oly)
    s=warning('off','backtrace');
    warning('asxymex:workingSpace','Working space (%d) too small, increase DIMW',L.dimw);
    warning(s);
  end

  nA = 0; nB = 0;
  [rB,zB,FB] = deal(zeros(nB,1));
  [rA,zA,FA,dr2FA,dz2FA,drzFA] = deal(zeros(nA,1));
  lB = false;
  lX = false(L.nD,1);
  Opy = zeros(L.nzy,L.nry,'int8');
  F0 = zeros(L.nD,1); F1 = F0;

  %% Postprocessing
  ag = zeros(L.ng,1);
  LYt = struct('shot',shot,'t',t);
  [LYt] = meqpost(L,LYt,ag,...
    Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lB,lX, ...
    rX,zX,FX,dr2FX,dz2FX,drzFX,...
    rBt,Ia,Iu,Iy,Opy,F0,F1);
  
  % override some values
  LYt.Ip = sum(Iy); 
  LYt.IpD = IpD;   
  LYt.rIpD = rIpD;
  LYt.zIpD = zIpD;
  LYt.nA = nA;      
  LYt.rY = rY;
  LYt.zY = zY;

  LYt.rYD = rYD;
  LYt.zYD = zYD;
  
  % lih specials
  LYt  = meqlarg(LYt,chih,Ip); 

  % Additional output arguments
  for k = 1:numel(L.argoutc)
    if isempty(L.argoutc{k}.file)
      LYt.(L.argoutc{k}.fld) = eval(L.argoutc{k}.exp);
    end
  end

  % time derivatives
  if kt==1
    LYt = meqLYdot(LYt,LYt);
  else
    LYt = meqLYdot(LYt,LY(kt-1));
  end

  % store in LY
  LY(kt) = LYt; %#ok<AGROW>
end

LY = meqlpack(LY);

end
