function [Ie,Jh,Zd] = lihlsqsolve(L,Yd,Xe,Jh,Ip)
% LIHLSQSOLVE Solve LIH linear least squares problem
%
%   [Ie,Jh,Zd] = lihlsqsolve(L,Yd,Xe,Jh,sIp)
%
% Inputs:
%   L    LIH/LIU ancillary data structure
%   Yd   Vector of weighted measurements [Bm;Ff;Ia;Iu;Ip;Xt]
%   Xe   Vector of external currents measurements (used as initial guess)
%   Jh   Initial guess for FE currents
%   Ip   Plasma current
%
% Outputs:
%   Ie   Fitted external currents
%   Jh   Fitted FE currents
%   Zd   Reconstructed values of the weighted measurements
%
% In the LIH linear least squares problem the unknowns are the external
% currents Ie and the vector Jh that make up the plasma current via the
% relation Iy=Tyh*Jh where Tyh is fixed. The residuals are Bm,Ff,Ia,Iu and
% Ip (Xt, the DML measurement, is excluded).
%
% For |Ip|< L.P.Ipvacuum, only Ie is fitted and Jh is constrained to 0
% For |Ip|>=L.P.Ipvacuum, both Ie and Jh are fitted. When furthermore
% L.P.iterh>0, the constraint sign(Ip)*Jh>=0 is enforced and an
% interior-point algorithm is used.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Ie constraints
ke = L.ke;
Ie = Xe;
if L.nee > 0
  % Update measurements depending on Ie
  dYr = L.Wre(:,~ke)*Ie(~ke);
  Yd(L.kdr) = Yd(L.kdr) - dYr;
end

%% Fit
if abs(Ip) < L.P.Ipvacuum
  % Constrained LSQ fit for Jh and Ie with Jh=0 (no plasma current on the grid)
  ZIe = L.Aed0*Yd;
  Jh  = zeros(L.nh,1); % force Jh = 0 
else
  if L.P.iterh
    % Constrained LSQ fit for Jh and Ie with sIp*Jh>0
    sIp = sign(Ip); if sIp==0, sIp=1; end % avoid sIp==0, unhandled by ipmhmex
    [ZIe,Jh,~] = ipmhmex(L.Ahd,L.Aed,L.Ahe,L.Aeh,L.Ahh,L.uAhh,Yd,Xe(ke),Jh,sIp,L.P.iterh,max(L.P.Ipmin/L.nh,abs(Jh(1)))*L.P.tolh);
  else
    % Unconstrained LSQ fit for Jh and Ie
    ZIe = L.Aed*Yd;
    Jh  = L.Ahd*Yd;
  end
end
Ie(ke) = ZIe;
% Synthetic measurements
Zd = L.Wdh*Jh+L.Wde*ZIe;
Zd(L.kdt) = Yd(L.kdt); % Wd(L.kdt) is effectively 0 for this FE fit but Yd(L.kdt) might not reflect it
if L.nee>0
  Zd(L.kdr) = Zd(L.kdr) + dYr; % Add contribution from exact Ie
end