function [ Mnf,  Mnm,  Mna,  Mnu,  Mnp,...
          Brnf, Brnm, Brna, Brnu, Brnp,...
          Bznf, Bznm, Bzna, Bznu, Bznp] = lih2shapobs(L,rn,zn)
% LIH2SHAPOBS get linear shape observers using LIH matrices/methods.
%         
%[ Mnf,  Mnm,  Mna,  Mnu,  Mnp,...
% Brnf, Brnm, Brna, Brnu, Brnp,...
% Bznf, Bznm, Bzna, Bznu, Bznp] = lih2shapobs(L,rn,zn)
% 
% Inputs:
% . L structure from lih.m
% . (rn, zn) coordinates of interpolation points
% 
% Returns observer matrices such that:
% Fn   = [ Mnf,  Mnm,  Mna,  Mnu,  Mnp] * Xd;
% Brn  = [Brnf, Brnm, Brna, Brnu, Brnp] * Xd;
% Bzn  = [Bznf, Bznm, Bzna, Bznu, Bznp] * Xd;
% where Xd are the (unweighted) measurements: [Ff;Bm;Ia;Iu;Ip]
% 
% See also LIHT, LIHLSQSOLVE
% 
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% get Lih matrices to produce Ie and Jh
[Med,Mhd,kdu,kdp] = lih2obs_setup(L);

% generate mutual inductance matrices (and B field mats) for interpolation points
[Mne, Mnh, Brne, Brnh, Bzne, Bznh] = get_mut_mats(L, rn, zn);

% combine observer matrices with lih matrices
[ Mnf,  Mnm,  Mna,  Mnu,  Mnp] = dispatch( Mne,  Mnh, Med, Mhd, L, kdu, kdp);
[Brnf, Brnm, Brna, Brnu, Brnp] = dispatch(Brne, Brnh, Med, Mhd, L, kdu, kdp);
[Bznf, Bznm, Bzna, Bznu, Bznp] = dispatch(Bzne, Bznh, Med, Mhd, L, kdu, kdp);
end


function [Anf, Anm, Ana, Anu, Anp] = dispatch(Ane, Anh, Med, Mhd, L, kdu, kdp)
Anf = Ane * Med(:,L.kdf) + Anh  * Mhd(:,L.kdf);
Anm = Ane * Med(:,L.kdm) + Anh  * Mhd(:,L.kdm);
Ana = Ane * Med(:,L.kda) + Anh  * Mhd(:,L.kda);
Anu = Ane * Med(:,  kdu) + Anh  * Mhd(:,  kdu);
Anp = Ane * Med(:,  kdp) + Anh  * Mhd(:,  kdp);
end


function [Mne, Mnh, Brne, Brnh, Bzne, Bznh] = get_mut_mats(L, rn, zn)
% init
nn = numel(rn);
inM = qintc([],L.drx,L.dzx);

% influence of Ie on flux and derivatives computed using grid interpolation
% to correctly treat coils overlapping the grid
Fxh = meqFx(L, zeros(L.nzy, L.nry, L.ne), eye(L.ne));
[Mne,Brne,Bzne] = qintmex(L.rx,L.zx,Fxh,rn,zn,inM);
% reshape
Mne  = reshape( Mne, nn, L.ne);
Brne = reshape(Brne, nn, L.ne);
Bzne = reshape(Bzne, nn, L.ne);

% generate influence from Jh using grid interpolation to avoid closeness
% to singularities
Iyh = reshape(L.Tyh,L.nzy,L.nry,L.nh);
Fxh = meqFx(L, Iyh, zeros(L.ne, L.nh));
[Mnh,Brnh,Bznh] = qintmex(L.rx,L.zx,Fxh,rn,zn,inM);
% reshape
Mnh  = reshape( Mnh, nn, L.nh);
Brnh = reshape(Brnh, nn, L.nh);
Bznh = reshape(Bznh, nn, L.nh);
end