/* CIZR Compute custom plasma cross-section integrals
 *   See description in cizrmex.m */

/* [+MEQ MatlabEQuilibrium Toolbox+]
 *
 *    Copyright 2022-2025 Swiss Plasma Center EPFL
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License. */

# include "meq.h"

/* function [IVQD] = cizr(Fx,Opy,Vy,dsx,FNQ,F0,F1) */
void FLT_NAME(cizr)(FLT *y1, FLT *f, int8_t *O, FLT *Vy,
                    FLT ds, FLT *FNQ, FLT *F0, FLT *F1,
                    int nV, int nQ, int nD, int nr, int nz) {
  int i,j,k,iQ,iD,iV;
  int n=nQ*nD, nr2 = nr-2, nz2 = nz-2, ny=nr2*nz2;
  FLT FN, idF[nD];
  FLT *y1p[nV], *Vyp[nV];
  /* Initialize pointers to different V functions and integrals */
  for (iV=0;iV<nV;iV++) {
    y1p[iV] = y1 + iV*n;
    Vyp[iV] = Vy + iV*ny;
  }
  /* Get inverse of flux variation over all domains */
  for (iD=0;iD<nD;iD++) {
    idF[iD] = F1[iD] - F0[iD];
    if (idF[iD]) idF[iD] = 1./idF[iD];
  }
  /* Initialize all integrals */
  memset(y1,0,n * nV * sizeof *y1);
  /* Get Fx to first point of y grid */
  f+=nz+1;
  for (i=nr2;i--;) {
    for (j=nz2;j--;) {
      /* Check domain of current point */
      iD = *O++;
      if (iD>0 && iD<=nD) {
        iD--;
        /* Find iQ such that FNQ[iQ-1]<FN<=FNQ[iQ] */
        FN=(*f-F0[iD])*idF[iD];
        iQ=0;
        while (iQ<nQ && FNQ[iQ]<FN) {
          iQ++;
        }
        /* If flux is between F0 and F1 (it always should if Opy is consistent with f,F0,F1) */
        if (iQ<nQ) {
          /* Add its contribution to corresponding integrals */
          k = iQ+nQ*iD;
          for (iV=0;iV<nV;iV++)
            y1p[iV][k] += *Vyp[iV] * ds;
        }
      }
      /* Update pointers */
      f++;
      for (iV=0;iV<nV;iV++)
        Vyp[iV]++;
    }
    /* Skip over Fx points to get to next y point */
    f+=2;
  }
  /* Cumulative sums */
  k = 1;
  for (iD=0;iD<nD;iD++) {
    for (iQ=1;iQ<nQ;iQ++) {
      for (iV=0;iV<nV;iV++) {
        y1p[iV][k] += y1p[iV][k-1];
      }
      k++;
    }
    k++;
  }
}
