/* [+MEQ MatlabEQuilibrium Toolbox+]
 *
 *    Copyright 2022-2025 Swiss Plasma Center EPFL
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License. */


/*
 * bfef.c contains the definiton of the bfef basis function set
 *
 * See also bfef.m
 */

# include "meq.h"

/* List of function pointers to use when creating the bfct_params structure */
#define FUNPARAMS FLT_NAME(bfef_f) , FLT_NAME(bfef_fN) , FLT_NAME(bfef_fag) , FLT_NAME(bfef_fA) , \
                  FLT_NAME(bfef_df), FLT_NAME(bfef_dfN), FLT_NAME(bfef_dfag), FLT_NAME(bfef_dfA)

/*
 * First we define some elementary functions to evaluate the necessary 
 * quantities for all polynomial orders. These will later be used when
 * mapped to each basis function
 */

/* Compute physical basis functions */
void FLT_NAME(bfef_f_)(FLT g[4], FLT Ig[4], FLT FxA, FLT FBA)
{
   g[0] =  1.0;                              /* 1                                             */
   g[1] =  FxA - FBA ;                       /* Fx - FB                                       */
   g[2] =  g[1] * FxA;                       /* (Fx - FA) (Fx - FB)                           */
   g[3] =  g[2] * (g[1] + FxA);              /* (Fx - FA) (Fx - FB) (2 Fx - FA - FB )         */
  Ig[0] =  g[1];                             /* Fx - FB                                       */
  Ig[1] =  0.5 * g[1] * g[1];                /* (Fx - FB)^2 / 2                               */
  Ig[2] =  Ig[1] * (TWOITHREE * g[1] + FBA); /* (Fx - FB)^2 ((Fx - FB) 2 / 3 + (FB - FA)) / 2 */
  Ig[3] =  0.5 * g[2] * g[2];                /* (Fx - FA)^2 (Fx - FB)^2 / 2                   */
}

/* Compute normalized basis functions */
void FLT_NAME(bfef_fN_)(FLT gN[4], FLT IgN[4], FLT FN)
{
   gN[0] = 1.0;                              /*  1                        */
   gN[1] = FN - 1.0;                         /*  FN - 1                   */
   gN[2] = gN[1] *  FN;                      /* (FN - 1) FN               */
   gN[3] = gN[2] * (gN[1] + FN);             /* (FN - 1) FN (2 FN - 1)    */
  IgN[0] = gN[1];                            /*  FN - 1                   */
  IgN[1] = 0.5 * gN[1] * gN[1];              /* (FN - 1)^2 / 2            */
  IgN[2] = IgN[1] * (TWOITHREE*FN + ITHREE); /* (FN - 1)^2 (2 FN + 1) / 6 */
  IgN[3] = IgN[1] *  FN  *  FN;              /* (FN - 1)^2 FN^2 / 2       */
}

/* Compute physical basis functions at magnetic axis */
void FLT_NAME(bfef_fA_)(FLT g[4], FLT Ig[4], FLT  FBA)
{
   g[0] =  1.0;                  /*  1                 */
   g[1] = -FBA ;                 /*  FA - FB           */
   g[2] =  0.0;                  /*  0                 */
   g[3] =  0.0;                  /*  0                 */
  Ig[0] = -FBA;                  /*  FA - FB           */
  Ig[1] =  0.5 * FBA * FBA;      /* (FA - FB)^2 / 2    */
  Ig[2] =  ITHREE * Ig[1] * FBA; /* (FA - FB)^3 / (-6) */
  Ig[3] =  0.0;                  /*  0                 */
}

/* Conversion factors from normalized to physical basis functions */
void FLT_NAME(bfef_fag_)(FLT alphapg[4], FLT alphag[4], FLT FBA)
{
  alphapg[0] =            1.0; /* 1         */
  alphapg[1] =            FBA; /* (FB-FA)   */
  alphapg[2] = alphapg[1]*FBA; /* (FB-FA)^2 */
  alphapg[3] = alphapg[2]*FBA; /* (FB-FA)^3 */
   alphag[0] = alphapg[1]    ; /* (FB-FA)   */
   alphag[1] = alphapg[2]    ; /* (FB-FA)^2 */
   alphag[2] = alphapg[3]    ; /* (FB-FA)^3 */
   alphag[3] = alphapg[3]*FBA; /* (FB-FA)^4 */
}

/* Compute normalized basis functions */
void FLT_NAME(bfef_dfN_)(FLT dgN[4], FLT dIgN[4], FLT FN)
{
   dgN[0] = 0.0;                             /*  0                         */
  dIgN[0] = 1.0;                             /*  1                         */
   dgN[1] = dIgN[0];                         /*  1                         */
  dIgN[1] = FN - 1.0;                        /* (FN - 1)                   */
   dgN[2] = dIgN[1] + FN;                    /*  2 FN - 1                  */
  dIgN[2] = dIgN[1] * FN;                    /* (FN - 1) * FN              */
   dgN[3] = dgN[2] * dgN[2] + 2.0 * dIgN[2]; /* (2 FN - 1)^2 + 2 (FN-1) FN */
  dIgN[3] = dIgN[2] * dgN[2];                /* (FN - 1) * FN * (2 FN - 1) */
}

/* Compute derivatives of physical basis functions */
void FLT_NAME(bfef_df_)(FLT  dgdFxA[4], FLT  dgdFBA[4],
                        FLT dIgdFxA[4], FLT dIgdFBA[4],
                        FLT FxA, FLT FBA)
{
   dgdFxA[0] =  0.0;                                              /* 0                                          */
  dIgdFxA[0] =  1.0;                                              /* 1                                          */
   dgdFxA[1] =  1.0;                                              /* 1                                          */
  dIgdFxA[1] =  FxA - FBA;                                        /* Fx - FB                                    */
   dgdFxA[2] =  dIgdFxA[1] + FxA;                                 /* 2 Fx - FA - FB                             */
  dIgdFxA[2] =  dIgdFxA[1] * FxA;                                 /* (Fx - FB) (Fx - FA)                        */
   dgdFxA[3] =  dgdFxA[2] * dgdFxA[2] + 2.0 * dIgdFxA[2];         /* 2 (Fx - FA) (Fx - FB) + (2 Fx - FA - FB)^2 */
  dIgdFxA[3] =  dIgdFxA[2] * dgdFxA[2];                           /* (Fx - FA) (Fx - FB) ( 2 Fx - FA - FB)      */
   dgdFBA[0] =  0.0;                                              /* 0                                          */
  dIgdFBA[0] = -1.0;                                              /* -1                                         */
   dgdFBA[1] = -1.0;                                              /* -1                                         */
  dIgdFBA[1] = -dIgdFxA[1];                                       /* FB - Fx                                    */
   dgdFBA[2] = -FxA;                                              /* FA - Fx                                    */
  dIgdFBA[2] =  0.5 * dIgdFxA[1] * dIgdFBA[1] + dIgdFBA[1] * FBA; /* -0.5*(FB - Fx)^2 + (FB - Fx) (FB - FA)     */
   dgdFBA[3] =  dgdFBA[2] * (dgdFxA[2] + dIgdFxA[1]);             /* (FA - Fx)*(3*Fx - 2 FB - FA)               */
  dIgdFBA[3] =  dIgdFxA[2] * dgdFBA[2];                           /* (Fx - FA)^2 (FB - Fx)                      */
}

/* Compute derivatives of physical basis functions at magnetic axis */
void FLT_NAME(bfef_dfA_)(FLT dgAdFBA[4], FLT dIgAdFBA[4], FLT FBA)
{
   dgAdFBA[0] =  0.0;             /* 0               */
  dIgAdFBA[0] = -1.0;             /* -1              */
   dgAdFBA[1] = -1.0;             /* -1              */
  dIgAdFBA[1] =  FBA;             /* FB - FA         */
   dgAdFBA[2] =  0.0;             /* 0               */
  dIgAdFBA[2] =  0.5 * FBA * FBA; /* 0.5*(FB - FA)^2 */
   dgAdFBA[3] =  0.0;             /* 0               */
  dIgAdFBA[3] =  0.0;             /* 0               */
}

/* Derivatives of conversion factors from normalized to physical basis functions */
void FLT_NAME(bfef_dfag_)(FLT dalphapg[4], FLT dalphag[4], FLT FBA)
{
  dalphapg[0] =                        0.0; /* 0           */
  dalphapg[1] =                        1.0; /* 1           */
  dalphapg[2] =                    2.0*FBA; /* 2 (FB-FA)   */
  dalphapg[3] = dalphapg[2]*       1.5*FBA; /* 3 (FB-FA)^2 */
   dalphag[0] = dalphapg[1]               ; /* 1           */
   dalphag[1] = dalphapg[2]               ; /* 2 (FB-FA)   */
   dalphag[2] = dalphapg[3]               ; /* 3 (FB-FA)^2 */
   dalphag[3] = dalphapg[3]*FOURITHREE*FBA; /* 4 (FB-FA)^3 */
}

/*
 * Next we define some functions to map all polynomial orders to each basis
 * function
 */

/* Function to dispatch all orders to 1...ng */
void FLT_NAME(bfef_dispatch)(int narray, FLT ** arrayin, FLT ** arrayout, void *params)
{
  int nP, nT;
  int i;

  /* Get number of basis functions for each class */
  nP = ((bfef_params *)params)->nP;
  nT = ((bfef_params *)params)->nT;
  /* Dispatch to output arrays */
  for (i=0; i<narray; i++) {
    memcpy(  arrayout[i]     ,arrayin[i], nP*sizeof(FLT));
    memcpy(&(arrayout[i][nP]),arrayin[i], nT*sizeof(FLT));
  }
}

/* Assign each function to P' or TT' */
void FLT_NAME(bfef_fPg)(int nP, int nT, FLT fPg[], FLT fTg[])
{
  int k;
  
  /* Do the work */
  for (k=0; k<nP; k++) {
    fPg[k   ] = 1;
    fTg[k   ] = 0;
  }
  for (k=0; k<nT; k++) {
    fPg[k+nP] = 0;
    fTg[k+nP] = 1;
  }
}

/*
 * Next we define functions that will evaluate the full basis function set
 * These functions first call the elementary functions defined at the top
 * of this file and then dispatches their results based on (nP,nT) values.
 * These functions are the ones used in the generic bfct functions.
 */

void FLT_NAME(bfef_f)(void *params, FLT g[], FLT Ig[], FLT FxA, FLT FBA)
{
  FLT g_[4], Ig_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {g_,Ig_};
  FLT * arrayout[2] = {g ,Ig };

  /* Compute all orders */
  FLT_NAME(bfef_f_)(g_, Ig_, FxA, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfef_fN)(void *params, FLT gN[], FLT IgN[], FLT FN)
{
  FLT gN_[4], IgN_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {gN_,IgN_};
  FLT * arrayout[2] = {gN ,IgN };

  /* Compute all orders */
  FLT_NAME(bfef_fN_)(gN_, IgN_, FN);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfef_fA)(void *params, FLT g[], FLT Ig[], FLT FBA)
{
  FLT g_[4], Ig_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {g_,Ig_};
  FLT * arrayout[2] = {g ,Ig };

  /* Compute all orders */
  FLT_NAME(bfef_fA_)(g_, Ig_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfef_fag)(void *params, FLT alphapg[], FLT alphag[], FLT FBA)
{
  FLT alphapg_[4], alphag_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {alphapg_,alphag_};
  FLT * arrayout[2] = {alphapg ,alphag };

  /* Compute all orders */
  FLT_NAME(bfef_fag_)(alphapg_, alphag_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfef_df)(void *params, FLT  d1g[], FLT  d2g[],
                                     FLT d1Ig[], FLT d2Ig[],
                                     FLT FxA, FLT FBA)
{
  FLT d1g_[4], d2g_[4], d1Ig_[4], d2Ig_[4];
  /* 4 arrays to compute */
  FLT * arrayin [4] = {d1g_,d2g_,d1Ig_,d2Ig_};
  FLT * arrayout[4] = {d1g ,d2g ,d1Ig ,d2Ig };

  /* Compute all orders */
  FLT_NAME(bfef_df_)(d1g_, d2g_, d1Ig_, d2Ig_, FxA, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(4, arrayin, arrayout, params);
}

void FLT_NAME(bfef_dfN)(void *params, FLT dgN[], FLT dIgN[], FLT FN)
{
  FLT dgN_[4], dIgN_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {dgN_,dIgN_};
  FLT * arrayout[2] = {dgN ,dIgN };

  /* Compute all orders */
  FLT_NAME(bfef_dfN_)(dgN_, dIgN_, FN);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfef_dfA)(void *params, FLT dg[], FLT dIg[], FLT FBA)
{
  FLT dg_[4], dIg_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {dg_,dIg_};
  FLT * arrayout[2] = {dg ,dIg };

  /* Compute all orders */
  FLT_NAME(bfef_dfA_)(dg_, dIg_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(2, arrayin, arrayout, params);
}

void FLT_NAME(bfef_dfag)(void *params, FLT dalphapg[], FLT dalphag[], FLT FBA)
{
  FLT dalphapg_[4], dalphag_[4];
  /* 2 arrays to compute */
  FLT * arrayin [2] = {dalphapg_,dalphag_};
  FLT * arrayout[2] = {dalphapg ,dalphag };

  /* Compute all orders */
  FLT_NAME(bfef_dfag_)(dalphapg_, dalphag_, FBA);
  /* Dispatch to 1...ng */
  FLT_NAME(bfef_dispatch)(2, arrayin, arrayout, params);
}

/*
 * These functions are the ones used in Simulink S-function blocks for each
 * mode and in MEX-files when a custom implementation is present or needed
 *  - modes 1-5, 8, 11, 15: the generic bfct functions are called
 *  - mode 6 (regularisation constraints) has a custom implementation
 *  - mode 7 (inequality constraints) is currently empty
 */

/* Transfer matrix and integrals over plasma domain */
void FLT_NAME(bfef1)(FLT *Tyg, FLT *Tpg, FLT *ITpg,
                     FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ry, FLT *iry, int nzy, int nry,
                     int nP, int nT, int err) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfef_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct1)(Tyg, Tpg, ITpg,
                  Fx, FA, FB, Opy,
                  ry, iry, nzy, nry, err,
                  &params);
}

/* Normalized basis functions */
void FLT_NAME(bfef2)(FLT *gQg, FLT *IgQg, FLT *FNQ, int nQ, int nP, int nT) {
  /* Parameters */
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct2)(gQg, IgQg, FNQ, nQ,
                  &params);
}

/* Coefficients to transform normalized basis functions into Pp/P/TTp/hqT */
void FLT_NAME(bfef3)(FLT *aPpg, FLT *aTTpg, FLT *aPg, FLT *ahqTg, FLT *ag, 
                     FLT FA, FLT FB, FLT ids, int nP, int nT) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfef_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct3)(aPpg, aTTpg, aPg, ahqTg, ag,
                  FA, FB, ids,
                  &params);
}

/* Vertical chord integrals */
void FLT_NAME(bfef4)(FLT *Tdg, FLT *Tgy, FLT *Fx, FLT FA, FLT FB, int8_t *Opy, int nzy, int nry,
                     unsigned int *kd, FLT *fd, int nd, int nP, int nT) {
  /* Parameters */
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct4)(Tdg, Tgy, Fx, FA, FB, Opy, nzy, nry,
                  kd, fd, nd,
                  &params);
}

/* Physical basis functions at magnetic axis */
void FLT_NAME(bfef5)(FLT *gA, FLT *IgA, FLT FA, FLT FB, int nP, int nT) {
  /* Parameters */
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct5)(gA, IgA, FA, FB,
                 &params);
}

/* Regularization constraints */
void FLT_NAME(bfef6)(FLT *Qqg, FLT *Xq, FLT FA, FLT FB, FLT rA, FLT irA, FLT ids, int nP, int nT, int nq) {
  int nPq = nP>2 ? nP-2 : 0;
  /* int nTq = nT>2 ? nT-2 : 0; */
  /* int nq = nPq + nTq + 2*((nP+nT)>0); */
  FLT FBA = FB-FA, FBA2 = FBA*FBA, FBA3 = FBA2*FBA;
  /* Arrays init */
  memset(Qqg,0,nq*(nP+nT)*sizeof(FLT));
  memset(Xq ,0,nq        *sizeof(FLT));
  /* */
  switch (nP) {
    case 4: Qqg[     3*nq+    1] = 3.464101615137754*FBA3*rA*ids; /* sqrt(12) */
    case 3: Qqg[     2*nq      ] = 2.000000000000000*FBA2*rA*ids;
    case 2:
    case 1: Qqg[           nq-2] = rA*ids;
            Qqg[           nq-1] = rA*ids;
  }
  switch (nT) {
    case 4: Qqg[(nP+3)*nq+nPq+1] = 3.464101615137754*FBA3*irA*ids;
    case 3: Qqg[(nP+2)*nq+nPq  ] = 2.000000000000000*FBA2*irA*ids;
    case 2:
    case 1: Qqg[(nP  )*nq+ nq-2] =  irA*ids;
            Qqg[(nP  )*nq+ nq-1] = -irA*ids;
  }
  /* See bfef.m for help on last 2 reg. constraints */
}

/* Toroidal magnetic field amplitude */
void FLT_NAME(bfef8)(FLT *Bty, FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ag, FLT rBt, FLT ids, FLT *iry, int nzy, int nry,
                     int nP, int nT, int err) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfef_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct8)(Bty, Fx, FA, FB, Opy,
                  ag, rBt, ids, iry, nzy, nry, err,
                  &params);
}

/* Derivatives of transfer matrix and integrals over plasma domain */
void FLT_NAME(bfef11)(FLT  *dTygdFy, FLT  *dTygdFA, FLT  *dTygdFB,
                                     FLT *dITygdFA, FLT *dITygdFB,
                      FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                      FLT *ry, FLT *iry, int nzy, int nry,
                      int nP, int nT, int err) {
  /* Parameters */
  FLT fPg[nP+nT], fTg[nP+nT];
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, fPg, fTg, &fparams, FUNPARAMS};

  /* Compute fPg and fTg */
  FLT_NAME(bfef_fPg)(nP, nT, fPg, fTg);
  /* Call generic function */
  FLT_NAME(bfct11)(dTygdFy, dTygdFA,  dTygdFB,
                           dITygdFA, dITygdFB,
                   Fx, FA, FB, Opy,
                   ry, iry, nzy, nry, err,
                   &params);
}

/* Derivatives of physical basis functions at magnetic axis */
void FLT_NAME(bfef15)(FLT *dgA1, FLT * dgA2, FLT *dIgA1, FLT *dIgA2,
                      FLT FA, FLT FB, int nP, int nT) {
  /* Parameters */
  bfef_params fparams = {nP, nT};
  FLT_NAME(bfct_params) params = {nP+nT, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct15)(dgA1, dgA2, dIgA1, dIgA2, FA, FB,
                   &params);
}

/* Derivatives of regularization constraints */
void FLT_NAME(bfef16)(FLT *dQqgdFA, FLT *dQqgdFB, FLT *dQqgdrA, FLT *dQqgdirA,
                      FLT FA, FLT FB, FLT rA, FLT irA, FLT ids, int nP, int nT, int nq) {
  int nPq = nP>2 ? nP-2 : 0;
  /* int nTq = nT>2 ? nT-2 : 0; */
  /* int nq = nPq + nTq + 2*((nP+nT)>0); */
  FLT FBA = FB-FA, FBA2 = FBA*FBA, FBA3 = FBA2*FBA;
  /* Arrays init */
  memset(dQqgdFA,0,nq*(nP+nT)*sizeof(FLT));
  memset(dQqgdFB,0,nq*(nP+nT)*sizeof(FLT));
  memset(dQqgdrA,0,nq*(nP+nT)*sizeof(FLT));
  memset(dQqgdirA,0,nq*(nP+nT)*sizeof(FLT));
  /* */
  switch (nP) {
    case 4: dQqgdFA[     3*nq+    1] = -10.39230484541326 *FBA2*rA*ids; /* 3*sqrt(12) */
            dQqgdFB[     3*nq+    1] =  10.39230484541326 *FBA2*rA*ids; /* 3*sqrt(12) */
            dQqgdrA[     3*nq+    1] =   3.464101615137754*FBA3   *ids; /* sqrt(12) */
    case 3: dQqgdFA[     2*nq      ] =  -4.000000000000000*FBA *rA*ids;
            dQqgdFB[     2*nq      ] =   4.000000000000000*FBA *rA*ids;
            dQqgdrA[     2*nq      ] =   2.000000000000000*FBA2   *ids;
    case 2:
    case 1: dQqgdrA[           nq-2] = ids;
            dQqgdrA[           nq-1] = ids;
  }
  switch (nT) {
    case 4: dQqgdFA[(nP+3)*nq+nPq+1] = -10.39230484541326 *FBA2*irA *ids; /* 3*sqrt(12) */
            dQqgdFB[(nP+3)*nq+nPq+1] =  10.39230484541326 *FBA2*irA *ids; /* 3*sqrt(12) */
            dQqgdirA[(nP+3)*nq+nPq+1] =   3.464101615137754*FBA3     *ids; /*   sqrt(12) */
    case 3: dQqgdFA[(nP+2)*nq+nPq  ] =  -4.000000000000000*FBA *irA *ids;
            dQqgdFB[(nP+2)*nq+nPq  ] =   4.000000000000000*FBA *irA *ids;
            dQqgdirA[(nP+2)*nq+nPq  ] =   2.000000000000000*FBA2*     ids;
    case 2:
    case 1: dQqgdirA[(nP  )*nq+ nq-2] = ids;
            dQqgdirA[(nP  )*nq+ nq-1] = -ids;
  }
}
