/* [+MEQ MatlabEQuilibrium Toolbox+]
 *
 *    Copyright 2022-2025 Swiss Plasma Center EPFL
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License. */

# include "meq.h"

/*
 * bfct.c contains the definitions of generic functions to evaluate each
 * "mode" of a basis function set
 *
 * These functions have an argument called params which should be of type
 * "bfct_params" (defined in the header file "bfct.h"). Using the actual
 * type will require the type definitons for all floating-point precisions
 * to be included in meq.h.
 *
 * See also: bfct.h
 *
 */

/*
 * First we have stand-in replacements for functions that are not provided
 * by the basis function set:
 *   - bfct_fag:  conversion factors from normalized to physical basis
 *                functions
 *   - bfct_f:    physical basis functions at any point
 *   - bfct_fA:   physical basis functions at the magnetic axis
 *   - bfct_dfag: derivatives of conversion factors 
 *   - bfct_df:   derivatives of physical basis functions
 *   - bfct_dfA:  derivatives of physical basis functions at the magnetic
 *                axis
 */

/* Conversion factors from normalized to physical basis functions */
void FLT_NAME(bfct_fag)(void *params, FLT *alphapg, FLT *alphag, FLT FBA)
{
  int k;
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;

  /* Set conversion factors */
  for (k=0; k<ng; k++) {
    *alphapg++ = FLTC(1.0);
    *alphag++  = FBA;
  }
}

/* Compute physical basis functions using normalized basis functions */
void FLT_NAME(bfct_f)(void *params, FLT *g, FLT *Ig, FLT FxA, FLT FBA)
{
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*fN)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->fN;
  void (*fag)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->fag;
  FLT alphapg[ng], alphag[ng];

  /* Compute normalized basis functions */
  fN(fparams, g, Ig, FxA/FBA);
  /* Compute conversion factors */
  /* Set to generic function if necessary */
  if (fag == NULL) {
    fag = FLT_NAME(bfct_fag);
    fparams = params;
  }
  fag(fparams, alphapg, alphag, FBA);
  /* Convert to physical basis functions */
  VMUL(ng, g, g,alphapg);
  VMUL(ng,Ig,Ig,alphag );
}

/* Compute physical basis functions at magnetic axis */
void FLT_NAME(bfct_fA)(void *params, FLT *g, FLT *Ig, FLT FBA)
{
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*f)(void *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->f;

  /* Compute basis functions */
  /* Set f to generic function if necessary */
  if (f == NULL) {
    f = FLT_NAME(bfct_f);
    fparams = params;
  }
  f(fparams, g, Ig, FLTC(0.0), FBA);
}

/* Derivatives of conversion factors */
void FLT_NAME(bfct_dfag)(void *params, FLT *dalphapg, FLT *dalphag, FLT FBA)
{
  int k;
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;

  /* Compute derivatives */
  for (k=0; k<ng; k++) {
    *dalphapg++ = FLTC(0.0);
    *dalphag++  = FLTC(1.0);
  }
}

/* Derivatives of physical basis functions */
void FLT_NAME(bfct_df)(void *params, FLT * dgdFxA, FLT * dgdFBA,
                                     FLT *dIgdFxA, FLT *dIgdFBA,
                                     FLT FxA, FLT FBA)
{
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*fN)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->fN;
  void (*fag)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->fag;
  void (*dfN)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->dfN;
  void (*dfag)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->dfag;
  FLT alphapg[ng], alphag[ng], dalphapg[ng], dalphag[ng];
  FLT gN[ng], IgN[ng], dgN[ng], dIgN[ng];
  FLT iFBA = INV(FBA), FN = FxA*iFBA;

  /* Compute normalized basis functions */
  fN(fparams, gN, IgN, FN);
  /* Compute normalized basis functions derivatives */
  dfN(fparams, dgN, dIgN, FN);
  /* Compute conversion factors */
  /* Set to generic function if necessary */
  if (fag == NULL) {
    FLT_NAME(bfct_fag)(params, alphapg, alphag, FBA);
  } else {
    fag(fparams, alphapg, alphag, FBA);
  }
  /* Compute conversion factors derivatives */
  /* Set to generic function if necessary */
  if (fag == NULL) {
    FLT_NAME(bfct_dfag)(params, dalphapg, dalphag, FBA);
  } else {
    dfag(fparams, dalphapg, dalphag, FBA);
  }
  /* Combine to form derivatives of physical basis functions */
  /*  dgdFxA = alphapg/FBA *  dgN */
  SCAL(ng,alphapg, iFBA);
  VMUL(ng, dgdFxA, dgN,alphapg);
  /* dIgdFxA = alphag /FBA * dIgN (Note: this is equal to alphapg * gN or simply g) */
  SCAL(ng,alphag , iFBA);
  VMUL(ng,dIgdFxA,dIgN,alphag );
  /*  dgdFBA = dalphapg *  gN - FN *  dgdFxA */
  VMUL(ng, dgdFBA, gN,dalphapg);
  AXPY(ng, dgdFBA,-FN, dgdFxA);
  /* dIgdFBA = dalphag  * IgN - FN * dIgdFxA */
  VMUL(ng,dIgdFBA,IgN,dalphag );
  AXPY(ng,dIgdFBA,-FN,dIgdFxA);
}

/* Derivatives of physical basis functions at the magnetic axis */
void FLT_NAME(bfct_dfA)(void *params, FLT *dgAdFBA, FLT *dIgAdFBA,
                                      FLT FBA)
{
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*df)(void *, FLT *, FLT *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->df;
  FLT dgAdFxA[ng], dIgAdFxA[ng];

  /* Compute derivative of physical basis functions */
  /* Set df to generic function if necessary */
  if (df == NULL) {
    df = FLT_NAME(bfct_df);
    fparams = params;
  }
  df(fparams, dgAdFxA, dgAdFBA, dIgAdFxA, dIgAdFBA, FLTC(0.0), FBA);
}

/*
 * Next we have generic functions for each of the modes
 */

/* Assignment to profiles and domains */
void FLT_NAME(bfct0)(FLT *fPg, FLT *fTg, FLT *TDg,
                     void *params) {
  FLT one = FLTC(1.0);
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  FLT *fPg_ = ((FLT_NAME(bfct_params) *)params)->fPg;
  FLT *fTg_ = ((FLT_NAME(bfct_params) *)params)->fTg;

  /* Copy fPg, fTg values */
  memcpy(fPg, fPg_, ng*sizeof *fPg);
  memcpy(fTg, fTg_, ng*sizeof *fTg);
  /* Set TDg = 1.0 */
  SET(ng,TDg,one);
}

/* Transfer matrix and integrals over plasma domain */
void FLT_NAME(bfct1)(FLT *Tyg, FLT *Tpg, FLT *ITpg,
                     FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ry, FLT *iry, int nzy, int nry, int err,
                     void *params) {
  int i,j,k,ny = nzy*nry;
  FLT FBA = FB - FA;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  FLT *fPg = ((FLT_NAME(bfct_params) *)params)->fPg;
  FLT *fTg = ((FLT_NAME(bfct_params) *)params)->fTg;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*f)(void *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->f;
  /* Pointers to arrays in return arguments for each basis functions */
  FLT * Tyg_[ng];
  /* Temporary variables for values of basis functions */
  FLT g[ng], Ig[ng], g_, fac;

  /* Set f to generic function if necessary */
  if (f == NULL) {
    f = FLT_NAME(bfct_f);
    fparams = params;
  }
  /* Initialise pointers and arrays */
  if (ng) {
    Tyg_[0] = Tyg;
     Tpg[0] = 0;
    ITpg[0] = 0;
  }
  for (k=1; k<ng; k++) {
    Tyg_[k] = Tyg_[k-1] + ny;
     Tpg[k] = 0;
    ITpg[k] = 0;
  }
  /* First y point */
  Fx += nzy + 3;
  /* Do the work */
  if (err) {
    /* Error flag set, set all values to 0 */
    memset(Tyg, 0, ny * ng * sizeof *Tyg);
  } else {
    for (i=nry; i--; ry++, iry++, Fx+=2)
      for (j=nzy; j--; Fx++)
        if (*Opy++) {
          /* get values of basis functions at this (r,z) point */
          f(fparams, g, Ig, *Fx - FA, FBA);
          /* Multiply by R, 1/R and dispatch to return arguments */
          for (k=0; k<ng; k++) {
            if (fPg[k]) {        /* Part of P'  */
              fac = * ry;
            } else if (fTg[k]) { /* Part of TT' */
              fac = *iry;
            } else {             /* Unassigned basis function */
              fac = 0;
            }
            *(Tyg_[k]++) = g_ = fac *  g[k];
             Tpg[k]     += g_;
            ITpg[k]     +=      fac * Ig[k];
          }
        } else {
          /* Not in plasma domain */
          for (k=0; k<ng; k++) {
            *(Tyg_[k]++) = 0;
          }
        }
  }
}

/* Normalized basis functions */
void FLT_NAME(bfct2)(FLT *gQg, FLT *IgQg, FLT *FNQ, int nQ,
                     void *params) {
  int i,k;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*fN)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->fN;
  /* Pointers to arrays in return arguments for each basis functions */
  FLT * gQg_[ng], *IgQg_[ng];
  /* Temporary variables for values of basis functions */
  FLT g[ng], Ig[ng];

  /* Initialise pointers and arrays */
  if (ng) {
     gQg_[0] =  gQg;
    IgQg_[0] = IgQg;
  }
  for (k=1; k<ng; k++) {
    gQg_[k]  =  gQg_[k-1] + nQ;
    IgQg_[k] = IgQg_[k-1] + nQ;
  }
  /* Do the work */
  for (i=nQ; i--;) {
    /* Evaluate normalized basis functions at this FN point */
    fN(fparams, g, Ig, *FNQ++);
    for (k=0; k<ng; k++) {
      *( gQg_[k])++ =  g[k];
      *(IgQg_[k])++ = Ig[k];
    }
  }
}

/* Coefficients to transform normalized basis functions into Pp/P/TTp/hqT */
void FLT_NAME(bfct3)(FLT *aPpg, FLT *aTTpg, FLT *aPg, FLT *ahqTg, FLT *ag,
                     FLT FA, FLT FB, FLT ids,
                     void *params) {
  int k;
  FLT cP = ids*0.159154943091895, /*   1/2/pi */
      cT = ids*2e-7;              /* mu0/2/pi */
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  FLT *fPg = ((FLT_NAME(bfct_params) *)params)->fPg;
  FLT *fTg = ((FLT_NAME(bfct_params) *)params)->fTg;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*fag)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->fag;
  /* Temporary variables for conversion factors */
  FLT alphapg[ng], alphag[ng];

  /* Set fag to generic function if necessary */
  if (fag == NULL) {
    fag = FLT_NAME(bfct_fag);
    fparams = params;
  }
  /* Get the conversion factors from physical to normalized */
  fag(fparams, alphapg, alphag, FB - FA);
  /* Add in coefficients for P and T */
  for (k=0; k<ng; k++) {
     aPpg[k] = cP*alphapg[k]*ag[k]*fPg[k];
    aTTpg[k] = cT*alphapg[k]*ag[k]*fTg[k];
      aPg[k] = cP* alphag[k]*ag[k]*fPg[k];
    ahqTg[k] = cT* alphag[k]*ag[k]*fTg[k];
  }
}

/* Vertical chord integrals */
void FLT_NAME(bfct4)(FLT *Tdg, FLT *Tgy, FLT *Fx, FLT FA, FLT FB, int8_t *Opy, int nzy, int nry,
                     unsigned int *kd, FLT *fd, int nd,
                     void *params) {
  int i, j, k;
  unsigned int *pkd;
  FLT *w1, *w2, *pf1, *pf2;
  FLT FBA = FB - FA;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*f)(void *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->f;
  /* Temporary variables for values of basis functions */
  FLT g[ng], Ig[ng];

  /* Set f to generic function if necessary */
  if (f == NULL) {
    f = FLT_NAME(bfct_f);
    fparams = params;
  }
  /* First y point */
  Fx += nzy + 3;
  /* Initialize array */
  memset(Tdg,0,nd*ng *sizeof(*Tdg));
  memset(Tgy,0,ng*nry*sizeof(*Tgy));
  /* Do the integrals over vertical chords */
  for (i=nry, w1=Tgy; i--; Fx+=2, w1+=ng)
    for (j=nzy; j--; Fx++)
      if (*Opy++) {
        /* get values of basis functions at this (r,z) point */
        f(fparams, g, Ig, *Fx - FA, FBA);
        /* add their contribution to the current chord */
        for (k=0, w2=w1; k<ng; k++)
          *w2++ += g[k];
      }
  /* Compute their weighted combinations */
  for (i=ng, w1=Tgy, w2=Tgy+ng; i--; w1++, w2++)
    for (j=nd, pkd=kd, pf1=fd, pf2=fd+nd; j--;) {
      k = *pkd++ * ng;
      *Tdg++ = w1[k] * *pf1++ + w2[k] * *pf2++;
    }
}

/* Physical basis functions at magnetic axis */
void FLT_NAME(bfct5)(FLT *gA, FLT *IgA, FLT FA, FLT FB,
                     void *params) {
  /* Unpack params */
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*fA)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->fA;

  /* Set fA to generic function if necessary */
  if (fA == NULL) {
    fA = FLT_NAME(bfct_fA);
    fparams = params;
  }
  /* get values of basis functions at magnetic axis */
  fA(fparams, gA, IgA, FB - FA);
}

/* Regularization constraints */
void FLT_NAME(bfct6)(FLT *Qqg, FLT *Xq, FLT FA, FLT FB, FLT rA, FLT irA, FLT ids, int nq,
                     void *params) {
  /* Not implemented */
}

/* Inequality constraints */
void FLT_NAME(bfct7)(FLT *Qcg, FLT *Xc, FLT FA, FLT FB, int nc,
                     void *params) {
  /* Not implemented */
}

/* Toroidal magnetic field amplitude */
void FLT_NAME(bfct8)(FLT *Bty, FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ag, FLT rBt, FLT ids, FLT *iry, int nzy, int nry, int err,
                     void *params) {
  int i,j,k,ny = nzy*nry;
  FLT FBA = FB - FA;
  FLT fac = ids*2e-7/rBt; /* mu0/2pi/rBt */
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  FLT *fTg = ((FLT_NAME(bfct_params) *)params)->fTg;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*f)(void *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->f;
  /* Temporary variables for values of basis functions */
  FLT g[ng], Ig[ng];

  /* Set f to generic function if necessary */
  if (f == NULL) {
    f = FLT_NAME(bfct_f);
    fparams = params;
  }
  /* First y point */
  Fx += nzy + 3;
  /* Do the work */
  if (err) {
    /* Error flag set, set all values to 0 */
    memset(Bty, 0, ny * sizeof *Bty);
  } else {
    for (i=nry; i--; iry++, Fx+=2)
      for (j=nzy; j--; Fx++, Bty++)
        if (*Opy++) {
          /* get values of basis functions at this (r,z) point */
          f(fparams, g, Ig, *Fx - FA, FBA);
          /* Compute toroidal magnetic field in small diamagnetism approx. */
          *Bty = 0;
          for (k=0; k<ng; k++) {
            if (fTg[k]) { /* Part of TT' */
              *Bty += ag[k]*Ig[k];
            }
          }
          *Bty = (rBt + fac*(*Bty)) * *iry;
        } else {
          /* Not in plasma domain */
          *Bty = rBt * *iry;
        }
  }
}

/* Derivatives of Transfer matrix and integrals over plasma domain */
void FLT_NAME(bfct11)(FLT  *dTygdFy, FLT  *dTygdFA, FLT  *dTygdFB,
                                     FLT *dITygdFA, FLT *dITygdFB, 
                      FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                      FLT *ry, FLT *iry, int nzy, int nry, int err,
                      void *params) {
  int i,j,k,ny = nzy*nry;
  FLT FBA = FB - FA;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  FLT *fPg = ((FLT_NAME(bfct_params) *)params)->fPg;
  FLT *fTg = ((FLT_NAME(bfct_params) *)params)->fTg;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*df)(void *, FLT *, FLT *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->df;
  /* Pointers to arrays in return arguments for each basis functions */
  FLT * dTygdFy_[ng], * dTygdFA_[ng], * dTygdFB_[ng],
      *dITygdFA_[ng], *dITygdFB_[ng];
  /* Temporary variables for values of basis functions derivatives */
  FLT dgdFxA[ng], dgdFBA[ng], dIgdFxA[ng], dIgdFBA[ng], fac;

  /* Set df to generic function if necessary */
  if (df == NULL) {
    df = FLT_NAME(bfct_df);
    fparams = params;
  }
  /* Initialise pointers and arrays */
  if (ng) {
     dTygdFy_[0] =  dTygdFy;
     dTygdFA_[0] =  dTygdFA;
     dTygdFB_[0] =  dTygdFB;
    dITygdFA_[0] = dITygdFA;
    dITygdFB_[0] = dITygdFB;
  }
  for (k=1; k<ng; k++) {
     dTygdFy_[k] =  dTygdFy_[k-1] + ny;
     dTygdFA_[k] =  dTygdFA_[k-1] + ny;
     dTygdFB_[k] =  dTygdFB_[k-1] + ny;
    dITygdFA_[k] = dITygdFA_[k-1] + ny;
    dITygdFB_[k] = dITygdFB_[k-1] + ny;
  }
  /* First y point */
  Fx += nzy + 3;
  /* Do the work */
  if (err) {
    /* Error flag set, set all values to 0 */
    memset( dTygdFy, 0, ny * ng * sizeof * dTygdFy);
    memset( dTygdFA, 0, ny * ng * sizeof * dTygdFA);
    memset( dTygdFB, 0, ny * ng * sizeof * dTygdFB);
    memset(dITygdFA, 0, ny * ng * sizeof *dITygdFA);
    memset(dITygdFB, 0, ny * ng * sizeof *dITygdFB);
  } else {
    for (i=nry; i--; ry++, iry++, Fx+=2)
      for (j=nzy; j--; Fx++)
        if (*Opy++) {
          /* get values of basis functions derivatives at this (r,z) point */
          df(fparams, dgdFxA, dgdFBA, dIgdFxA, dIgdFBA, *Fx - FA, FBA);
          /* Multiply by R, 1/R and dispatch to return arguments */
          for (k=0; k<ng; k++) {
            if (fPg[k]) {        /* Part of P'  */
              fac = * ry;
            } else if (fTg[k]) { /* Part of TT' */
              fac = *iry;
            } else {             /* Unassigned basis function */
              fac = 0;
            }
            *( dTygdFy_[k]++) =  fac *   dgdFxA[k];
            *( dTygdFA_[k]++) = -fac * ( dgdFxA[k] +  dgdFBA[k]);
            *( dTygdFB_[k]++) =  fac *   dgdFBA[k];
            *(dITygdFA_[k]++) = -fac * (dIgdFxA[k] + dIgdFBA[k]);
            *(dITygdFB_[k]++) =  fac *  dIgdFBA[k];
          }
        } else {
          /* Not in plasma domain */
          for (k=0; k<ng; k++) {
            *( dTygdFy_[k]++) = 0;
            *( dTygdFA_[k]++) = 0;
            *( dTygdFB_[k]++) = 0;
            *(dITygdFA_[k]++) = 0;
            *(dITygdFB_[k]++) = 0;
          }
        }
  }
}

/* Derivatives of normalized basis functions */
void FLT_NAME(bfct12)(FLT * dgQgdFN, FLT * dgQgdF0, FLT * dgQgdF1,
		      FLT *dIgQgdFN, FLT *dIgQgdF0, FLT *dIgQgdF1,
                      FLT *FNQ, int nQ, void *params) {
  int i,k;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*dfN)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->dfN;
  /* Pointers to arrays in return arguments for each basis functions */
  FLT * dgQgdFN_[ng], * dgQgdF0_[ng], * dgQgdF1_[ng],
      *dIgQgdFN_[ng], *dIgQgdF0_[ng], *dIgQgdF1_[ng];
  /* Temporary variables for values of basis functions derivatives */
  FLT dg[ng], dIg[ng];

  /* Initialise pointers and arrays */
  if (ng) {
     dgQgdFN_[0] =  dgQgdFN;
     dgQgdF0_[0] =  dgQgdF0;
     dgQgdF1_[0] =  dgQgdF1;
    dIgQgdFN_[0] = dIgQgdFN;
    dIgQgdF0_[0] = dIgQgdF0;
    dIgQgdF1_[0] = dIgQgdF1;
  }
  for (k=1; k<ng; k++) {
    dgQgdFN_[k]  =  dgQgdFN_[k-1] + nQ;
    dgQgdF0_[k]  =  dgQgdF0_[k-1] + nQ;
    dgQgdF1_[k]  =  dgQgdF1_[k-1] + nQ;
    dIgQgdFN_[k] = dIgQgdFN_[k-1] + nQ;
    dIgQgdF0_[k] = dIgQgdF0_[k-1] + nQ;
    dIgQgdF1_[k] = dIgQgdF1_[k-1] + nQ;
  }
  /* Do the work */
  for (i=nQ; i--;) {
    /* Evaluate normalized basis functions derivatives at this FN point
     * For single domain basis functions, the derivatives with respect to
     * F0 and F1 are zero */
    dfN(fparams, dg, dIg, *FNQ++);
    for (k=0; k<ng; k++) {
      *( dgQgdFN_[k])++ =  dg[k];
      *( dgQgdF0_[k])++ =      0;
      *( dgQgdF1_[k])++ =      0;
      *(dIgQgdFN_[k])++ = dIg[k];
      *(dIgQgdF0_[k])++ =      0;
      *(dIgQgdF1_[k])++ =      0;
    }
  }
}

/* Derivatives of coefficients to transform normalized basis functions into Pp/P/TTp/hqT */
void FLT_NAME(bfct13)(FLT *daPpgdFA, FLT *daTTpgdFA, FLT *daPgdFA, FLT *dahqTgdFA,
                      FLT *daPpgdFB, FLT *daTTpgdFB, FLT *daPgdFB, FLT *dahqTgdFB,
                      FLT *ag, FLT FA, FLT FB, FLT ids,
                      void *params) {
  int k;
  FLT cP = ids*0.159154943091895, /*   1/2/pi */
      cT = ids*2e-7;              /* mu0/2/pi */
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  FLT *fPg = ((FLT_NAME(bfct_params) *)params)->fPg;
  FLT *fTg = ((FLT_NAME(bfct_params) *)params)->fTg;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*dfag)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->dfag;
  /* Temporary variables for conversion factors */
  FLT dalphapg[ng], dalphag[ng];

  /* Set fag to generic function if necessary */
  if (dfag == NULL) {
    dfag = FLT_NAME(bfct_dfag);
    fparams = params;
  }
  /* Get the conversion factors from physical to normalized */
  dfag(fparams, dalphapg, dalphag, FB - FA);
  /* Add in coefficients for P and T */
  for (k=0; k<ng; k++) {
     daPpgdFB[k] = cP*dalphapg[k]*ag[k]*fPg[k];
    daTTpgdFB[k] = cT*dalphapg[k]*ag[k]*fTg[k];
      daPgdFB[k] = cP* dalphag[k]*ag[k]*fPg[k];
    dahqTgdFB[k] = cT* dalphag[k]*ag[k]*fTg[k];
     daPpgdFA[k] = - daPpgdFB[k];
    daTTpgdFA[k] = -daTTpgdFB[k];
      daPgdFA[k] = -  daPgdFB[k];
    dahqTgdFA[k] = -dahqTgdFB[k];
  }
}

/* Derivatives of Physical basis functions at magnetic axis */
void FLT_NAME(bfct15)(FLT *dgA1, FLT *dgA2, FLT *dIgA1, FLT *dIgA2,
                      FLT FA, FLT FB, void *params) {
  int k;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*dfA)(void *, FLT *, FLT *, FLT) = ((FLT_NAME(bfct_params) *)params)->dfA;
  /* Temporary variables for values of basis functions derivatives */
  FLT dgA[ng], dIgA[ng];

  /* Set dfA to generic function if necessary */
  if (dfA == NULL) {
    dfA = FLT_NAME(bfct_dfA);
    fparams = params;
  }
  /* get derivatives of basis functions at magnetic axis */
  dfA(fparams, dgA, dIgA, FB - FA);

  /* Convert to derivatives with respect to FA and FB */
  for (k = 0; k < ng; k++) {
    *( dgA1++) = - dgA[k];
    *( dgA2++) = + dgA[k];
    *(dIgA1++) = -dIgA[k];
    *(dIgA2++) = +dIgA[k];
  }
}

/* Derivatives of regularization constraints */
void FLT_NAME(bfct16)(FLT *dQqgdFA, FLT *dQqgdFB, FLT *dQqgdrA, FLT *dQqgdirA,
                      FLT FA, FLT FB, FLT rA, FLT irA, FLT ids, int nq,
                      void *params) {
  /* Not implemented */
}

/* Values of physical basis functions */
void FLT_NAME(bfct91)(FLT *g, FLT *Ig, FLT * F, FLT FA, FLT FB, int nF,
                      void *params) {
  int i, k;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*f)(void *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->f;
  /* Pointers to arrays in return arguments for each basis functions */
  FLT * g_[ng], * Ig_[ng];
  /* Temporary variables for values of basis functions */
  FLT g1[ng], Ig1[ng];

  /* Set f to generic function if necessary */
  if (f == NULL) {
    f = FLT_NAME(bfct_f);
    fparams = params;
  }
  /* Initialise pointers and arrays */
  if (ng) {
     g_[0] =  g;
    Ig_[0] = Ig;
  }
  for (k=1; k<ng; k++) {
     g_[k] =  g_[k-1] + nF;
    Ig_[k] = Ig_[k-1] + nF;
  }
  /* get physical basis functions values */
  for (i=nF; i--; F++) {
    f(fparams, g1, Ig1, *F -FA, FB - FA);
    for (k=0; k<ng; k++) {
      *( g_[k]++) =  g1[k];
      *(Ig_[k]++) = Ig1[k];
    }
  }
}

/* Derivatives of physical basis functions */
void FLT_NAME(bfct92)(FLT  *dgx, FLT  *dgA, FLT  *dgB,
                      FLT *dIgx, FLT *dIgA, FLT *dIgB,
                      FLT * F, FLT FA, FLT FB, int nF,
                      void *params) {
  int i, k;
  FLT FBA = FB - FA;
  /* Unpack params */
  int ng = ((FLT_NAME(bfct_params) *)params)->ng;
  void *fparams = ((FLT_NAME(bfct_params) *)params)->fparams;
  void (*df)(void *, FLT *, FLT *, FLT *, FLT *, FLT, FLT) = ((FLT_NAME(bfct_params) *)params)->df;
  /* Pointers to arrays in return arguments for each basis functions */
  FLT * dgx_[ng], * dgA_[ng], * dgB_[ng],
      *dIgx_[ng], *dIgA_[ng], *dIgB_[ng];
  /* Temporary variables for values of basis functions derivatives*/
  FLT dgdFxA[ng], dgdFBA[ng], dIgdFxA[ng], dIgdFBA[ng];

  /* Set df to generic function if necessary */
  if (df == NULL) {
    df = FLT_NAME(bfct_df);
    fparams = params;
  }
  /* Initialise pointers and arrays */
  if (ng) {
     dgx_[0] =  dgx;
     dgA_[0] =  dgA;
     dgB_[0] =  dgB;
    dIgx_[0] = dIgx;
    dIgA_[0] = dIgA;
    dIgB_[0] = dIgB;
  }
  for (k=1; k<ng; k++) {
     dgx_[k] =  dgx_[k-1] + nF;
     dgA_[k] =  dgA_[k-1] + nF;
     dgB_[k] =  dgB_[k-1] + nF;
    dIgx_[k] = dIgx_[k-1] + nF;
    dIgA_[k] = dIgA_[k-1] + nF;
    dIgB_[k] = dIgB_[k-1] + nF;
  }
  /* get physical basis functions derivatives */
  for (i=nF; i--; F++) {
    df(fparams, dgdFxA, dgdFBA, dIgdFxA, dIgdFBA, *F -FA, FBA);
    for (k=0; k<ng; k++) {
      *( dgx_[k]++) =    dgdFxA[k];
      *( dgA_[k]++) = -( dgdFxA[k] +  dgdFBA[k]);
      *( dgB_[k]++) =    dgdFBA[k];
      *(dIgx_[k]++) =   dIgdFxA[k];
      *(dIgA_[k]++) = -(dIgdFxA[k] + dIgdFBA[k]);
      *(dIgB_[k]++) =   dIgdFBA[k];
    }
  }
}
