/* [+MEQ MatlabEQuilibrium Toolbox+]
 *
 *    Copyright 2022-2025 Swiss Plasma Center EPFL
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License. */


/*
 * bf3i.c contains the definiton of the bf3i basis function set
 *
 * The bf3i set corresponds to the linear interpolation of a set of three
 * functions of the normalized poloidal flux defined on a regular grid.
 *
 * See also bfct.c
 */

# include "meq.h"

/* List of function pointers to use when creating the bfct_params structure */
#define FUNPARAMS FLT_NAME(bf3i_f) , FLT_NAME(bf3i_fN) , FLT_NAME(bf3i_fag) , FLT_NAME(bf3i_fA) , \
                  FLT_NAME(bf3i_df), FLT_NAME(bf3i_dfN), FLT_NAME(bf3i_dfag), FLT_NAME(bf3i_dfA)

/*
 * First we define some elementary functions to evaluate the necessary 
 * quantities for all basis functions.
 * These functions are the ones used in the generic bfct functions.
 */

/* Compute normalized basis functions */
void FLT_NAME(bf3i_fN)(void *params, FLT gN[3], FLT IgN[3], FLT FN)
{
  FLT f, f1, idF, *gNg, *IgNg;
  int nN, nN2;
  unsigned int k, k1;

  gNg  = ((FLT_NAME(bf3i_params) *) params)->gNg;
  IgNg = ((FLT_NAME(bf3i_params) *) params)->IgNg;
  nN   = ((FLT_NAME(bf3i_params) *) params)->nN;
  nN2 = nN-2;
  /* determine cell and interpolation factor */
  idF = (nN-1);
  f = FN * idF;
  k = f;
  if (k > nN2) k = nN2;
  k1 = k+1;
  f = f - k; f1 = FLTC(1.0) - f;
  /* interpolate */
   gN[0] =  gNg[k     ]*f1 +  gNg[k1     ]*f;
   gN[1] =  gNg[k+  nN]*f1 +  gNg[k1+  nN]*f;
   gN[2] =  gNg[k+2*nN]*f1 +  gNg[k1+2*nN]*f;
  /* Exactly integrate linearly interpolated function */
  idF = -f1*INV(idF); f1 = 0.5*f1; f = 0.5*(1+f);
  IgN[0] = IgNg[k1     ] + idF *(gNg[k     ]*f1 + gNg[k1     ]*f);
  IgN[1] = IgNg[k1+  nN] + idF *(gNg[k+  nN]*f1 + gNg[k1+  nN]*f);
  IgN[2] = IgNg[k1+2*nN] + idF *(gNg[k+2*nN]*f1 + gNg[k1+2*nN]*f);
}

/* Conversion factors from normalized to physical basis functions */
void FLT_NAME(bf3i_fag)(void *params, FLT alphapg[3], FLT alphag[3], FLT FBA)
{
  alphapg[0] = 1.0;
  alphapg[1] = 1.0;
  alphapg[2] = 1.0;
   alphag[0] = FBA;
   alphag[1] = FBA;
   alphag[2] = FBA;
}

/* Compute physical basis functions */
void FLT_NAME(bf3i_f)(void *params, FLT g[3], FLT Ig[3], FLT FxA, FLT FBA)
{
  /* Compute normalized basis functions */
  FLT_NAME(bf3i_fN)(params, g, Ig, FxA/FBA);
  /* Apply scaling for Ig */
  Ig[0] = Ig[0]*FBA;
  Ig[1] = Ig[1]*FBA;
  Ig[2] = Ig[2]*FBA;
}

/* Compute physical basis functions at magnetic axis */
void FLT_NAME(bf3i_fA)(void *params, FLT g[3], FLT Ig[3], FLT FBA)
{
  /* Compute normalized basis functions */
  FLT_NAME(bf3i_fN)(params, g, Ig, FLTC(0.0));
  /* Apply scaling for Ig */
  Ig[0] = Ig[0]*FBA;
  Ig[1] = Ig[1]*FBA;
  Ig[2] = Ig[2]*FBA;
}

/* Compute derivatives of normalized basis functions */
void FLT_NAME(bf3i_dfN)(void *params, FLT dgN[3], FLT dIgN[3], FLT FN)
{
  FLT f, f1, idF, *gNg;
  int nN, nN2;
  unsigned int k, k1;

  gNg  = ((FLT_NAME(bf3i_params) *) params)->gNg;
  nN   = ((FLT_NAME(bf3i_params) *) params)->nN;
  nN2 = nN-2;
  /* determine cell */
  idF = (nN-1);
  f = FN * idF;
  k = f;
  if (k > nN2) k = nN2;
  k1 = k+1;
  /* assign */
   dgN[0] = ( gNg[k1     ] -  gNg[k     ])*idF;
   dgN[1] = ( gNg[k1+  nN] -  gNg[k+  nN])*idF;
   dgN[2] = ( gNg[k1+2*nN] -  gNg[k+2*nN])*idF;
  /* interpolate (dIgN=gN) */
  f = f - k; f1 = FLTC(1.0) - f;
  dIgN[0] =  gNg[k     ]*f1 +  gNg[k1     ]*f;
  dIgN[1] =  gNg[k+  nN]*f1 +  gNg[k1+  nN]*f;
  dIgN[2] =  gNg[k+2*nN]*f1 +  gNg[k1+2*nN]*f;
}

/* Derivatives of conversion factors from normalized to physical basis functions */
void FLT_NAME(bf3i_dfag)(void *params, FLT dalphapg[3], FLT dalphag[3], FLT FBA)
{
  dalphapg[0] = 0.0;
  dalphapg[1] = 0.0;
  dalphapg[2] = 0.0;
   dalphag[0] = 1.0;
   dalphag[1] = 1.0;
   dalphag[2] = 1.0;
}

/* Compute derivatives of physical basis functions */
void FLT_NAME(bf3i_df)(void *params, FLT  d1g[3], FLT  d2g[3],
                                     FLT d1Ig[3], FLT d2Ig[3],
                                     FLT FxA, FLT FBA)
{
  FLT gN[3], IgN[3], dgN[3], dIgN[3];
  FLT iFBA = INV(FBA), FN = FxA*iFBA;
  /* Compute derivatives of normalized basis functions */
  FLT_NAME(bf3i_dfN)(params, dgN, dIgN, FN);
  /* Compute normalized basis functions */
  FLT_NAME(bf3i_fN) (params,  gN,  IgN, FN);
  /* Combine to form derivatives of physical basis functions */
  /*  dgdFxA = 1/FBA *  dgN */
  d1g[0]  = dgN[0]*iFBA;
  d1g[1]  = dgN[1]*iFBA;
  d1g[2]  = dgN[2]*iFBA;
  /* dIgdFxA = dIgN (Note: this is equal to gN) */
  d1Ig[0] =  gN[0];
  d1Ig[1] =  gN[1];
  d1Ig[2] =  gN[2];
  /*  dgdFBA = - FN *  dgdFxA */
  d2g[0]  = d1g[0]*-FN;
  d2g[1]  = d1g[1]*-FN;
  d2g[2]  = d1g[2]*-FN;
  /* dIgdFBA = IgN - FN * dIgdFxA (Note: this is equal to IgN - FN * gN */
  d2Ig[0] = IgN[0] -FN*gN[0];
  d2Ig[1] = IgN[1] -FN*gN[1];
  d2Ig[2] = IgN[2] -FN*gN[2];
}

/* Compute derivatives of physical basis functions at magnetic axis */
void FLT_NAME(bf3i_dfA)(void *params, FLT dgA[3], FLT dIgA[3], FLT FBA)
{
  /* Compute normalized basis functions */
  FLT_NAME(bf3i_fN)(params, dgA, dIgA, FLTC(0.0));
  /* dgA = 0 */
  dgA[0] = 0;
  dgA[1] = 0;
  dgA[2] = 0;
  /* dIgA = IgN(0) */
}

/*
 * These functions are the ones used in Simulink S-function blocks for each
 * mode and in MEX-files when a custom implementation is present or needed
 *  - modes 1-5, 8: the generic bfct functions are called
 *  - mode 6 (regularisation constraints) has a custom implementation
 *  - mode 7 (inequality constraints) is not implemented
 */

/* Transfer matrix and integrals over plasma domain */
void FLT_NAME(bf3i1)(FLT *Tyg, FLT *Tpg, FLT *ITpg, FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ry, FLT *iry, int nzy, int nry,
                     FLT *gNg, FLT *IgNg, FLT *fPg, FLT *fTg, int err, int nN) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {gNg, IgNg, nN};
  FLT_NAME(bfct_params) params = {3, fPg, fTg, &fparams, FUNPARAMS}; 

  /* Call generic function */
  FLT_NAME(bfct1)(Tyg, Tpg, ITpg,
                 Fx, FA, FB, Opy,
                 ry, iry, nzy, nry, err,
                 &params);
}

/* Normalized basis functions */
void FLT_NAME(bf3i2)(FLT *gQg, FLT *IgQg, FLT *FNQ, int nQ,
                     FLT *gNg, FLT *IgNg, int nN) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {gNg, IgNg, nN};
  FLT_NAME(bfct_params) params = {3, NULL, NULL, &fparams, FUNPARAMS}; 

  /* Call generic function */
  FLT_NAME(bfct2)(gQg, IgQg, FNQ, nQ,
                  &params);
}

/* Coefficients to transform normalized basis functions into Pp/P/TTp/hqT */
void FLT_NAME(bf3i3)(FLT *aPpg, FLT *aTTpg, FLT *aPg, FLT *ahqTg, FLT *ag,
                     FLT FA, FLT FB, FLT *fPg, FLT *fTg, FLT ids) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {NULL, NULL, 0};
  FLT_NAME(bfct_params) params = {3, fPg, fTg, &fparams, FUNPARAMS}; 

  /* Call generic function */
  FLT_NAME(bfct3)(aPpg, aTTpg, aPg, ahqTg, ag,
                  FA, FB, ids,
                  &params);
}

/* Vertical chord integrals */
void FLT_NAME(bf3i4)(FLT *Tdg, FLT *Tgy, FLT *Fx, FLT FA, FLT FB, int8_t *Opy, int nzy, int nry,
                     unsigned int *kd, FLT *fd, int nd, FLT *gNg, int nN, int ng) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {gNg, gNg, nN};
  FLT_NAME(bfct_params) params = {3, NULL, NULL, &fparams, FUNPARAMS}; 
  /* This does not depend on IgNg, but passing NULL would cause a crash
   * as bf3i would still evaluate both the functions and their integrals */

  /* Call generic function */
  FLT_NAME(bfct4)(Tdg, Tgy, Fx, FA, FB, Opy, nzy, nry,
                  kd, fd, nd,
                  &params);
}

/* Physical basis functions at magnetic axis */
void FLT_NAME(bf3i5)(FLT *gNg, FLT *IgNg, FLT FA, FLT FB, int ng, int nN,
                     FLT *gA, FLT *IgA) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {gNg, IgNg, nN};
  FLT_NAME(bfct_params) params = {3, NULL, NULL, &fparams, FUNPARAMS}; 

  /* Call generic function */
  FLT_NAME(bfct5)(gA, IgA, FA, FB,
                  &params);
}

/* Regularization constraints */
void FLT_NAME(bf3i6)(FLT *Qg, FLT *Xq) {
  Qg[0] = Qg[1] = Qg[2] = Xq[0] = FLTC(0.0);
}

/* Toroidal magnetic field amplitude */
void FLT_NAME(bf3i8)(FLT *Bty, FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                     FLT *ag, FLT rBt, FLT ids, FLT *iry, int nzy, int nry,
                     FLT *gNg, FLT *IgNg, FLT *fPg, FLT *fTg, int err, int nN) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {gNg, IgNg, nN};
  FLT_NAME(bfct_params) params = {3, fPg, fTg, &fparams, FUNPARAMS}; 

  /* Call generic function */
  FLT_NAME(bfct8)(Bty, Fx, FA, FB, Opy,
                  ag, rBt, ids, iry, nzy, nry, err,
                  &params);
}

/* Derivatives of transfer matrix and integrals over plasma domain */
void FLT_NAME(bf3i11)(FLT  *dTygdFy, FLT  *dTygdFA, FLT  *dTygdFB,
                                     FLT *dITygdFA, FLT *dITygdFB,
                      FLT *Fx, FLT FA, FLT FB, int8_t *Opy,
                      FLT *ry, FLT *iry, int nzy, int nry,
                      FLT *gNg, FLT *IgNg, FLT *fPg, FLT *fTg, int err, int nN) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {gNg, IgNg, nN};
  FLT_NAME(bfct_params) params = {3, fPg, fTg, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct11)(dTygdFy, dTygdFA,  dTygdFB,
                           dITygdFA, dITygdFB,
                   Fx, FA, FB, Opy,
                   ry, iry, nzy, nry, err,
                   &params);
}

/* Derivatives of physical basis functions at magnetic axis */
void FLT_NAME(bf3i15)(FLT *dgA1, FLT * dgA2, FLT *dIgA1, FLT *dIgA2,
                      FLT FA, FLT FB, FLT *gNg, FLT *IgNg, int nN) {
  /* Parameters */
  FLT_NAME(bf3i_params) fparams = {gNg, IgNg, nN};
  FLT_NAME(bfct_params) params = {3, NULL, NULL, &fparams, FUNPARAMS};

  /* Call generic function */
  FLT_NAME(bfct15)(dgA1, dgA2, dIgA1, dIgA2, FA, FB,
                   &params);
}

/* Derivatives of regularization constraints */
void FLT_NAME(bf3i16)(FLT *dQqgdFA, FLT *dQqgdFB, FLT *dQqgdrA, FLT *dQqgdirA) {
  dQqgdFA[0] = dQqgdFA[1] = dQqgdFA[2] = FLTC(0.0);
  dQqgdFB[0] = dQqgdFB[1] = dQqgdFB[2] = FLTC(0.0);
  dQqgdrA[0] = dQqgdrA[1] = dQqgdrA[2] = FLTC(0.0);
  dQqgdirA[0] = dQqgdirA[1] = dQqgdirA[2] = FLTC(0.0);
}
