/* [+MEQ MatlabEQuilibrium Toolbox+]
 *
 *    Copyright 2022-2025 Swiss Plasma Center EPFL
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License. */

# ifndef MEQH_H

# define MEQH_H

# include <stdint.h>
# include <math.h>
# include <stdlib.h>
# include <stdio.h>
# include <string.h>
# include <sys/time.h>

/* Types */
# ifdef SINGLE
# define FLT float
# define FLT_NAME(name) s ## name
# define FLT_CBLAS(name) cblas_s ## name
# define FLT_LAPACKE(name) LAPACKE_s ## name
# define FLT_BLAS(name) s ## name
# define FLT_LAPACK(name) s ## name
# define FLT_VBLAS(name) vs ## name
# define ABS fabsf
# define SIGN copysignf
# define MEQ_HUGE HUGE_VALF
# else
# define FLT double
# define FLT_NAME(name) d ## name
# define FLT_CBLAS(name) cblas_d ## name
# define FLT_LAPACKE(name) LAPACKE_d ## name
# define FLT_BLAS(name) d ## name
# define FLT_LAPACK(name) d ## name
# define FLT_VBLAS(name) vd ## name
# define ABS fabs
# define SIGN copysign
# define MEQ_HUGE HUGE_VAL
# endif
# define FLTC(a) ((FLT)(a))
/* Define a type equivalent to bool
 * Altough on Linux MATLAB's mxLogical is identical to bool, 
 * this is not true for all platform and also not for Octave
 * but most importantly Simulink's boolean_T type is not the
 * same as mxLogical and is unsigned char instead.
 * The following solution should work for our limited use of
 * boolean types */
typedef unsigned char meq_bool;
# define meq_true ((meq_bool)(1))
# define meq_false ((meq_bool)(0))

/* Types for generic basis function handling */
# include "bfct.h"

/* Math */
# if defined CBLAS
 # include <cblas.h>
 # include <vblas.h>
 # include <lapacke.h>
# elif defined MKL
 # include <mkl.h>
# else
 # include <matmkl.h>
 # include <vblas.h>
# endif
# define INV(x) ( (x == FLTC(0.0)) ? FLTC(0.0) : FLTC(1.0) / (x) )
# define COPY(n,y,x)         FLT_CBLAS(copy)(n,x,1,y,1)                                                       /* y = x          */
# define SET(n,y,a)          FLT_CBLAS(copy)(n,&a,0,y,1)                                                      /* y = a          */
# define SCAL(n,x,a)         FLT_CBLAS(scal)(n,a,x,1)                                                         /* x =  a*x       */
# define AXPY(n,y,a,x)       FLT_CBLAS(axpy)(n,a,x,1,y,1)                                                     /* y =  a*x + y   */
# define MVM(y,A,x,m,n)      FLT_CBLAS(gemv)(CblasColMajor,CblasNoTrans,m,n,FLTC(-1.0),A,m,x,1,FLTC(1.0),y,1) /* y = -A*x + y   */
# define GEMV(y,A,x,m,n)     FLT_CBLAS(gemv)(CblasColMajor,CblasNoTrans,m,n,FLTC( 1.0),A,m,x,1,FLTC(1.0),y,1) /* y =  A*x + y   */
# define SPMV(n,u,x,y)       FLT_CBLAS(spmv)(CblasColMajor,CblasUpper,n,1.0,u,x,1,1.0,y,1)                    /* y = U*X+y      */
# define GNMV(m,n,a,A,x,b,y) FLT_CBLAS(gemv)(CblasColMajor,CblasNoTrans,m,n,a,A,m,x,1,b,y,1)                  /* y = a*A *x+b*y */
# define GTMV(m,n,a,A,x,b,y) FLT_CBLAS(gemv)(CblasColMajor,CblasTrans  ,m,n,a,A,m,x,1,b,y,1)                  /* y = a*A'*x+b*y */
# define PTRF(n,u)           FLT_LAPACKE(pptrf)(LAPACK_COL_MAJOR,'U',n,u)                                     /* U = chol(U)    */
# define PTRS(n,u,x)         FLT_LAPACKE(pptrs)(LAPACK_COL_MAJOR,'U',n,1,u,x,n)                               /* x = U\(U'\x)   */
# define VADD(n,z,x,y)       FLT_VBLAS(Add)(n,x,y,z)                                                          /* z = x+y        */
# define VSUB(n,z,x,y)       FLT_VBLAS(Sub)(n,x,y,z)                                                          /* z = x-y        */
# define VMUL(n,z,x,y)       FLT_VBLAS(Mul)(n,x,y,z)                                                          /* z = x.*y       */
# define VDIV(n,z,x,y)       FLT_VBLAS(Div)(n,x,y,z)                                                          /* z = x./y       */
# define TWOPI               FLTC( 6.28318530717958623)                                                       /* 2*pi           */
# define HIPI                FLTC( 0.15915494309189535)                                                       /* 1/(2*pi)       */
# define ITHREE              FLTC( 0.33333333333333333)                                                       /* 1/3            */
# define TWOITHREE           FLTC( 0.66666666666666666)                                                       /* 2/3            */
# define FOURITHREE          FLTC( 1.33333333333333333)                                                       /* 4/3            */
# define IMU0                FLTC( 7.95774715459476690E+05)                                                   /* 1/mu0          */

/* Mex */
# if defined MATLAB_MEX_FILE || defined OCTAVE_MEX_FILE
 /* Necessary for MEX-file compilation */
 # include "mex.h"

 /* All checks now defined in genlib/mexargs */
 # include "../genlib/mexargs/include/mexargs.h"

# endif

/* printf */
/* Allows printing to terminal or MATLAB desktop */
# define printf mexPrintf

int mexPrintf(
    const char  *fmt, /* printf style format */
    ...       /* any additional arguments */
    );

# endif 
