/* [+MEQ MatlabEQuilibrium Toolbox+]
 *
 *    Copyright 2022-2025 Swiss Plasma Center EPFL
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License. */


/*
 * bfctmex.h contains function definitons for use in MEX-files for basis
 * function sets
 *
 * The list of implemented modes is as follows:
 *  [fPg,fTg,TDg          ] = bf*mex( 0,par                              )
 *  [Tyg,TpDg,ITpDg       ] = bf*mex( 1,par, Fx,F0,F1,Opy, ry,iry        )
 *  [gQDg,IgQDg           ] = bf*mex( 2,par,FNQ,F0,F1                    )
 *  [aPpg,aTTpg,aPg,ahqTg ] = bf*mex( 3,par, ag,F0,F1,fPg,fTg,ids        )
 *  [TDg,Tgy              ] = bf*mex( 4,par, Fx,F0,F1,Opy, KD, FD        )
 *  [g0g,Ig0g             ] = bf*mex( 5,par, ~ ,F0,F1                    )
 *  [Qqg,Xq               ] = bf*mex( 6,par, ~ ,F0,F1, r0,ir0,ids        )
 *  [Qcg,Xc               ] = bf*mex( 7,par, ~ ,F0,F1                    )
 *  [Bty                  ] = bf*mex( 8,par, Fx,F0,F1,Opy, ag,rBt,ids,iry)
 *  [dgy,dg0,dg1,dIg0,dIg1] = bf*mex(11,par, Fx,F0,F1,Opy, ry,iry        )
 *  [dgQDg,... ,dIgQDg,...] = bf*mex(12,par,FNQ,F0,F1                    )
 *  [daPp,daTTp,daP,dahqT ] = bf*mex(13,par, ag,F0,F1,fPg,fTg,ids        )
 *  [dg0g,dIg0g           ] = bf*mex(15,par, ~ ,F0,F1                    )
 *  [dQgF0,dQgF1,dQgdr0...] = bf*mex(16,par, ~ ,F0,F1, r0,ir0,ids        )
 *  [dQcgdF0,dQcgdF1      ] = bf*mex(17,par, ~ ,F0,F1                    )
 *  [g,Ig                 ] = bf*mex(91,par, F ,F0,F1                    )
 *  [dgdFx,...,dIgdFx,... ] = bf*mex(92,par, F ,F0,F1                    )
 *  [y1,y2,y3,y4          ] = bf*mex(##,par, u1,F0,F1, q1, q2, q3, q4, q5)
 *
 * Examples:
 *  - bfabmex.c, bfefmex.c and bf3imex.c make full use of these generic
 * functions with a few customizations for modes 6 and 7.
 *  - bf3pmex.c only uses the functions for arguments checks and
 * allocations
 *
 * See also bfct.c bfct.m bfhelp.m
 */

# include "meq.h"

# define NARGOUT_MAX 8

# define y1    pout[0]
# define y2    pout[1]
# define y3    pout[2]
# define y4    pout[3]
# define y5    pout[4]
# define y6    pout[5]
# define y7    pout[6]
# define y8    pout[7]

# define MODE  prhs[0]  /* numeric, scalar       */
# define par   prhs[1]  /* double, size=2        */
# define u1    prhs[2]  /* ... see checks     ...*/
# define FA    prhs[3]  /* ... for remaining ... */
# define FB    prhs[4]  /* ... arguments in  ... */
# define q1    prhs[5]  /* ... separate mode ... */
# define q2    prhs[6]  /* ... blocks        ... */
# define q3    prhs[7]  /* ...               ... */
# define q4    prhs[8]  /* ...               ... */
# define q5    prhs[9]  /* ...               ... */

/*
 * The first set of functions handles checks of the input arguments and
 * allocates the return arguments for each mode.
 */

void bfctmex_init_checks(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[])
{
  /* MODE     prhs[0]  numeric, scalar */
  CHECK_NARGIN_GE(2);
  CHECK_NUMERIC(MODE);
  CHECK_SCALAR(MODE);

  if (nrhs>3)
  {
    CHECK_NUMERIC(FA);
    CHECK_NUMERIC(FB);
    if (!mxIsScalar(FA) || !mxIsScalar(FB)) {
      mexErrMsgIdAndTxt(MEXARGS_STR(MEXNAME)":multiD","To use "MEXARGS_STR(MEXNAME)" with multiple domains, please use bfgenD");
    }
  }
}

void bfctmex_mode_0_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng)
{
  CHECK_NARGIN_EQ(2);
  CHECK_NARGOUT_LE(3);
  
  /* y1 fPg   pout[0] double, size=[ng,1] */
  /* y2 fTg   pout[1] double, size=[ng,1] */
  /* y3 TDg   pout[2] double, size=[1,ng] */
  y1 = mxCreateDoubleMatrix(ng, 1,mxREAL);
  y2 = mxCreateDoubleMatrix(ng, 1,mxREAL);
  y3 = mxCreateDoubleMatrix( 1,ng,mxREAL);
}

void bfctmex_mode_1_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nzy, int *nry)
{
  /* u1 Fx    prhs[2]  double, size=(nzy+2)*(nry+2) */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 Opy   prhs[5]  int8, def size=[nzy,nry] */
  /* q2 ry    prhs[6]  double, size=nry */
  /* q3 iry   prhs[7]  double, size=nry */
  CHECK_NARGIN_EQ(8);
  *nzy = mxGetM(q1);
  *nry = mxGetN(q1);
  CHECK_DOUBLE(u1);
  CHECK_NUMEL(u1,(*nzy+2)*(*nry+2));
  CHECK_INT8(q1);
  CHECK_DOUBLE(q2);
  CHECK_NUMEL(q2,*nry);
  CHECK_DOUBLE(q3);
  CHECK_NUMEL(q3,*nry);
  CHECK_NARGOUT_LE(3);
  
  /* y1 Tyg   pout[0] double, size=[nzy*nry,ng] */
  /* y2 Tpg   pout[1] double, size=[1,ng] */
  /* y3 ITpg  pout[2] double, size=[1,ng] */
  y1 = mxCreateDoubleMatrix((*nzy)*(*nry),ng,mxREAL);
  y2 = mxCreateDoubleMatrix(1            ,ng,mxREAL);
  y3 = mxCreateDoubleMatrix(1            ,ng,mxREAL);
}

void bfctmex_mode_2_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nQ)
{
  /* u1 FQ    prhs[2]  double, def size=nQ */
  /* FA FA    prhs[3]  unused */
  /* FB FB    prhs[4]  unused */
  CHECK_NARGIN_EQ(5);
  *nQ = mxGetNumberOfElements(u1);
  CHECK_DOUBLE(u1);
  CHECK_NARGOUT_LE(2);

  /* y1 gQg   pout[0] double, size=[nQ,ng] */
  /* y2 IgQg  pout[1] double, size=[nQ,ng] */
  y1 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
}

void bfctmex_mode_3_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng)
{
  /* u1 ag    prhs[2]  double, size=ng */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 fPg   prhs[5]  double, size=ng */
  /* q2 fTg   prhs[6]  double, size=ng */
  /* q3 ids   prhs[7]  numeric, scalar */
  CHECK_NARGIN_EQ(8);
  CHECK_DOUBLE(u1);
  CHECK_NUMEL(u1,ng);
  CHECK_DOUBLE(q1);
  CHECK_NUMEL(q1,ng);
  CHECK_DOUBLE(q2);
  CHECK_NUMEL(q2,ng);
  CHECK_NUMERIC(q3);
  CHECK_SCALAR(q3);
  CHECK_NARGOUT_LE(4);

  /* y1 aPpg  pout[0] double, size=ng */
  /* y2 aTTpg pout[1] double, size=ng */
  /* y3 aPg   pout[2] double, size=ng */
  /* y4 ahqTg pout[3] double, size=ng */
  y1 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y2 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y3 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y4 = mxCreateDoubleMatrix(ng,1,mxREAL);
}

void bfctmex_mode_4_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nzy, int *nry, int *nd)
{
  /* u1 Fx    prhs[2]  double, size=(nzy+2)*(nry+2) */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 Opy   prhs[5]  int8, def size=[nzy,nry] */
  /* q2 kd    prhs[6]  uint32, def size=nd */
  /* q3 fd    prhs[7]  double, size=[nd,2] */
  CHECK_NARGIN_EQ(8);
  *nzy = mxGetM(q1);
  *nry = mxGetN(q1);
  *nd = mxGetNumberOfElements(q2);
  CHECK_DOUBLE(u1);
  CHECK_NUMEL(u1,(*nzy+2)*(*nry+2));
  CHECK_INT8(q1);
  CHECK_UINT32(q2);
  CHECK_DOUBLE(q3);
  CHECK_NUMEL(q3,2*(*nd));
  CHECK_NARGOUT_LE(2);

  /* y1 Tdg   pout[0] double, size=[nd,ng] */
  /* y2 Tgy   pout[1] double, size=[ng,nry] */
  y1  = mxCreateDoubleMatrix(*nd,ng  ,mxREAL);
  y2  = mxCreateDoubleMatrix(ng ,*nry,mxREAL);
}

void bfctmex_mode_5_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng)
{
  /* u1 Fx    prhs[2]  unused */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  CHECK_NARGIN_EQ(5);
  CHECK_NARGOUT_LE(2);

  /* y1 gAg   pout[0] double, size=[1,ng] */
  /* y2 IgAg  pout[1] double, size=[1,ng] */
  y1 = mxCreateDoubleMatrix(1,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(1,ng,mxREAL);
}

void bfctmex_mode_6_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int nq)
{
  /* u1 Fx    prhs[2]  unused */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 rA    prhs[5]  numeric, scalar */
  /* q2 irA   prhs[6]  numeric, scalar */
  /* q3 ids   prhs[7]  numeric, scalar */
  CHECK_NARGIN_EQ(8);
  CHECK_NUMERIC(q1);
  CHECK_SCALAR(q1);
  CHECK_NUMERIC(q2);
  CHECK_SCALAR(q2);
  CHECK_NUMERIC(q3);
  CHECK_SCALAR(q3);
  CHECK_NARGOUT_LE(2);

  /* y1 Qqg   pout[0] double, size=[nq,ng] */
  /* y2 Xq    pout[1] double, size=nq */
  y1 = mxCreateDoubleMatrix(nq,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(nq,1 ,mxREAL);
}

void bfctmex_mode_7_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int nc)
{
  /* u1 Fx    prhs[2]  unused */
  /* FA FA    prhs[3]  unused */
  /* FB FB    prhs[4]  unused */
  CHECK_NARGIN_EQ(5);
  CHECK_NARGOUT_LE(2);

  /* y1 Qcg   pout[0] double, size=[nc,ng] */
  /* y2 Xc    pout[1] double, size=nc */
  y1 = mxCreateDoubleMatrix(nc,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(nc,1 ,mxREAL);
}

void bfctmex_mode_8_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nzy, int *nry)
{
  /* u1 Fx    prhs[2]  double, size=(nzy+2)*(nry+2) */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 Opy   prhs[5]  int8, def size=[nzy,nry] */
  /* q2 ag    prhs[6]  double, size=ng */
  /* q3 rBt   prhs[7]  numeric, scalar */
  /* q4 ids   prhs[8]  numeric, scalar */
  /* q5 iry   prhs[9]  double, size=nry */
  CHECK_NARGIN_EQ(10);
  *nzy = mxGetM(q1);
  *nry = mxGetN(q1);
  CHECK_DOUBLE(u1);
  CHECK_NUMEL(u1,(*nzy+2)*(*nry+2));
  CHECK_INT8(q1);
  CHECK_DOUBLE(q2);
  CHECK_NUMEL(q2,ng);
  CHECK_NUMERIC(q3);
  CHECK_SCALAR(q3);
  CHECK_NUMERIC(q4);
  CHECK_SCALAR(q4);
  CHECK_DOUBLE(q5);
  CHECK_NUMEL(q5,*nry);
  CHECK_NARGOUT_LE(1);

  /* y1 Bty   pout[0] double, size=[nzy,nry] */
  y1 = mxCreateDoubleMatrix(*nzy,*nry,mxREAL);
}

void bfctmex_mode_11_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nzy, int *nry)
{
  /* u1 Fx    prhs[2]  double, size=(nzy+2)*(nry+2) */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 Opy   prhs[5]  int8, def size=[nzy,nry] */
  /* q2 ry    prhs[6]  double, size=nry */
  /* q3 iry   prhs[7]  double, size=nry */
  CHECK_NARGIN_EQ(8);
  *nzy = mxGetM(q1);
  *nry = mxGetN(q1);
  CHECK_DOUBLE(u1);
  CHECK_NUMEL(u1,(*nzy+2)*(*nry+2));
  CHECK_INT8(q1);
  CHECK_DOUBLE(q2);
  CHECK_NUMEL(q2,*nry);
  CHECK_DOUBLE(q3);
  CHECK_NUMEL(q3,*nry);
  CHECK_NARGOUT_LE(5);
  
  /* y1 d1g   pout[0] double, size=[nzy*nry,ng] */
  /* y2 d2g   pout[1] double, size=[nzy*nry,ng] */
  /* y3 d3g   pout[1] double, size=[nzy*nry,ng] */
  /* y4 d2Ig  pout[2] double, size=[nzy*nry,ng] */
  /* y5 d3Ig  pout[3] double, size=[nzy*nry,ng] */
  y1 = mxCreateDoubleMatrix((*nzy)*(*nry),ng,mxREAL);
  y2 = mxCreateDoubleMatrix((*nzy)*(*nry),ng,mxREAL);
  y3 = mxCreateDoubleMatrix((*nzy)*(*nry),ng,mxREAL);
  y4 = mxCreateDoubleMatrix((*nzy)*(*nry),ng,mxREAL);
  y5 = mxCreateDoubleMatrix((*nzy)*(*nry),ng,mxREAL);
}

void bfctmex_mode_12_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nQ)
{
  /* u1 FQ    prhs[2]  double, def size=nQ */
  /* FA FA    prhs[3]  unused */
  /* FB FB    prhs[4]  unused */
  CHECK_NARGIN_EQ(5);
  *nQ = mxGetNumberOfElements(u1);
  CHECK_DOUBLE(u1);
  CHECK_NARGOUT_LE(6);

  /* y1 dgQdFQ  pout[0] double, size=[nQ,ng] */
  /* y2 dgQdF0  pout[1] double, size=[nQ,ng] */
  /* y3 dgQdF1  pout[2] double, size=[nQ,ng] */
  /* y4 dIgQdFQ pout[3] double, size=[nQ,ng] */
  /* y5 dIgQdF0 pout[4] double, size=[nQ,ng] */
  /* y6 dIgQdF1 pout[5] double, size=[nQ,ng] */
  y1 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
  y3 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
  y4 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
  y5 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
  y6 = mxCreateDoubleMatrix(*nQ,ng,mxREAL);
}

void bfctmex_mode_13_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng)
{
  /* u1 ag    prhs[2]  double, size=ng */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 fPg   prhs[5]  double, size=ng */
  /* q2 fTg   prhs[6]  double, size=ng */
  /* q3 ids   prhs[7]  numeric, scalar */
  CHECK_NARGIN_EQ(8);
  CHECK_DOUBLE(u1);
  CHECK_NUMEL(u1,ng);
  CHECK_DOUBLE(q1);
  CHECK_NUMEL(q1,ng);
  CHECK_DOUBLE(q2);
  CHECK_NUMEL(q2,ng);
  CHECK_NUMERIC(q3);
  CHECK_SCALAR(q3);
  CHECK_NARGOUT_LE(8);

  /* y1 daPpgdF0  pout[0] double, size=ng */
  /* y2 daTTpgdF0 pout[1] double, size=ng */
  /* y3 daPgdF0   pout[2] double, size=ng */
  /* y4 dahqTgdF0 pout[3] double, size=ng */
  /* y5 daPpgdF0  pout[0] double, size=ng */
  /* y6 daTTpgdF0 pout[1] double, size=ng */
  /* y7 daPgdF0   pout[2] double, size=ng */
  /* y8 dahqTgdF0 pout[3] double, size=ng */
  y1 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y2 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y3 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y4 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y5 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y6 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y7 = mxCreateDoubleMatrix(ng,1,mxREAL);
  y8 = mxCreateDoubleMatrix(ng,1,mxREAL);
}

void bfctmex_mode_15_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng)
{
  /* u1 Fx    prhs[2]  unused */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  CHECK_NARGIN_EQ(5);
  CHECK_NARGOUT_LE(4);

  /* y1 dgA1  pout[0] double, size=[1,ng] */
  /* y2 dgA2  pout[1] double, size=[1,ng] */
  /* y3 dIgA1 pout[2] double, size=[1,ng] */
  /* y4 dIgA2 pout[3] double, size=[1,ng] */
  y1 = mxCreateDoubleMatrix(1,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(1,ng,mxREAL);
  y3 = mxCreateDoubleMatrix(1,ng,mxREAL);
  y4 = mxCreateDoubleMatrix(1,ng,mxREAL);
}

void bfctmex_mode_16_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int nq)
{
  /* u1 Fx    prhs[2]  unused */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  /* q1 rA    prhs[5]  numeric, scalar */
  /* q2 irA   prhs[6]  numeric, scalar */
  /* q3 ids   prhs[7]  numeric, scalar */
  CHECK_NARGIN_EQ(8);
  CHECK_NUMERIC(q1);
  CHECK_SCALAR(q1);
  CHECK_NUMERIC(q2);
  CHECK_SCALAR(q2);
  CHECK_NUMERIC(q3);
  CHECK_SCALAR(q3);
  CHECK_NARGOUT_LE(4);

  /* y1 dQgF0 pout[0] double, size=[nq,ng] */
  /* y2 dQgF1 pout[1] double, size=[nq,ng] */
  /* y3 dQgr0 pout[2] double, size=[nq,ng] */
  /* y3 dQgir0 pout[3] double, size=[nq,ng] */
  y1 = mxCreateDoubleMatrix(nq,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(nq,ng,mxREAL);
  y3 = mxCreateDoubleMatrix(nq,ng,mxREAL);
  y4 = mxCreateDoubleMatrix(nq,ng,mxREAL);
}

void bfctmex_mode_17_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int nc)
{
  /* u1 Fx    prhs[2]  unused */
  /* FA FA    prhs[3]  numeric, scalar [checked] */
  /* FB FB    prhs[4]  numeric, scalar [checked] */
  CHECK_NARGIN_EQ(5);
  CHECK_NARGOUT_LE(2);

  /* y1 dQcgdF0 pout[0] double, size=[nc,ng] */
  /* y2 dQcgdF1 pout[1] double, size=[nc,ng] */
  y1 = mxCreateDoubleMatrix(nc,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(nc,ng,mxREAL);
}

void bfctmex_mode_91_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nF)
{
  /* u1 F     prhs[2]  double, def size=nF */
  /* FA FA    prhs[3]  unused */
  /* FB FB    prhs[4]  unused */
  CHECK_NARGIN_EQ(5);
  *nF = mxGetNumberOfElements(u1);
  CHECK_DOUBLE(u1);
  CHECK_NARGOUT_LE(2);

  /* y1 g     pout[0] double, size=[nF,ng] */
  /* y2 Ig    pout[1] double, size=[nF,ng] */
  y1 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
}

void bfctmex_mode_92_checks_alloc(int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[], int ng, int *nF)
{
  /* u1 F     prhs[2]  double, def size=nF */
  /* FA FA    prhs[3]  unused */
  /* FB FB    prhs[4]  unused */
  CHECK_NARGIN_EQ(5);
  *nF = mxGetNumberOfElements(u1);
  CHECK_DOUBLE(u1);
  CHECK_NARGOUT_LE(6);

  /* y1 dgx   pout[0] double, size=[nF,ng] */
  /* y2 dgA   pout[1] double, size=[nF,ng] */
  /* y3 dgB   pout[1] double, size=[nF,ng] */
  /* y4 dIgx  pout[2] double, size=[nF,ng] */
  /* y5 dIgA  pout[3] double, size=[nF,ng] */
  /* y6 dIgB  pout[3] double, size=[nF,ng] */
  y1 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
  y2 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
  y3 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
  y4 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
  y5 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
  y6 = mxCreateDoubleMatrix(*nF,ng,mxREAL);
}

/*
 * bfctmex is a generic function that can be called from MEX-files to do
 * all necessary checks, allocations and computations using the generic
 * bfct functions.
 */

void bfctmex(int mode, dbfct_params *params, int nlhs, mxArray *pout[], int nrhs, const mxArray *prhs[])
{
  int ng = params->ng;

  /* Chinese menu */
  switch (mode) {
    case 0 : {
      /* Inputs checks */
      bfctmex_mode_0_checks_alloc(nlhs, pout, nrhs, prhs, ng);

      /* Call generic bfct */
      dbfct0(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3),
            params);
      break;
    }
    case 1 : {
      int nzy, nry;

      /* Inputs checks */
      bfctmex_mode_1_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nzy, &nry);

      /* Call generic bfct */
      dbfct1(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3),
             mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB),
             (int8_t *)mxGetData(q1), mxGetPr(q2), mxGetPr(q3),
             nzy, nry, 0,
             params);
      break;
    }
    case 2: {
      int nQ;

      /* Inputs checks */
      bfctmex_mode_2_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nQ);

      /* Call generic bfct */
      dbfct2(mxGetPr(y1), mxGetPr(y2), mxGetPr(u1), nQ, params);
      break;
    }
    case 3: {
      /* Inputs checks */
      bfctmex_mode_3_checks_alloc(nlhs, pout, nrhs, prhs, ng);

      /* Call generic bfct */
      dbfct3(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3), mxGetPr(y4),
             mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB),
             mxGetScalar(q3), params);
      break;
    }
    case 4 : {
      int nzy, nry, nd;

      /* Inputs checks */
      bfctmex_mode_4_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nzy, &nry, &nd);

      /* Call generic bfct */
      dbfct4(mxGetPr(y1), mxGetPr(y2),
             mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB),
             (int8_t *)mxGetData(q1), nzy, nry,
             (unsigned int *)mxGetData(q2), mxGetPr(q3), nd,
             params);
      break;
    }
    case 5: {
      /* Inputs checks */
      bfctmex_mode_5_checks_alloc(nlhs, pout, nrhs, prhs, ng);

      /* Call generic bfct */
      dbfct5(mxGetPr(y1), mxGetPr(y2), mxGetScalar(FA), mxGetScalar(FB),
             params);
      break;
    }
    case 6: {
      int nq = 0;

      /* Inputs checks */
      bfctmex_mode_6_checks_alloc(nlhs, pout, nrhs, prhs, ng, nq);

      /* No regularization constraints */
      break;
    }
    case 7: {
      int nc = 0;

      /* Inputs checks */
      bfctmex_mode_7_checks_alloc(nlhs, pout, nrhs, prhs, ng, nc);

      /* No inequality constraints */
      break;
    }
    case 8: {
      int nzy, nry;

      /* Inputs checks */
      bfctmex_mode_8_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nzy, &nry);

      /* Call generic bfct */
      dbfct8(mxGetPr(y1), mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB),
             (int8_t *)mxGetData(q1), mxGetPr(q2), mxGetScalar(q3),
             mxGetScalar(q4), mxGetPr(q5), nzy, nry, 0, params);
      break;
    }
    case 11: {
      int nzy, nry;

      /* Inputs checks */
      bfctmex_mode_11_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nzy, &nry);

      /* Call generic bfct */
      dbfct11(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3), mxGetPr(y4), mxGetPr(y5),
              mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB),
              (int8_t *)mxGetData(q1), mxGetPr(q2), mxGetPr(q3),
              nzy, nry, 0, params);
      break;
    }
    case 12: {
      int nQ;

      /* Inputs checks */
      bfctmex_mode_12_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nQ);

      /* Call generic bfct */
      dbfct12(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3), 
              mxGetPr(y4), mxGetPr(y5), mxGetPr(y6),
              mxGetPr(u1), nQ, params);
      break;
    }
    case 13: {
      /* Inputs checks */
      bfctmex_mode_13_checks_alloc(nlhs, pout, nrhs, prhs, ng);

      /* Call generic bfct */
      dbfct13(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3), mxGetPr(y4),
              mxGetPr(y5), mxGetPr(y6), mxGetPr(y7), mxGetPr(y8),
              mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB),
              mxGetScalar(q3), params);
      break;
    }
    case 15: {
      /* Inputs checks */
      bfctmex_mode_15_checks_alloc(nlhs, pout, nrhs, prhs, ng);

      /* Call generic bfct */
      dbfct15(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3), mxGetPr(y4), 
              mxGetScalar(FA), mxGetScalar(FB), params);
      break;
    }
    case 16: {
      int nq = 0;

      /* Inputs checks */
      bfctmex_mode_16_checks_alloc(nlhs, pout, nrhs, prhs, ng, nq);

      /* No regularization constraints */
      break;
    }
    case 17: {
      int nc = 0;

      /* Inputs checks */
      bfctmex_mode_17_checks_alloc(nlhs, pout, nrhs, prhs, ng, nc);

      /* No inequality constraints */
      break;
    }
    case 91: {
      int nF;

      /* Inputs checks */
      bfctmex_mode_91_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nF);

      /* Call generic bfct */
      dbfct91(mxGetPr(y1), mxGetPr(y2),
              mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB), nF, params);
      break;
    }
    case 92: {
      int nF;

      /* Inputs checks */
      bfctmex_mode_92_checks_alloc(nlhs, pout, nrhs, prhs, ng, &nF);

      /* Call generic bfct */
      dbfct92(mxGetPr(y1), mxGetPr(y2), mxGetPr(y3),
              mxGetPr(y4), mxGetPr(y5), mxGetPr(y6),
              mxGetPr(u1), mxGetScalar(FA), mxGetScalar(FB), nF, params);
      break;
    }
    default: {
      mexErrMsgIdAndTxt(MEXARGS_STR(MEXNAME)":unknownmode","Mode %d not defined for "MEXARGS_STR(MEXNAME),mode);
    }
  }
}
