function [Tve,Le,Re,sym] = vveig(Mvv,Rv,type)

% VVEIG   Vessel eigenmode representation
%   [TVE,LE,RE,SYM] = VVEIG(MVV,RV) returns the vessel eigenmodes TVE, their
%   self inductance LE, their nominal resistance RE and a 0-1 vector flagging
%   the symetric eigenmodes in SYM. MVV is the external conductor mutual
%   (can be only filament vessel or vessel + coils) and RV their resistance.
%   type is a string which select the renormalisation used.
%
%   The transformation from eigenmode currents to vessel current distribution is
%   IV = TVE*IE. The transformation of a mutual matrix to the current
%   distribution to the mutual to the eigenmodes is MXE = MXV*TVE. In particular
%   LEE = TVE'*MVV*TVE and REE = TVE'*diag(RV)*TVE.
%
%   The transformation is based on the diagonalisation of the matrix
%   diag(RV)^{-1/2}*MVV*diag(RV)^{-1/2}. TVE = sqrt(RV)\QVE*sqrt(RE) where
%   WVE is the orthogonal matrix that perform the aforementioned
%   diagonalisation. The difference in the two type is the normalisation 
%   RE selected. In the first case (selected using type='R') the normalisation
%   will make the resistance RE a number (=1 ./ sum(1 ./ RV)),
%   while LE an array. In the second case (selected using type='M') RE is 
%   an array and LE is a number (=1).
%
%   Note that inv(TVE) ~= TVE' in general. If all RV are equal and in the 
%   case type='R', inv(TVE) is proportional to TVE', the proportionality 
%   constant being nv=length(Rv).
%
% [+GenLib General Purpose Library+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin < 3
  type = 'R';
end

nv = length(Rv);

MM = Mvv ./ (sqrt(Rv)*sqrt(Rv')); 
[Tve,gammae] = eig((MM+MM')/2); % ensure symmetry for real eigenvalues
gammae = diag(gammae);
k = flipud(isort(gammae));
if strcmp(type,'R')
  % resistance normalisation
  Re = 1 ./ sum(1 ./ Rv);
  Le = Re.*gammae(k);
elseif strcmp(type,'M')
  % inductance normalisation
  Re = 1./(gammae(k));
  Le = 1;
else
  error('Normalisation not possible');
end
Tve = (Tve(:,k)./sqrt(Rv(:,ones(1,nv))))*sqrt(diag(Re));

% parity
sym = sum((Tve(1:floor((nv+1)/2),:) + Tve(nv:-1:ceil((nv+1)/2),:)).^2) > 1e-6;
fact = sign(Tve(1,:));
fact(~sym) = sign(Tve(2,~sym) - Tve(1,~sym));
Tve = Tve .* fact(ones(nv,1),:);
end
