/* [+GenLib General Purpose Library+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved. */
# include "genlib.h"

/* Compute interpolation of regular 2D cubic B-spline */
void cs2deval(double *V, double *Vx, double *Vy, double *Vxx, double *Vxy, double *Vyy,
              double *Vxxx, double *Vxxy, double *Vxyy, double *Vyyy,
              double *Tx, int nTx, double *Ty, int nTy, double *C, int nCx, int nCy,
              double *xp, double *yp, int np, int dmax, double idx, double idy)
{
 int k = 4;
 int i;
 int ix, iy;
 double dx, dy;
 double facx, facy;
 double a00, a01, a02, a03;
 double a10, a11, a12, a13;
 double a20, a21, a22, a23;
 double a30, a31, a32, a33;
 double b00, b01, b02, b03;
 double b10, b11, b12, b13;
 double b20, b21, b22, b23;
 double b30, b31, b32, b33;
 double c00, c01, c02, c03;
 double c10, c11, c12, c13;
 double c20, c21, c22, c23;
 double c30, c31, c32, c33;
 
 if (dmax < 0) return;

 for (i = 0; i < np; i++) {
  /* Determine ix such that Tx[ix] <= xp[i] < Tx[ix+1] */
  ix = (int)((xp[i]-Tx[0])*idx);
  ix = ix < k ? k-1 : (ix > nCx - 1 ? nCx - 1 : ix);
  /* Determine iy such that Ty[iy] <= yp[i] < Ty[iy+1] */
  iy = (int)((yp[i]-Ty[0])*idy);
  iy = iy < k ? k-1 : (iy > nCy - 1 ? nCy - 1 : iy);

  /* Compute grid spacing */
  dx = (xp[i]-Tx[ix])*idx;
  dy = (yp[i]-Ty[iy])*idy;

  /* Compute function interpolation at xp[i] and yp[i] */
  csint(&a03,&a02,&a01,&a00,dx,0,1.);
  csint(&b03,&b02,&b01,&b00,dy,0,1.);

  /* Compute spline coefficients for V(xp[i],*) near yp[i] */
  adotc(&c00,&c01,&c02,&c03,a00,a01,a02,a03,C,ix,iy,nCx);

  /* Compute V at (xp[i],yp[i]) */
  V[i] = b00*c00 + b01*c01 + b02*c02 + b03*c03;

  if (dmax > 0) {
   facx = idx;
   facy = idy;

   /* Compute 1st derivative interpolation at xp[i] and yp[i] */
   csint(&a13,&a12,&a11,&a10,dx,1,facx);
   csint(&b13,&b12,&b11,&b10,dy,1,facy);
  
   /* Compute spline coefficients for dVdx(xp[i],*) near yp[i] */
   adotc(&c10,&c11,&c12,&c13,a10,a11,a12,a13,C,ix,iy,nCx);

   /* Compute dVdx, dVdy at (xp[i],yp[i]) */
   Vx[i] = b00*c10 + b01*c11 + b02*c12 + b03*c13;
   Vy[i] = b10*c00 + b11*c01 + b12*c02 + b13*c03;
  }
  
  if (dmax > 1) {
   facx *= idx;
   facy *= idy;

   /* Compute 2nd derivative interpolation at xp[i] and yp[i] */
   csint(&a23,&a22,&a21,&a20,dx,2,facx);
   csint(&b23,&b22,&b21,&b20,dy,2,facy);
  
   /* Compute spline coefficients for d2Vdxx(xp[i],*) near yp[i] */
   adotc(&c20,&c21,&c22,&c23,a20,a21,a22,a23,C,ix,iy,nCx);

   /* Compute d2Vdxx, d2Vdxy, d2Vdyy at (xp[i],yp[i]) */
   Vxx[i] = b00*c20 + b01*c21 + b02*c22 + b03*c23;
   Vxy[i] = b10*c10 + b11*c11 + b12*c12 + b13*c13;
   Vyy[i] = b20*c00 + b21*c01 + b22*c02 + b23*c03;
  }
  
  if (dmax > 2) {
   facx *= idx;
   facy *= idy;

   /* Compute 2nd derivative interpolation at xp[i] and yp[i] */
   csint(&a33,&a32,&a31,&a30,dx,3,facx);
   csint(&b33,&b32,&b31,&b30,dy,3,facy);
  
   /* Compute spline coefficients for d3Vdxxx(xp[i],*) near yp[i] */
   adotc(&c30,&c31,&c32,&c33,a30,a31,a32,a33,C,ix,iy,nCx);

   /* Compute d3Vdxxx, d3Vdxxy, d3Vdxyy, d3Vdyyy at (xp[i],yp[i]) */
   Vxxx[i] = b00*c30 + b01*c31 + b02*c32 + b03*c33;
   Vxxy[i] = b10*c20 + b11*c21 + b12*c22 + b13*c23;
   Vxyy[i] = b20*c10 + b21*c11 + b22*c12 + b23*c13;
   Vyyy[i] = b30*c00 + b31*c01 + b32*c02 + b33*c03;
  }
 }
}

/* Evaluation of regular 1D cubic B-spline
 *   given coefficients C and knots T, evaluation of d-th derivative at 
 *   T[k]+rho (0<=rho<dT) is
 *   V = c0*C[k] + c1*C[k-1] + c2*C[k-2] + c3*C[k-3]
 */ 
void csint(double *c0, double *c1, double *c2, double *c3, double rho, int d,double fac)
{
 switch(d)
 {
   case 0:
   {
     *c0 =    rho;
     *c3 = 1.-rho;
     
     *c1 =   0.5*(*c0*(2.-rho) + *c3*(rho+1.));
     *c0 =   0.5*(*c0*(   rho)               );
     *c3 =   0.5*(               *c3*(1.-rho));
     
     *c2 = 1./3.*(*c1*(2.-rho) + *c3*(rho+2.));
     *c1 = 1./3.*(*c0*(3.-rho) + *c1*(rho+1.));
     *c0 = 1./3.*(*c0*(   rho)              );
     *c3 = 1./3.*(               *c3*(1.-rho));
     break;
   }
   case 1:
   {
     *c0 = (   rho)*fac;
     *c3 = (1.-rho)*fac;
     
     *c1 = 0.5*(*c0*(2.-rho) + *c3*(rho+1.));
     *c0 = 0.5*(*c0*(   rho)               );
     *c3 = 0.5*(               *c3*(1.-rho));
     
     *c2 = *c3 - *c1;
     *c1 = *c1 - *c0;
     *c3 =     - *c3;
     break;
   }
   case 2:
   {
     *c0 = (rho      )*fac;
     *c1 = (1.-3.*rho)*fac;
     *c2 = (3.*rho-2.)*fac;
     *c3 = (1.-rho   )*fac;
     break;
   }
   case 3:
   {
     *c0 =  fac;
     *c1 = -3.*fac;
     *c2 =  3.*fac;
     *c3 = -fac;
     break;
   }
 }
}

/* Helper function
 *   For A = [a0,a1,a2,a3] and C this computes 
 *   [c0;c1;c2;c3] = A*C(ix-3:ix,iy-3:iy)
 */
void adotc(double *c0, double *c1, double *c2, double *c3,
           double a0, double a1, double a2, double a3,
           double *C, int ix, int iy, int nCx)
{
 int iCw;
 
 iCw = (iy-3)*nCx+ix-3;
 *c0 = a0*C[iCw] + a1*C[iCw+1] + a2*C[iCw+2] + a3*C[iCw+3];
 iCw = iCw+nCx;
 *c1 = a0*C[iCw] + a1*C[iCw+1] + a2*C[iCw+2] + a3*C[iCw+3];
 iCw = iCw+nCx;
 *c2 = a0*C[iCw] + a1*C[iCw+1] + a2*C[iCw+2] + a3*C[iCw+3];
 iCw = iCw+nCx;
 *c3 = a0*C[iCw] + a1*C[iCw+1] + a2*C[iCw+2] + a3*C[iCw+3];
}
