/* [+GenLib General Purpose Library+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved. */
# include "genlib.h"

void mexFunction(int nlhs,       mxArray *plhs[],
                 int nrhs, const mxArray *prhs[])
{

# define MEXNAME greenemmex

 /* [y,...] = greenemmex(r1,z1,r2,z2,mode); */

# define R1     prhs[0] /* double, def size=n1 */
# define Z1     prhs[1] /* double, size=n1 */
# define R2     prhs[2] /* double, def size=n2 */
# define Z2     prhs[3] /* double, size=n1 */
# define MODE   prhs[4] /* int32,  def size=nm */
# define NPAR   prhs[5] /* numeric, scalar */

 CHECK_NARGIN_GE(5);
 CHECK_NARGIN_LE(6);

 int n1 = mxGetNumberOfElements(R1), n2 = mxGetNumberOfElements(R2),
     nm = mxGetNumberOfElements(MODE);

 CHECK_DOUBLE(R1);
 CHECK_DOUBLE(Z1);
 CHECK_NUMEL(Z1,n1);
 CHECK_DOUBLE(R2);
 CHECK_DOUBLE(Z2);
 CHECK_NUMEL(Z2,n2);
 CHECK_INT32(MODE);
 if (nrhs>5) { CHECK_NUMERIC(NPAR);
               CHECK_SCALAR(NPAR);}

 CHECK_NARGOUT_LE(nm);

 /* If the number of output arguments is less than the number
  * of requested modes, then the remaining modes will simply
  * be ignored */
 if (nm<=0)
  mexErrMsgIdAndTxt("greenemmex:mode","MODE argument to greenemmex must not be empty");
 int nm_ = (nlhs ? nlhs : 1); /* Actual number of computed modes */

 /* default values */
 int npar = (nrhs>5) ? mxGetScalar(NPAR) : 1;
 if (npar<=0)
  mexErrMsgIdAndTxt("greenemmex:npar","NPAR argument to greenemmex must be a strictly positive integer");

 genlib_bool der1 = 0, der2 = 0;
 int i,j,k;
 int    *mode = (int *) mxGetData(MODE);
 double **y = (double **) mxCalloc(nm_, sizeof *y),
       *pr1 = mxGetPr(R1), *pr2 = mxGetPr(R2), 
       *pz1 = mxGetPr(Z1), *pz2 = mxGetPr(Z2);

 for (k=0; k<nm_; k++) {
  y[k] = mxGetPr(plhs[k] = mxCreateDoubleMatrix(n1,n2,mxREAL));
  der1 |= mode[k] >  0;
  der2 |= mode[k] > 10;
 }
 
# ifdef GREENEMPAR
# pragma omp parallel for collapse(2) num_threads(npar)
# endif
 for (j=0; j<n2; j++) {
  for (i=0; i<n1; i++) {
   greenem(mode, nm_, der1, der2, pr1[i], pz1[i], pr2[j], pz2[j], y, j*n1+i);
  } /* for i */
 } /* for j */
}
