function varargout = greenem(mode,r1,z1,r2,z2,T)

%GREENEM  Electromagnetic Green's functions
% [Y1,...] = GREENEM({G1,...},R1,Z1[,R2,Z2]) computes according to the G
% selectors:
% 'm[ut]' the mutual inductance between the turn set R1,Z1 and R2,Z2 (R1,Z1
% if omitted)
% 'br','bz' the field at R1,Z1 producted by unit currents at R2,Z2
% 'dr1m','dr1br','dr1bz','dz1..','dr2..','dz2..' the partial derivates of
%  the mutual inductance or magnetic field
% 'dr1r1m','dr1r2m',... the second derivatives of the mutual inductance
% GREENEM({G1,...},R1,Z1,R2,Z2,T) post multiplies the output with T.
% GREENEM('m[ut]',R,Z,A|[A B]) computes the mutual between loops at R,Z
%  and their self with a circular cross section of radius A or a
%  rectangular cross section AxB
% GREENEM('s[elf]',R,~,A|[A B][,B]) returns only the self
%  'self' cannot be combined with any other mode or with the T argument
%
% Notes about the size of input arguments:
% * The number of elements in R1 and Z1 must match
% * The number of elements in R2 and Z2 must match
% * For a circular conductor, the number of elements in A must match that
% of R1
% * For a rectangular conductor, the number of elements in A and B must
% match that of R1
%
% Notes about the size of output arguments:
% * When asking for the self inductance, the returned value will have the
% same size as R1
% * Otherwise the size of all return arguments will be [N1,N2] where N1/N2
% are the number of elements in R1/R2
%
% [+GenLib General Purpose Library+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% Check input argument mode
if isempty(mode), error('greenem:mode','Input argument mode must not be empty');end
if ischar(mode), mode = {mode}; end
if ~iscellstr(mode), error('greenem:mode','Input argument mode must be a char array or a cell array of string');end
% Check compatibility with number of output arguments
if (nargout > numel(mode)), error('greenem:nargout','The number of output arguments must be lower than the number of requested modes');end
nm = max(1,nargout);
mode = mode(1:nm); % Discard extra modes

% Convert text mode to int32
m = zeros(1,numel(mode),'int32');
for k = 1:numel(m)
 switch mode{k}
  case {'self' 's'}              , m(k) = -1;
  case {'mut' 'm'}               , m(k) =  0;
  case {'dr1m' 'dr1mut' 'dmdr1'} , m(k) =  1;
  case {'dz1m' 'dz1mut' 'dmdz1'} , m(k) =  2;
  case {'dr2m' 'dr2mut' 'dmdr2'} , m(k) =  3;
  case {'dz2m' 'dz2mut' 'dmdz2'} , m(k) =  4;
  case  'br'                     , m(k) =  5;
  case  'bz'                     , m(k) =  6;
  case {'dr1r1m' 'dr1r1mut'}     , m(k) = 11;
  case {'dr1z1m' 'dr1z1mut'}     , m(k) = 12;
  case {'dr1r2m' 'dr1r2mut'}     , m(k) = 13;
  case {'dr1z2m' 'dr1z2mut'}     , m(k) = 14;
  case {'dz1r1m' 'dz1r1mut'}     , m(k) = 12; % 21
  case {'dz1z1m' 'dz1z1mut'}     , m(k) = 22;
  case {'dz1r2m' 'dz1r2mut'}     , m(k) = 23;
  case {'dz1z2m' 'dz1z2mut'}     , m(k) = 24;
  case {'dr2r1m' 'dr2r1mut'}     , m(k) = 13; % 31
  case {'dr2z1m' 'dr2z1mut'}     , m(k) = 23; % 32
  case {'dr2r2m' 'dr2r2mut'}     , m(k) = 33;
  case {'dr2z2m' 'dr2z2mut'}     , m(k) = 34;
  case {'dz2r1m' 'dz2r1mut'}     , m(k) = 14; % 41
  case {'dz2z1m' 'dz2z1mut'}     , m(k) = 24; % 42
  case {'dz2r2m' 'dz2r2mut'}     , m(k) = 34; % 43
  case {'dz2z2m' 'dz2z2mut'}     , m(k) = 22; % 44
  case {'dr1br' 'dbrdr' 'dbrdr1'}, m(k) = 51;
  case {'dz1br' 'dbrdz' 'dbrdz1'}, m(k) = 52;
  case {'dr2br' 'dbrdr2'}        , m(k) = 53;
  case {'dz2br' 'dbrdz2'}        , m(k) = 54;
  case {'dr1bz' 'dbzdr' 'dbzdr1'}, m(k) = 61;
  case {'dz1bz' 'dbzdz' 'dbzdz1'}, m(k) = 62;
  case {'dr2bz' 'dbzdr2'}        , m(k) = 63;
  case {'dz2bz' 'dbzdz2'}        , m(k) = 64;
  case {'0' 'zero'}              , m(k) = 99;
  otherwise,	error('Invalid G specifier: %s',mode{k})
 end
end
hasself = any(m<0);
hasmut  = any(m>=0);
if hasself && hasmut, error('greenem:self','Cannot ask for self with other modes in greenem'); end

% Dispatch r2/z2/a/b
a = []; b = [];
switch nargin
 case 3
  r2 = r1; z2 = z1;
  if hasself, error('greenem:nargin','self requires at least 4 arguments to specify the conductor size'); end
 case 4
  c = r2;
  r2 = r1; z2 = z1;
  n1 = numel(r1); nc = numel(c); sc = size(c);
  if isequal(n1,nc)
   a = c; b = [];
  elseif nc == 2*n1 && any(sc([1,end]) == 2)
   if sc(1) == 2 % sc = [2,*n1]
    a = c(1,:); b = c(2,:);
   else          % sc = [*n1,2]
    c = reshape(c,[],2);
    a = c(:,1); b = c(:,2);
   end
  else
   error('greenem:size','R,Z A|[A B] size mismatch')
  end
 case 5
  if hasself
   a = r2; b = z2;
   if ~(numel(a)==numel(r1) && numel(b)==numel(r1))
    error('greenem:size','R,Z A,B size mismatch')
   end
  end
 case 6
  if hasself
   error('greenem:nargin','Transfer matrix is incompatible with self-inductance computation')
  end
 otherwise
  error('greenem:nargin','Invalid arguments')
end

% Call greenemmex
if hasmut
 % Distribute at least 10.000 pairs per thread but cap number of threads to maxNumCompThreads
 npar = max(1,min(maxNumCompThreads,ceil(numel(r1)*numel(r2)/1e4)));
 varargout = cell(1,nm);
 % greenemmex assumes doubles as inputs
 [varargout{:}] = greenemmex(double(r1),double(z1),...
                             double(r2),double(z2),...
                             m,npar);
end

% Self inductance
kmut  = (m ==  0); % mut
kself = (m == -1); % self
if any(kself) || (any(kmut) && nargin == 4)
 if isempty(b)
  s = 4e-7*pi * r1(:) .* (log(8*r1(:)./ a(:)      ) - 1.75); % circular
 else
  s = 4e-7*pi * r1(:) .* (log(8*r1(:)./(a(:)+b(:))) - 0.50); % rectangular
 end
 if any(kmut)
  n1 = numel(r1);
  varargout{kmut}(1:(n1+1):n1*n1) = s; % Replace diagonal elements
 else
  varargout{kself} = reshape(s,size(r1));
 end
end

% Transfer matrix
if nargin == 6
 for k = 1:nm
  varargout{k} = varargout{k} * T;
 end
end
