function [Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ,...
          VQ,AQ,IpVQ,FtPVQ,...
          rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
          rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = fsgi(L, LY, rq, zq, aq, rO, zO)
% FSGI Computes flux surfaces averaged quantities and geometric quantities
% [Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ,...
%  Q,AQ,IpVQ,FtPVQ,...
%  rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
%  rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = fsgi(L, LY, rq, zq, aq, rO, zO)
%
% Computes a multitude of flux surface averaged quantities as well as
% geometrical quantities of the plasma. Uses flux surfaces as computed by
% rtci.m. Outputs are the combined outputs of fsgimex, fsg2mex and shapmex.
% Arguments:
%   L:    struct, standard L structure
%   LY:   struct, standard LY structure containing converged GS-equilibrium
%   rq:   double(noq,npq,nD), r-positions of discreitzed flux surfaces
%   zq:   double(noq,npq,nD), z-positions of discreitzed flux surfaces
%   aq:   double(noq,npq,nD), radial positions (how far away from origin 
%                               points) for the discreitzed flux surfaces
%   rO:   double(noq,nD), r-positions of constant-theta ray origins
%   zO:   double(noq,nD), z-positions of constant-theta ray origins
% returns: (inside varargout, in this order)
%   Q0Q:     double(nQ,nD), <1/R>
%   Q1Q:     double(nQ,nD), -dpsi/dV = -int(dl/|Bp|)
%   Q2Q:     double(nQ,nD), <1/R^2>
%   Q3Q:     double(nQ,nD), <|grad psi|^2/R^2> = (2pi)^2<Bp^2>
%   Q4Q:     double(nQ,nD), <|grad psi|^2> = (2pi)^2<(R*Bp)^2>
%   iqQ:     double(nQ,nD), 1/q
%   ItQ:     double(nQ,nD), Ip contained inside FS = int(Bp dl)
%   LpQ:     double(nQ,nD), length of flux surface = int(dl)
%   rbQ:     double(nQ,nD), R-barycenter = int(R dl) / int(dl)
%   Q5Q:     double(nQ,nD), <|grad psi|/(2*pi)> = <R*Bp>
%   SlQ:     double(nQ,nD), 2*pi*int(R dl)
%   VQ:      double(nQ,nD), Volume inside FS = int(2*pi*R dA)
%   AQ:      double(nQ,nD), Surface inside contour = int(dA)
%   IpVQ:    double(nQ,nD), Ip contained inside FS = int(p' + <1/R^2> TT' / mu0 dV)
%   FtPVQ:   double(nQ,nD), Toroidal flux contained inside FS = int(<1/R^2> T / 2pi dV)
%   rgeom:   double(nQ,nD), rgeom = 0.5 * (rmin + rmax)
%   zgeom:   double(nQ,nD), zgeom = 0.5 * (zmin + zmax)
%   aminor:  double(nQ,nD), minor radius = 0.5 * (rmax - rmin)
%   epsilon: double(nQ,nD), inverse aspect ratio = aminor./rgeom
%   kappa:   double(nQ,nD), elongation = (zmax - zmin) / (rmax - rmin)
%   delta:   double(nQ,nD), triangularity, see shapmexm.m
%   deltal:  double(nQ,nD), lower triangularity, see shapmexm.m
%   deltau:  double(nQ,nD), upper triangularity, see shapmexm.m
%   rrmax:   double(nQ,nD), rmax
%   zrmax:   double(nQ,nD), z at rmax point
%   rrmin:   double(nQ,nD), rmin
%   zrmin:   double(nQ,nD), z at rmin point
%   rzmax:   double(nQ,nD), r at zmax point
%   zzmax:   double(nQ,nD), zmax
%   rzmin:   double(nQ,nD), r at zmin point
%   zzmin:   double(nQ,nD), zmin
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

is_doublet = L.nD == 3 && LY.nA == 2 && LY.nB == 3;
nAeff = min([LY.nA,LY.nB,L.nD]);

% init
Q0Q = zeros(L.nQ,L.nD); Q1Q = Q0Q; Q2Q = Q0Q; Q3Q = Q0Q; Q4Q = Q0Q;
iqQ = Q0Q; ItQ = Q0Q; LpQ = Q0Q; rbQ = Q0Q; Q5Q = Q0Q; SlQ = Q0Q;
VQ = Q0Q; AQ = Q0Q; IpVQ = Q0Q; FtPVQ = Q0Q;
rgeom = Q0Q; zgeom = Q0Q; aminor = Q0Q; epsilon = Q0Q;
kappa = Q0Q; delta = Q0Q; deltal = Q0Q; deltau = Q0Q;
rrmax = Q0Q; zrmax = Q0Q; rrmin = Q0Q; zrmin = Q0Q;
rzmax = Q0Q; zzmax = Q0Q; rzmin = Q0Q; zzmin = Q0Q;

% fill out output for domains with axes
for iD=1:nAeff
  [Q0Q(:,iD),Q1Q(:,iD),Q2Q(:,iD),Q3Q(:,iD),Q4Q(:,iD),...
   iqQ(:,iD),ItQ(:,iD),LpQ(:,iD),rbQ(:,iD),Q5Q(:,iD),SlQ(:,iD),...
   VQ(:,iD),AQ(:,iD),IpVQ(:,iD),FtPVQ(:,iD),...
   rgeom(:,iD),zgeom(:,iD),aminor(:,iD),epsilon(:,iD),...
   kappa(:,iD),delta(:,iD),deltal(:,iD),deltau(:,iD),...
   rrmax(:,iD),zrmax(:,iD),rrmin(:,iD),zrmin(:,iD),...
   rzmax(:,iD),zzmax(:,iD),rzmin(:,iD),zzmin(:,iD)] = fsgiD(L, LY, rq(:, :, iD), zq(:, :, iD), aq(:, :, iD), iD);
end

% if plasma is a doublet, fill out mantle domain (needs special treatment)
if is_doublet
  % get separatrix quantities from computation of inner two lobes
  ItS = sum(ItQ(end, [1,2])); % Toroidal current inside separatrix fsgi computation
  LpS = sum(LpQ(end, [1,2])); % length of separatrix
  SlS = sum(SlQ(end, [1,2])); % R-moment of separatrix 
  IpVS = sum(IpVQ(end, [1,2])); % Toroidal current inside separatrix volume int computation
  FtPVS = sum(FtPVQ(end, [1,2])); % Toroidal flux inside separatrix volume int computation
  
  [Q0Q(:,3),Q1Q(:,3),Q2Q(:,3),Q3Q(:,3),Q4Q(:,3),...
   iqQ(:,3),ItQ(:,3),LpQ(:,3),rbQ(:,3),Q5Q(:,3),SlQ(:,3),...
   VQ(:,3),AQ(:,3),IpVQ(:,3),FtPVQ(:,3),...
   rgeom(:,3),zgeom(:,3),aminor(:,3),epsilon(:,3),...
   kappa(:,3),delta(:,3),deltal(:,3),deltau(:,3),...
   rrmax(:,3),zrmax(:,3),rrmin(:,3),zrmin(:,3),...
   rzmax(:,3),zzmax(:,3),rzmin(:,3),zzmin(:,3)] = ...
      fsgi_mantle(L, LY, rq(:, :, 3), zq(:, :, 3), rO(:, 3), zO(:, 3), ItS, LpS, SlS, IpVS, FtPVS);
end
end


function [Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ,...
          VQ,AQ,IpVQ,FtPVQ,...
          rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
          rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = fsgiD(L, LY, rqD, zqD, aqD, iD)
% FSGID Computes flux surface averages and geometric quantities on single axis domain
% [...] = fsgiD(L, LY, rqD, zqD, aqD, iD)
%
% Computes a multitude of flux surface averaged quantities as well as
% geometrical quantities for a single plasma domain with magnetic axis.
% Arguments:
%   L:    struct, standard L structure
%   LY:   struct, standard LY structure containing converged GS-equilibrium
%   rqD:  double(noq,npq,1), r-positions of discreitzed flux surfaces of iD domain
%   zqD:  double(noq,npq,1), z-positions of discreitzed flux surfaces of iD domain
%   aqD:  double(noq,npq,1), radial positions for the discreitzed flux surfaces of iD domain
%   iD:   ind, index of axis domain to run processing on
% For outputs, see fsgi above.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

qaq = aqD.^2;
M1q = qaq*L.M1q; % da^2/qdq
M2q = ((L.M2q*qaq).^2./qaq+qaq)./M1q; % ((da^2/2do)^2/a^2+a^2)/(da^2/qdq) = ((da/do)^2+a^2)/(da^2/qdq)

% some info from LY
rA = LY.rA(iD); zA = LY.zA(iD); FA = LY.FA(iD); FB = LY.FB(iD);
dr2FA = LY.dr2FA(iD); dz2FA = LY.dz2FA(iD);
detFA = sign(LY.IpD(iD))*sqrt(dr2FA*dz2FA-LY.drzFA(iD).^2);
lX = LY.lX(iD); rB = LY.rB(iD); zB = LY.zB(iD); iTQ = LY.iTQ(:, iD);

% contour integrals
[Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ] = ...
  fsgimex(M1q,M2q,rqD,1./rqD,rA,FA,FB,detFA,lX,rB,iTQ,1/L.doq);
% FS volume/area
[VQ, AQ] = fsg2mex(aqD,rA,L.crq,L.doq);
% Plasma current inside flux suface computation by volume integral
IpVQ = cumtrapz(VQ, LY.PpQ(:, iD) + LY.TTpQ(:, iD) .* Q2Q / mu0);
% Toroidal flux computation by volume integral
FtPVQ = cumtrapz(VQ, (LY.TQ(:, iD) .* Q2Q) / (2 * pi));
% shape profiles
[rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
 rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = ...
  shapmex(rqD,zqD,rB,zB,rA,zA,dr2FA,dz2FA);
end


function [Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ,...
          VQ,AQ,IpVQ,FtPVQ,...
          rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
          rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = fsgi_mantle(L, LY, rq, zq, rS, zS, ItS, LpS, SlS, IpVS, FtPVS)
% FSGI_MANTLE Computes flux surfaces averaged quantities and geometric quantities
%   for a mantle domain
% [...] = fsgi_mantle(L, LY, rq, zq, rS, zS, ItS, LpS, SlS, IpVS, FtPVS)
%
% As the mantle domain does not have a single axis from which
% constant-theta rays originate, some flux surface integral computations
% have to be adapted.
% Arguments:
%   L:    struct, standard L structure
%   LY:   struct, standard LY structure containing converged GS-equilibrium
%   rq:   double(noq,npq,1), r-positions of discreitzed flux surfaces of mantle
%   zq:   double(noq,npq,1), z-positions of discreitzed flux surfaces of mantle
%   rS:   double(noq,1), r-positions of separatrix points
%   zS:   double(noq,1), z-positions of separatrix points
%   ItS:  double, Ip contained inside separatrix from fsgi computation
%   LpS:  double, Length of separatrix
%   SlS:  double, R-barycenter of separatrix
%   IpVS: double, Ip contained inside separatrix from volume integral computation
%   FtPVS:double, Toroidal flux inside separatrix from volume integral computation
% For outputs, see fsgi above.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

% wrapper of shapmex adding separatrix as innermost surface
[rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
 rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = shapmantle(rq,zq,rS,zS,LY.rB(3),LY.zB(3));

% volume and surface integrals using separatrix as innermost surface
[VQ, AQ] = fsg2mantle([rS, rq], [zS, zq]);

% flux surface averages
[Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ] = ...
  fsavgmantle(rq,zq,rS,zS,L.M3q,L.M2q,L.doq,LY.F1(3)-LY.F0(3),LY.rX,LY.lX(3),LY.rB(3),LY.iTQ(2:end,3),ItS,LpS,SlS);

% Plasma current inside flux suface computation by volume integral
IpVQ = IpVS + cumtrapz(VQ, LY.PpQ(:, 3) + LY.TTpQ(:, 3) .* Q2Q / mu0);
% Toroidal flux computation by volume integral
FtPVQ = FtPVS + cumtrapz(VQ, (LY.TQ(:, 3) .* Q2Q) / (2 * pi));
end


function varargout = shapmantle(rq, zq, rS, zS, rB, zB)
% SHAPMANTLE Computes geometric quantities for mantle domain
% varargout = shapmantle(rq, zq, rS, zS, rB, zB)
%
% Adds separatrix as flux surface, runs shapmex and truncates values
% corresponding to the "axis"
% Arguments:
%   rq:   double(noq,npq), r-positions of discreitzed flux surfaces
%   zq:   double(noq,npq), z-positions of discreitzed flux surfaces
%   rS:   double(noq, 1), r-positions of separatrix
%   zS:   double(noq, 1), z-positions of separatrix
%   rB:   double, r-position of point defining mantle LCFS
%   zB:   double, z-position of point defining mantle LCFS
% returns:
%   See fsgi above or shapmexm
%
% [+MEQ MatlabEQuilibrium Toolbox+]

[varargout{1:nargout}] = shapmex([rS,rq],[zS,zq],rB,zB,nan,nan,nan,nan);
for ii=1:nargout % cut first "axis" element of shape comp out
  varargout{ii} = varargout{ii}(2:end);
end
end


function [Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ] = fsavgmantle(rq,zq,rS,zS,M3q,M2q,doq,FAB,rX,lXm,rBm,iTQ,ItS,LpS,SlS)
% FSAVGMANTLE Computes flux surface averages for the mantle domain
% [Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ] = 
%   fsavgmantle(rq,zq,rS,zS,M3q,M2q,doq,FAB,rX,lXm,rBm,iTQ,ItS,LpS,SlS)
%
% Flux surface averages are computed without assumptions on magnetic
% geometry. The surface integrals are discretized as integrals over the
% edge elements and quantities are summed up using the midpoint rule
% Arguments:
%   rq:   double(noq,npq), r-positions of discreitzed flux surfaces
%   zq:   double(noq,npq), z-positions of discreitzed flux surfaces
%   rS:   double(noq, 1), r-positions of separatrix
%   zS:   double(noq, 1), z-positions of separatrix
%   M3q:  double(npq, npq), radial spline derivative matrix
%   M2q:  double(noq, noq), azimuthal spline derivative matrix
%   doq:  double, 2 * pi / noq
%   FAB:  double, F1(mantle) - F0(mantle)
%   VQ:   double(nQ, 1), Volume inside FS = int(2*pi*R dA)
%   rX:   double(nX), r-positions of X points (the first should be the central one)
%   lXm:  bool, flag indicating a diverted mantle
%   rBm:  double, radial coordinate of point defining mantle LCFS (limiter- or X-point)
%   iTQ:  double(nQ, 1), 1 / TQ (see meqpost)
%   ItS:  double, Ip contained inside separatrix
%   LpS:  double, Length of separatrix
%   SlS:  double, R-barycenter of separatrix
% returns:
%   See fsgi above
%
% [+MEQ MatlabEQuilibrium Toolbox+]

drdpsi = 0.5 * (rq - rS) * M3q / FAB;
dzdpsi = 0.5 * (zq - zS) * M3q / FAB;
drdtheta = M2q * rq * 2 * doq;
dzdtheta = M2q * zq * 2 * doq;
% grad_psi perp grad_theta => |grad_psi| * dl = det([grad_psi, grad_theta]) = det(J^-1)
dl = sqrt(drdtheta.^2 + dzdtheta.^2);
detJ = drdpsi .* dzdtheta - dzdpsi .* drdtheta;
dpsi = -dl ./ detJ; % minus due to point orientation
idpsidl = -detJ;

% useful things
irq = 1 ./ rq;
irX = 1 / rX(1);
% more useful things
dpsidV          = 1 ./ (2 * pi * sum(rq  .*     idpsidl, 1));
int_iR_idpsi_dl =                sum(irq .*     idpsidl, 1);
int_iR_dpsi_dl  =                sum(irq .*  dpsi .* dl, 1);
int_R_dl        =                sum( rq .*          dl, 1);

% flux surface averages
Q1Q = [0,                                                -dpsidV ].'; % -dpsi/dV
Q0Q = [irX,     2 * pi * sum(          idpsidl, 1) .*     dpsidV ].'; % <1/R>, 1/R * 1/Bp = 2 pi / gradpsi
Q2Q = [irX*irX, 2 * pi * int_iR_idpsi_dl           .*     dpsidV ].'; % <1/R^2>
Q3Q = [0,       2 * pi * int_iR_dpsi_dl            .*     dpsidV ].'; % <|grad psi|^2/R^2>
Q4Q = [0,       2 * pi * sum( rq .* dpsi .* dl, 1) .*     dpsidV ].'; % <|grad psi|^2>
Q5Q = [0,                int_R_dl                  .* abs(dpsidV)].'; % <|grad psi|/(2*pi)>
iqQ = [0, -iTQ' ./ int_iR_idpsi_dl].';
ItQ = [ItS, -int_iR_dpsi_dl / (2 * pi * mu0)].';
LpQ = [LpS, sum(dl, 1)].';
SlQ = [SlS, 2 * pi * int_R_dl].';
rbQ = SlQ ./ (2 * pi * LpQ);

if lXm % if the mantle is diverted
  irX2 = 1/rBm; % r coordinate defining mantle flux is at mantle X point
  Q0Q(end) = irX2;
  Q1Q(end) = 0;
  Q2Q(end) = irX2*irX2;
  Q3Q(end) = 0;
  Q4Q(end) = 0;
  iqQ(end) = 0;
  Q5Q(end) = 0;
end
end


function [VQ, AQ] = fsg2mantle(rq, zq)
% FSG2MANTLE Computes volume and area enclosed by flux surfaces
% [VQ, AQ] = fsg2mantle(rq, zq)
%
% To avoid assumptions on flux surface geometry, these computations are
% carried out fully by using greens formula
% Arguments:
%   rq:   double(noq,npq), r-positions of discreitzed flux surfaces
%   zq:   double(noq,npq), z-positions of discreitzed flux surfaces
% returns:
%   VQ:   double(nQ, 1), Volume inside FS = int(2*pi*R dA)
%   AQ:   double(nQ, 1), Surface inside contour = int(dA)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

dr = rq([2:end,1],:) - rq;
rc = 0.5 * (rq([2:end,1],:) + rq);
zc = 0.5 * (zq([2:end,1],:) + zq);
AQ = sum(zc .* dr, 1).'; % int(1 dA) = intC(z dr)
VQ = sum(2 * pi * zc .* rc .* dr, 1).'; % int(2piR dA) = intC(2pi R Z dr)
end
