function [rq, zq, aq] = find_contours(L, LY, F, aq, rO, zO, crq, czq, Fo, Opo)
% FIND_CONTOURS flux contour finding algorithm based on newton iterations
% [rq, zq, aq] = find_contours(L, LY, F, aq, ro, zo, cr, cz, Fo, Opo)
%
% Finds contours either by rtcics.m or rtcimex.m. Contours are computated as
% points where, origninating in ro,zo and going along cr,cz, the values of
% LY.Fx are equal to the supplied F
% Arguments:
%   L:    struct, standard L structure
%   LY:   struct, standard LY structure containing converged GS-equilibrium
%   F:    double(1, npq), requested flux values of flux surfaces
%   aq:   double(noq,npq), radial positions init
%   rO:   double(1 or noq, 1), r-positions of constant-theta ray origins
%   zO:   double(1 or noq, 1), z-positions of constant-theta ray origins
%   crq:  double(noq, 1), r-directions of constant-theta rays
%   czq:  double(noq, 1), z-directions of constant-theta rays
%   Fo:   double(1 or noq, 1), Flux values at origin points
%   Opo:  int8(1 or noq, 1), domain identifier of origin points
% returns:
%   rq:   double(noq,npq), r-positions of discretized flux surfaces
%   zq:   double(noq,npq), z-positions of discretized flux surfaces
%   aq:   double(noq,npq), radial positions (distance to rO, zO)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if L.P.icsint
  [aq,s] = rtcics(aq,LY.Fx,F,LY.Opy,rO,zO,Opo,crq,czq,L);
  if ~s
    meqmsge('w',mfilename,L.P.tokamak,LY.t,NaN,LY.shot,'meqpostq contour finding using rtcics did not converge','iterq');
  end
else
  ero = (rO-L.G.rx(1))*L.idrx;
  ezo = (zO-L.G.zx(1))*L.idzx;
  cdr = crq*L.idrx; cdz = czq*L.idzx;
  for k = 1:L.P.iterq
    [aq,daq] = rtcimex(aq,ero,ezo,LY.Fx,F,cdr,cdz,LY.Opy,Fo,Opo,L.drx);
    if ~L.liurtemu
      if daq < L.drx*L.P.tolq, break; % convergence tests only for non-itert
      elseif k==L.P.iterq
        meqmsge('w',mfilename,L.P.tokamak,LY.t,k,LY.shot,'meqpostq contour finding using rtcimex did not converge','iterq');
      end
    end
  end
end
% compute r, z coordinates in the end
rq = rO + crq .* aq;
zq = zO + czq .* aq;
end