%FGST  Forward Grad-Shafranov solver
% LY = FGST(L,LX) solves the Grad-Shafranov equation
% for a given conductor current distribution.
%
% parameters in L as obtained by FGSC and the inputs in LX
% as obtained by FGSP<TOK>
% returns a structure LY with the computed equilibrium.
%
% FGS-specific outputs:
%
% .mkryl    Number of Krylov iterations carried out
% .nfeval   Number of function evaluations
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [LY] = fgst(L,LX,varargin)

%% Arguments
for k = 1:2:numel(varargin)
  switch varargin{k}
    % Additional output arguments
    case 'argout', L.argoutc = meqargoutc(varargin{k+1},L.argoutc);
    % Additional parameters
    otherwise, L.P.(varargin{k}) = varargin{k+1};
  end
end

%% Checks
if isequal(L.P.algoNL,'Picard') && L.P.usepreconditioner
  error('can not use preconditioner for algoNL=''Picard'', please set usepreconditioner=0')
end

%% F options
opts = optsF('dopost',true,'doplot',L.P.debugplot>1);

%% Debug

dispinputs = L.P.debug>3;
dispiter = L.P.debug > 0; % display iteration information
plotiter = (L.P.debugplot==1); % plot per timeslice, if >1 more complex plots will be shown by solver calls.

if dispinputs
  fprintf('Call to fgst with these parameters:\n')
  dispparams(L.P);
end

if dispiter
  meqinfo(L,LX)
  tstart=tic;
end

%% SolveF parameters
P = struct('algoF', L.P.algoF,'mkryl', L.P.mkryl,...
  'relax', L.P.relax, 'kmax', L.P.kmax, ...
  'debug', L.P.debug, 'debugplot',L.P.debugplot,...
  'debugfail',L.P.debugfail,'debuglinesrch',L.P.debuglinesrch,...
  'tolF', L.P.tolF, 'algoGMRES', L.P.algoGMRES, 'anajac', L.P.anajac,...
  'userowmask',L.P.userowmask,'nrestarts',L.P.nrestarts, ...
  'jacobian_handle',L.P.jacobian_handle,...
  'restart_random',L.P.restart_random,...
  'use_inverse',L.P.use_inverse,'large_system',L.P.large_system,...
  'is_factored',L.P.is_factored,...
  'ldim',L.P.ldim,'group_size',L.P.group_size,...
  'iy',L.ind.ixGS,'ratio_gb',L.P.ratio_gb);

%% adding preconditioner
if L.P.usepreconditioner
  P.prectype    = L.P.prectype;
  P.precupdate  = L.P.precupdate;
  P.precside    = L.P.precside;
  P.ilu_algo    = L.P.ilu_algo;
  P.ilu_droptol = L.P.ilu_droptol;
  P.Prec        = L.P.Prec;
else
  P.prectype    = 'user';
  P.precupdate  = 'once';
  P.Prec        = [];
end

%% Time loop
for it = 1:numel(LX.t)
  LXt = meqxk(LX,it); % select inputs at this time

  % Update preconditioner
  if (strcmp(P.precupdate,'time') || (it==1 && strcmp(P.precupdate,'once')))
    P.Prec = fgepre(L,LXt,P,0);
  end

  % Function handle to compute equation residual.
  if strcmpi(L.P.algoNL,'picard')
    F = @(x_,opts_) fgeFpicard(x_,L,LXt,LXt,opts_);
  else % 'all-nl','all-nl-Fx','Newton-GS'
    Ie = [LXt.Ia;LXt.Iu];
    Iyie = reshape(L.Tye*Ie,L.nzy,L.nry);
    Fbe = L.Mbe*Ie;
    F = @(x_,opts_) fgeF(x_,L,LXt,LXt,opts_,{Fbe,Iyie});
  end

  % Initial state
  x0 = L.LX2x(LXt);

  % Solve the equation F(x)=0
  [X1, solverinfo] = solveF(F,x0,P);

  % Compute outputs
  if solverinfo.isconverged
    [~, LYt] = F(X1,opts);
  else
    [~, LYt] = F(x0,opts);
    meqmsge('i',mfilename,L.P.tokamak,LXt.t,solverinfo.niter,...
      L.P.shot,'Newton solver did not converge','FGS:nosolution');
  end

  % Add other output info
  LYt.t = LXt.t; LYt.shot = LXt.shot;
  LYt.res    = solverinfo.res;
  LYt.mkryl  = solverinfo.mkryl;
  LYt.nfeval = solverinfo.nfeval;
  LYt.niter  = solverinfo.niter;
  LYt.isconverged = solverinfo.isconverged;

  % Additional output arguments
  for k = 1:numel(L.argoutc)
    if isempty(L.argoutc{k}.file)
      LYt.(L.argoutc{k}.fld) = eval(L.argoutc{k}.exp);
    end
  end

  if ~LYt.isconverged
    meqmsge('w',mfilename,L.P.tokamak,LYt.t,LYt.niter,L.P.shot,'fgs: No convergence','fgstFail');
  end

  % debug output
  if dispiter
    if ~rem(it-1,10)
      meqdebug(L);
    end
    meqdebug(L,LYt,it);
  end
  if plotiter, clf; meqplott(L,LYt); end

  % time derivatives
  if it==1
    LYt = meqLYdot(LYt,LYt);
  else
    LYt = meqLYdot(LYt,LY(it-1));
  end

  % store in LY
  LY(it) = LYt; %#ok<AGROW>
end

% pack outputs
LY = meqlpack(LY);

% mask non-converged
if ~L.P.LYall
  LY = meqxk(LY,LY.isconverged);
end

if dispiter, fprintf('FGS done: time elapsed: %2.2f[s]\n',toc(tstart)); end
end
